package com.ejianc.foundation.report.controller;

import java.io.Serializable;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;

import com.ejianc.foundation.permission.api.IRoleApi;
import com.ejianc.foundation.report.vo.TableVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.controller.param.GridHeader;
import com.ejianc.foundation.report.es.EsSqlQueryTemplate;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.framework.core.response.CommonResponse;

@RestController
@RequestMapping("ai")
public class AiSqlController implements Serializable {
	private static final Logger logger = LoggerFactory.getLogger(AiSqlController.class);
	private static final long serialVersionUID = 169770636614492806L;
    private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
    private SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private DecimalFormat decimalFormat = new DecimalFormat("###,##0.00");

	@Autowired
	private EsSqlQueryTemplate sqlQueryTemplate;
	@Autowired
	private ITableService tableService;
	@Autowired
	private IColumnService columnService;
	@Autowired
	private IRoleApi roleApi;

	@PostMapping( "getDataBySql")
	@ResponseBody
	public CommonResponse<List<LinkedHashMap<String, Object>>> getDataBySql(@RequestBody Map<String,String> params) {
		String sql = params.get("sql");
		logger.info("**********************************执行ES查询参数:{}",JSONObject.toJSONString(params));
		String index = params.get("tableId");
		Long tenantId = Long.parseLong(params.get("tenantId"));
		TableEntity table = tableService.selectById(index);
		if(table==null){
			logger.error("根据索引{}未找到对应的表信息",index);
			return CommonResponse.error("没有相关信息！");
		}
		CommonResponse<Boolean> response = roleApi.hasPermissionByAppPathAndUserId("ejc-report-frontend/#/report?tableCode="+table.getCode(), InvocationInfoProxy.getUserid());
		if(response.getData()!=null&&!response.getData()){
			response = roleApi.hasPermissionByAppPathAndUserId("ejc-report-frontend/#/report?tableId="+table.getId(), InvocationInfoProxy.getUserid());
		}
		if(response.getData()==null || !response.getData()){
			return CommonResponse.error("您没有查询该数据的权限！");
		}
		List<LinkedHashMap<String, Object>> res = sqlQueryTemplate.execute(sql);
		List<LinkedHashMap<String, Object>> result = new ArrayList<>();
		if(!res.isEmpty()){
			List<GridHeader> gridHeaderList = columnService.queryGridHeadList(table.getId(), tenantId);
			if(gridHeaderList.isEmpty()){
				logger.error("根据表id{}未找到对应的列信息",table.getId());
				return CommonResponse.success(res);
			}
			for (int i = 0; i < res.size(); i++) {
				LinkedHashMap<String, Object> obj = res.get(i);
				LinkedHashMap<String, Object> copy = new LinkedHashMap<>();
				for(String key:obj.keySet()){
					boolean isExist = false;
					for (GridHeader gridHeader : gridHeaderList) {
						if (gridHeader.getCode().equals(key)) {
							if(obj.get(key)!=null){
								String value = obj.get(key).toString();
								if(StringUtils.isNotEmpty(gridHeader.getExportFormat())){
									value = formatField(value, gridHeader.getExportFormat());
								}else if(Objects.equals("decimal",gridHeader.getType())){
									value = formatField(value, "mnyFormat");
								}
								if(StringUtils.isNotEmpty(gridHeader.getSuffixStr())){
									value = value+gridHeader.getSuffixStr();
								}
								copy.put(gridHeader.getName(), value);
							}else {
								copy.put(gridHeader.getName(), "");
							}
							isExist = true;
							break;
						}
					}
					if(!isExist){
						if(key.toLowerCase().contains("count(")){
							copy.put(table.getTableName()+"数量是", obj.get(key)==null?"0":obj.get(key));
						}else if(key.toLowerCase().contains("sum(")){
							String column = key.substring(key.indexOf("(")+1, key.indexOf(")")).trim();
							String value = obj.get(key)==null?"0.00":obj.get(key).toString();
							for (GridHeader gridHeader : gridHeaderList) {
								if (gridHeader.getCode().equals(column)) {
									column = gridHeader.getName();
									value = formatField(value, gridHeader.getExportFormat());
									if(StringUtils.isNotEmpty(gridHeader.getSuffixStr())){
										value = value+gridHeader.getSuffixStr();
									}
									break;
								}
							}
							copy.put(column, value);
						}else {
							copy.put(key, obj.get(key));
						}
					}
				}
				result.add(copy);
			}
		}
		logger.info("************************查询sql，返回结果：{}",JSONObject.toJSONString(result));
		return CommonResponse.success(result);
	}

	private String formatField(String value, String format) {
        String result = null;

        if(StringUtils.isBlank(value) || StringUtils.isBlank(format)) {
            return value;
        }
        try {
            switch (format) {
                case "twoDecimalNumbers":
                    result = new BigDecimal(value).setScale(2, BigDecimal.ROUND_HALF_UP).toString();
                    break;
                case "integer":
                    result = new BigDecimal(value).setScale(0, BigDecimal.ROUND_HALF_UP).toString();
                    break;
                case "mnyFormat":
                    result = decimalFormat.format( new BigDecimal(value).setScale(6, BigDecimal.ROUND_HALF_UP));
                    break;
                case "yyyy-MM-dd":
                    result = dateFormat.format(dateFormat.parse(value));
                    break;
                case "yyyy-MM-dd HH:mm:ss":
                    result = dateTimeFormat.format(dateTimeFormat.parse(value));
                    break;
                default:
            }
        } catch (Exception e) {
            logger.error("依据模式[{}],格式化格式化[{}]异常：", format, value, e);
            return value;
        }
        return result;
    }

	/**
	 * 根据表ID获取提示模板ID。
	 * <p>
	 * 通过表ID查询数据库中的表信息，并返回该表对应的提示模板ID。如果表不存在，则返回错误信息。
	 * 此接口用于前端获取特定表的提示模板ID，以便进行进一步的操作或展示。
	 *
	 * @param tableId 表的唯一标识ID，用于查询表信息。
	 * @return 返回包含提示模板ID的响应对象。如果表不存在，则响应对象中包含错误信息。
	 */
	@GetMapping("getPromptTmpIdByTableId")
	@ResponseBody
	public CommonResponse<TableVO> getPromptTmpIdByTableId(Long tableId) {
	    // 根据表ID查询表信息
	    TableEntity table = tableService.selectById(tableId);
	    // 检查查询结果，如果表不存在，则记录错误日志并返回错误响应
	    if (table == null) {
	        logger.error("根据TableId---{}未找到对应的表信息", tableId);
	        return CommonResponse.error("未找到相关报表");
	    }
	    // 表存在时，返回成功响应，并包含表的提示模板ID
	    return CommonResponse.success("查询成功！", BeanMapper.map(table, TableVO.class));
	}
}
