package com.ejianc.foundation.report.controller;

import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.report.consts.FinanceUseConsts;
import com.ejianc.foundation.report.custom.vo.FinanceUseResSubVO;
import com.ejianc.foundation.report.custom.vo.FinanceUseResVO;
import com.ejianc.foundation.report.service.CustomReportService;
import com.ejianc.foundation.report.util.EJCDateUtil;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.Aggregations;
import org.elasticsearch.search.aggregations.bucket.terms.ParsedLongTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.ParsedSum;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 自定义报表查询
 * @date 2022/4/14
 */
@RestController
@RequestMapping("/customReport")
public class CustomReportController {

    public final static Integer QUERY_TIMEOUT = 60;

    @Autowired
    private RestHighLevelClient client;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private CustomReportService customReportService;


    /**
     * @param dateIn
     * @description: 公司领导门户资金使用
     * * "production":产值统计1
     * * "partAReport":甲方报量1
     * * "yingshouweishou":应收未收 = 甲方报量 - 累计收款
     * * "backMoney":回款 = 累计收款1
     * * "openInvince":"开票
     * * "jiesuanjine":结算金额1
     * * "receiveInvince":收票金额
     * * "payMoney":"付款金额1
     * * "yingfuweifu":"应付未付 = 结算金额 - 付款金额
     * * "zhangmianjine":账面金额 = 回款 - 付款金额
     * * "monthData":['2021-09', '2021-10', '2021-11', '2021-12', '2022-01', '2022-02', '2022-03']
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.income.vo.FinanceUseVO>
     * @author songlx
     * @date: 2022/4/14
     */
    @RequestMapping(value = "/queryFinanceUse", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<FinanceUseResVO> queryFinanceUse(@RequestParam String dateIn) {
        SearchRequest searchRequest = new SearchRequest(FinanceUseConsts.INDEX_EJC_FINANCE_USE);
        SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
        sourceBuilder.from(0);
        sourceBuilder.size(10000);
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        boolQuery.must(QueryBuilders.termQuery("tenantId", InvocationInfoProxy.getTenantid().toString()));

        if (InvocationInfoProxy.getOrgId() != null) {
            boolQuery.must(QueryBuilders.termsQuery("orgId", orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            UserContext userContext = sessionManager.getUserContext();
            String authOrgIds = userContext.getAuthOrgIds();
            if (StringUtils.isNotEmpty(authOrgIds)) {
                String[] orgIds = authOrgIds.split(",");
                boolQuery.must(QueryBuilders.termsQuery("orgId", orgIds));
            }
        }

        if (FinanceUseConsts.TIME_RANGE_THIS_YEAR.equals(dateIn)) {
            boolQuery.must(QueryBuilders.termQuery("yyear", EJCDateUtil.getYear()));
        } else if (FinanceUseConsts.TIME_RANGE_3_MONTH.equals(dateIn)) {
            boolQuery.must(QueryBuilders.termsQuery("yyearMonth", EJCDateUtil.getShortMonthIntBetween(EJCDateUtil.getMonth(-2), EJCDateUtil.getMonth(0))));
        } else if (FinanceUseConsts.TIME_RANGE_6_MONTH.equals(dateIn)) {
            boolQuery.must(QueryBuilders.termsQuery("yyearMonth", EJCDateUtil.getShortMonthIntBetween(EJCDateUtil.getMonth(-5), EJCDateUtil.getMonth(0))));
        } else {
            boolQuery.must(QueryBuilders.termQuery("yyear", EJCDateUtil.getLastYear()));
        }
        //f分组
        TermsAggregationBuilder aggregation = AggregationBuilders.terms("timeGroup").field("yyearMonth");

        //合计字段
        List<String> keyList = Arrays.asList(
                "production",
                "partAReport",
                "yingshouweishou",
                "backMoney",
                "openInvince",
                "jiesuanjine",
                "receiveInvince",
                "payMoney",
                "yingfuweifu",
                "zhangmianjine"
        );
        for (String key : keyList) {
            aggregation.subAggregation(AggregationBuilders.sum(key).field(key));
        }

        sourceBuilder.aggregation(aggregation);
        sourceBuilder.query(boolQuery);
        sourceBuilder.trackTotalHits(true);
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        FinanceUseResVO resVO = new FinanceUseResVO();
        try {
            SearchResponse search = client.search(searchRequest, RequestOptions.DEFAULT);
            Aggregations aggregations = search.getAggregations();
            ParsedLongTerms terms = aggregations.get("timeGroup");
            List<? extends Terms.Bucket> buckets = terms.getBuckets();

            ArrayList<String> monthData = new ArrayList<>();
            HashMap<String, FinanceUseResSubVO> subVOMap = new HashMap<>();

            for (Terms.Bucket bucket : buckets) {
                String s = String.valueOf(bucket.getKey());
                if (s.length() > 4) {
                    s = s.substring(0, 4) + "-" + s.substring(4, 6);
                }
                monthData.add(s);
                Aggregations bucketAggregations = bucket.getAggregations();
                for (String key : keyList) {
                    ParsedSum productionSumRes = bucketAggregations.get(key);
                    FinanceUseResSubVO subVO = subVOMap.get(key);
                    if (subVO == null) {
                        subVO = new FinanceUseResSubVO();
                        subVOMap.put(key, subVO);
                    }
                    BigDecimal sumVal = ComputeUtil.toBigDecimal(productionSumRes.getValue());
                    subVO.setTotal(ComputeUtil.safeAdd(subVO.getTotal(), sumVal));
                    List<BigDecimal> monthDataList = subVO.getMonthData();
                    if (monthDataList == null) {
                        monthDataList = new ArrayList<>();
                        subVO.setMonthData(monthDataList);
                    }
                    monthDataList.add(sumVal);
                }

            }
            resVO.setMonthData(monthData);
            resVO.setProduction(subVOMap.get(keyList.get(0)));
            resVO.setPartAReport(subVOMap.get(keyList.get(1)));
            resVO.setShouldRecNotRec(subVOMap.get(keyList.get(2)));
            resVO.setBackMoney(subVOMap.get(keyList.get(3)));
            resVO.setOpenInvince(subVOMap.get(keyList.get(4)));
            resVO.setSettleMoney(subVOMap.get(keyList.get(5)));
            resVO.setReceiveInvince(subVOMap.get(keyList.get(6)));
            resVO.setPayMoney(subVOMap.get(keyList.get(7)));
            resVO.setShouldPayNotPay(subVOMap.get(keyList.get(8)));
            resVO.setBillMoney(subVOMap.get(keyList.get(9)));

        } catch (IOException e) {
            e.printStackTrace();
        }

        return CommonResponse.success("查询资金使用数据成功！", resVO);
    }


    /**
     * @description: 定时任务更新资金使用数据
     * @return:
     * @author songlx
     * @date: 2022/4/14
     */
    @RequestMapping(value = "/executeFinanceUseTask", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> executeFinanceUseTask(@RequestParam(required = false) Integer isUpdAll,HttpServletRequest request) {
        return customReportService.refreshAllTenantData(request, isUpdAll);
    }



    public static void main(String[] args) {
        String s = "2020";
        if (s.length() > 4)
            s = s.substring(0, 4) + "-" + s.substring(4, 6);
        System.out.println(s);
    }
}
