package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.foundation.metadata.vo.MdProjectVO;
import com.ejianc.foundation.report.bean.ColumnEntity;
import com.ejianc.foundation.report.bean.DatasetEntity;
import com.ejianc.foundation.report.bean.RefreshTimeEntity;
import com.ejianc.foundation.report.bean.TableEntity;
import com.ejianc.foundation.report.service.IColumnService;
import com.ejianc.foundation.report.service.IDatasetService;
import com.ejianc.foundation.report.service.IRefreshTimeService;
import com.ejianc.foundation.report.service.ITableService;
import com.ejianc.foundation.report.util.CalculatorUtils;
import com.ejianc.foundation.tenant.api.ITenantApi;
import com.ejianc.framework.cache.redissonlock.RedissonLocker;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.ejianc.framework.skeleton.util.JdkBase64Util;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/data")
@SuppressWarnings("unchecked")
public class DataExtractController implements Serializable {
	private static final Logger logger = LoggerFactory.getLogger(DataExtractController.class);
	private static final long serialVersionUID = 169770636614492806L;

	private final static Integer BATCH_SIZE = 1000;

	@Autowired
	private IDatasetService datasetService;
	@Autowired
	private ITableService tableService;
	@Autowired
	private IColumnService columnService;
	@Autowired
	private RedissonLocker redissonLocker;
	@Autowired
    private ITenantApi tenantApi;
	@Autowired
	private IRefreshTimeService refreshTimeService;
	@Autowired
	private EnvironmentTools environmentTools;

	@RequestMapping(value = "/extract", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> extractData(@RequestBody Map<String, String> param, HttpServletRequest request) {
        String tableIdStr = param.get("tableId");
        String tableCode = param.get("tableCode");

        // 查询数据表
        TableEntity tableEntity = null;
        if(StringUtils.isNotBlank(tableIdStr)) {
            tableEntity = tableService.selectById(Long.valueOf(tableIdStr));
        } else if(StringUtils.isNotBlank(tableCode)) {
            tableEntity = tableService.getByCode(tableCode);
        }
        if(null == tableEntity) {
            return CommonResponse.error("执行失败，没有找到匹配的报表信息！");
        }
        return syncDbDataToReport(tableEntity, param, request.getHeader("authority"), RequestContextHolder.getRequestAttributes());
	}

    /**
     * 刷新所有租户报表数据
     *
     * @return
     */
    @PostMapping(value = "/refreshAllTenantData", consumes = "application/x-www-form-urlencoded")
    public CommonResponse<String> refreshAllTenantData(@RequestParam Map<String, String> params, HttpServletRequest request) {
        logger.info("报表刷新接收到的参数：{}",JSONObject.toJSONString(params));
        JSONArray param = JSONObject.parseArray(params.get("param"));
        String pageIndex = null;
        String pageSize = null;
        if(param.getJSONObject(0).getJSONArray("parameters")!=null){
            JSONArray parameters = param.getJSONObject(0).getJSONArray("parameters");
            for (Object o : parameters) {
                JSONObject pa = (JSONObject) o;
                if ("pageIndex".equals(pa.getString("property"))) {
                    pageIndex = pa.getString("value");
                } else if ("pageSize".equals(pa.getString("property"))) {
                    pageSize = pa.getString("value");
                }
            }
        }
	    Long startTime = System.currentTimeMillis();
	    //将待刷新数据的租户列表分为10份
	    int sperateLen = 10;

	    //1、获取所有有效的租户Id列表
        CommonResponse<List<Long>> tenantIdListResp = tenantApi.getAllValidTenantId();
        if(!tenantIdListResp.isSuccess()) {
            logger.error("获取所有有效租户Id列表失败，{}", tenantIdListResp.getMsg());
            return CommonResponse.error("刷新租户统计数据失败，获取所有有效租户Id列表失败");
        }
        List<Long> tenantIdList = tenantIdListResp.getData();
        if(CollectionUtils.isEmpty(tenantIdList)) {
            return CommonResponse.success("刷新租户统计数据完成，待处理数据的租户列表为空！");
        }
        //2、获取所有有效的报表列表
        QueryWrapper<TableEntity> tableQuery = new QueryWrapper<>();
        tableQuery.eq("dr", BaseVO.DR_UNDELETE);
        tableQuery.isNotNull("dataset_id");
        if(StringUtils.isNotEmpty(pageIndex)&& StringUtils.isNotEmpty(pageSize)){
            tableQuery.last(" LIMIT "+pageIndex+","+pageSize);
        }
        if(StringUtils.isNotBlank(pageIndex)&& StringUtils.isBlank(pageSize)) {
            tableQuery.last(" LIMIT "+pageIndex+",10000");
        }
        List<TableEntity> tableList = tableService.list(tableQuery);
        if(CollectionUtils.isEmpty(tableList)) {
            return CommonResponse.success("刷新租户统计数据完成，未发现需刷新数据的报表信息！");
        }
        logger.info("此次刷新报表数量："+tableList.size()+" LIMIT "+pageIndex+","+pageSize);
        for(TableEntity table : tableList) {
            String dropTblSql = "DROP TABLE IF EXISTS `report_"+table.getIndexName()+"` ";

            tableService.executeSql(dropTblSql);
            String createTblSql = columnService.getCurrentTableCreateSql(table.getId());
            tableService.executeSql(createTblSql);
        }
        //3、多线程分别刷新每个租户的统计数据
        int len = tenantIdList.size() < sperateLen ? tenantIdList.size() : tenantIdList.size() / sperateLen;
        List<Future> handlers = new ArrayList<>();
        ExecutorService threadPool = Executors.newFixedThreadPool(sperateLen);
        List<Long> tenantIds = null;
        for(int i=0; i<=sperateLen && i * len < tenantIdList.size(); i++) {
            tenantIds = tenantIdList.subList(i*len, (i+1)*len < tenantIdList.size() ? (i+1)*len : tenantIdList.size());
            TenantDataRefreshHandle handler = new TenantDataRefreshHandle(tenantIds,
                    request.getHeader("authority"), tableList, RequestContextHolder.getRequestAttributes());

            Future<String> childFuture = threadPool.submit(handler);
            handlers.add(childFuture);
        }

        for(Future<String> f : handlers) {
            try {
                String result = f.get();
                logger.info(result);
            } catch (Exception e) {
                logger.error("刷新租户-{}统计报表数据异常，", StringUtils.join(tenantIds, ","), e);
            }
        }

        Long endTime = System.currentTimeMillis();
        logger.info("刷新租户统计数据完成, 耗时（ms）：{}", endTime - startTime);

        return CommonResponse.success("刷新租户统计数据完成！");
    }
    /**
     * 刷新项目资金报表数据
     *
     * @return
     */
    @PostMapping(value = "/refreshPrjTenantData")
    public CommonResponse<String> refreshPrjTenantData(HttpServletRequest request) {
        Long startTime = System.currentTimeMillis();
        //将待刷新数据的租户列表分为10份
        int sperateLen = 10;

        //1、获取所有有效的租户Id列表
        List<Long> arrayList = new ArrayList();
        arrayList.add(1607618595976462337L);
        arrayList.add(1382528964577914881L);
//        CommonResponse<List<Long>> tenantIdListResp =tenantApi.getAllValidTenantId();
//        if(!tenantIdListResp.isSuccess()) {
//            logger.error("获取所有有效租户Id列表失败，{}", tenantIdListResp.getMsg());
//            return CommonResponse.error("刷新租户统计数据失败，获取所有有效租户Id列表失败");
//        }
        List<Long> tenantIdList = arrayList;
        if(CollectionUtils.isEmpty(tenantIdList)) {
            return CommonResponse.success("刷新租户统计数据完成，待处理数据的租户列表为空！");
        }

        //2、获取所有有效的报表列表
        QueryWrapper<TableEntity> tableQuery = new QueryWrapper<>();
        tableQuery.eq("dr", BaseVO.DR_UNDELETE);
        tableQuery.isNotNull("dataset_id");
        TableEntity tableEntity = tableService.selectById("473856794272874585");
        ArrayList<TableEntity> arrayList1 = new ArrayList();
        arrayList1.add(tableEntity);
        List<TableEntity> tableList =arrayList1 ;
        if(CollectionUtils.isEmpty(tableList)) {
            return CommonResponse.success("刷新租户统计数据完成，未发现需刷新数据的报表信息！");
        }

        Map<String, String> param = new HashMap<>();

        //3、多线程分别刷新每个租户的统计数据
        int len = tenantIdList.size() < sperateLen ? tenantIdList.size() : tenantIdList.size() / sperateLen;
        List<Future> handlers = new ArrayList<>();
        ExecutorService threadPool = Executors.newFixedThreadPool(sperateLen);
        List<Long> tenantIds = null;
        for(int i=0; i<=sperateLen && i * len < tenantIdList.size(); i++) {
            tenantIds = tenantIdList.subList(i*len, (i+1)*len < tenantIdList.size() ? (i+1)*len : tenantIdList.size());
            TenantDataRefreshHandle handler = new TenantDataRefreshHandle(tenantIds,
                    request.getHeader("authority"), tableList, RequestContextHolder.getRequestAttributes());

            Future<String> childFuture = threadPool.submit(handler);
            handlers.add(childFuture);
        }

        for(Future<String> f : handlers) {
            try {
                String result = f.get();
                logger.info(result);
            } catch (Exception e) {
                logger.error("刷新租户-{}统计报表数据异常，", StringUtils.join(tenantIds, ","), e);
            }
        }

        Long endTime = System.currentTimeMillis();
        logger.info("刷新租户统计数据完成, 耗时（ms）：{}", endTime - startTime);

        return CommonResponse.success("刷新租户统计数据完成！");
    }

    class TenantDataRefreshHandle implements Callable<String> {

	    private List<Long> tenantIds;
	    private String authority;
	    private List<TableEntity> tableList;
        private RequestAttributes context;

	    private TenantDataRefreshHandle(List<Long> tenantIds, String authority, List<TableEntity> tableList, RequestAttributes context) {
            this.tableList = tableList;
            this.tenantIds = tenantIds;
            this.authority = authority;
            this.context = context;
        }

        @Override
        public String call() throws Exception {
            Map<String, String> param = new HashMap<>();
            CommonResponse<String> dataRefresh = null;
            for(Long tenantId : tenantIds) {
                param.put("tenantId", tenantId.toString());
                for(TableEntity table : tableList) {
                    dataRefresh = syncDbDataToReport(table, param, authority, context);
                    logger.info("刷新租户-{},报表-[名称:{}，编码：{}]数据结果：{}", tenantId, table.getTableName(), table.getCode(), dataRefresh.getMsg());
                }
            }
            return "租户-["+StringUtils.join(tenantIds, ",")+"]数据刷新任务执行完成！";
        }
    }

    @RequestMapping(value = "/extractByCode", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> extractDataByCode(@RequestBody Map<String, String> param, HttpServletRequest request) {
        // 查询数据表
        TableEntity tableEntity = tableService.getByCode(param.get("tableCode"));
        if(null == tableEntity) {
            return CommonResponse.error("执行失败，没有找到匹配的报表信息！");
        }
        return syncDbDataToReport(tableEntity, param, request.getHeader("authority"), RequestContextHolder.getRequestAttributes());
    }

    private CommonResponse<String> syncDbDataToReport(TableEntity tableEntity, Map<String, String> param, String authority, RequestAttributes context) {
        Long tenantId = null != param.get("tenantId") ? Long.valueOf(param.get("tenantId")) : InvocationInfoProxy.getTenantid();
//        String authority = request.getHeader("authority");

        List<Long> datasetIds = new ArrayList<>(Arrays.asList(tableEntity.getDatasetIds().split(","))).stream().map(s -> Long.valueOf(s)).collect(Collectors.toList());
        boolean hasReq = false;
        final AtomicInteger count = new AtomicInteger(0);
        ConcurrentHashMap<Long, MdProjectVO> projectCache = new ConcurrentHashMap<>();

        String baseHost = environmentTools.getBaseHost(tenantId);
        ExecutorService threadPool = null;
        String lockKey = tenantId+"_"+tableEntity.getId();
        boolean locked = redissonLocker.isLocked(lockKey);
        if(locked){
            throw new BusinessException("数据刷新中，请稍后再获取数据");
        }
        try {
            redissonLocker.lock(lockKey, 1200);
            if(CollectionUtil.isNotEmpty(datasetIds)) {
                threadPool = Executors.newFixedThreadPool(datasetIds.size());
            }
            logger.info("----------------主线程池数量：{}",datasetIds.size());
            List<Future> futureList = new ArrayList<>();
            for(Long datasetId : datasetIds) {
                Callable<List<Map<String, Object>>> mainDatasetCallable = new MainDatasetCallable(baseHost, count, context, datasetId, tableEntity, authority, tenantId, projectCache);
                Future<List<Map<String, Object>>> childFuture = threadPool.submit(mainDatasetCallable);
                futureList.add(childFuture);
            }

            List<Map<String, Object>> allIndexRequest = new ArrayList<>();
            for (Future<List<Map<String, Object>>> future : futureList) {
                List<Map<String, Object>> list = future.get(3000, TimeUnit.SECONDS);

                if(CollectionUtils.isNotEmpty(list)) {
                	logger.info("mainDatasetCallable,组装要插入的数据-------{}",list.size());
                    hasReq = true;
                    for(Map<String, Object> ir : list) {
                    	allIndexRequest.add(ir);
                    }
                }
            }

            if(hasReq) {
                logger.info("插入数据开始------");
                //设置立即更新以解决数据查询不到的问题。
            	Long batchInsertTimes = allIndexRequest.size() / 500  +  (allIndexRequest.size() % 500 == 0 ? 0L : 1L);

                logger.info("------插入数据页数：{}",batchInsertTimes);
            	for(int i=0;i<batchInsertTimes;i++) {
            		int fromIndex = i*500;
            		int toIndex = Math.min(fromIndex + 500, allIndexRequest.size());

                    logger.info("------插入数据第：{}次循环",i);
            		List<Map<String, Object>> insertRequestList = allIndexRequest.subList(fromIndex, toIndex);

                    if(insertRequestList != null && insertRequestList.size() > 0) {
                        List<ColumnEntity> columnList = columnService.queryColumnListByTblId(tableEntity.getId());

                        String insertSql = "INSERT INTO report_" + tableEntity.getIndexName() + " ( `uuid`,`creator_space`,`data_sequence`,";
                        for(ColumnEntity column : columnList) {
                            insertSql += "`"+column.getProperty()+"`,";
                        }
                        insertSql = insertSql.substring(0, insertSql.length() - 1);
                        insertSql += ") VALUES ";
                        for(int index = 0; index < insertRequestList.size(); index++) {
                            Map<String, Object> indexRequest = insertRequestList.get(index);
                            insertSql += "( "+ IdWorker.getId() + ","+tenantId+","+((i*500)+index+1)+",";
                            for(ColumnEntity column : columnList) {
                                String columnType = column.getType();
                                Object columnObject = indexRequest.get(column.getProperty());
                                switch (columnType) {
                                    case "number":
                                    case "decimal":
                                        if(columnObject == null || "null".equals(columnObject)) {
                                            insertSql += "null,";
                                        }else{
                                            insertSql += ""+columnObject+",";
                                        }
                                        break;
                                    case "string":
                                    case "time":
                                    default:
                                        if(columnObject == null || "null".equals(columnObject)) {
                                            insertSql += "null,";
                                        }else{
                                            String paramValue = columnObject.toString();
                                            if(StringUtils.isNotBlank(paramValue)) {
                                                int textLength = paramValue.length();
                                                int end = Math.min(250, textLength);
                                                paramValue = paramValue.substring(0, end);
                                                paramValue = paramValue.replace("\"", "");
                                                paramValue = paramValue.replace("'", "");
                                                paramValue = paramValue.replace("\\", "");
                                                paramValue = paramValue.replace("/", "");
                                                paramValue = paramValue.replace("[", "");
                                                paramValue = paramValue.replace("]", "");
                                                paramValue = paramValue.replace("\n", "");
                                                paramValue = paramValue.replace("\t", "");
                                                paramValue = paramValue.replace(";", "；");
                                                paramValue = paramValue.replace("^", "");
                                                paramValue = paramValue.replace("-", "―");
                                                insertSql += "'"+paramValue+ "',";
                                            }else{
                                                insertSql += "'',";
                                            }
                                        }
                                }

                            }
                            insertSql = insertSql.substring(0, insertSql.length() - 1);
                            insertSql += "),";
                        }
                        insertSql = insertSql.substring(0, insertSql.length() - 1);
                        tableService.executeInsertSql(insertSql);

                        logger.info("------插入数据第：{}次循环结束---------",i);
                    }
            	}
            }

            try{
                QueryWrapper<RefreshTimeEntity> refreshTimeWrapper = new QueryWrapper<>();
                refreshTimeWrapper.eq("table_id", tableEntity.getId());
                refreshTimeWrapper.eq("tenant_id", tenantId);
                RefreshTimeEntity refreshTimeEntity = refreshTimeService.getOne(refreshTimeWrapper);
                if(refreshTimeEntity == null) {
                    refreshTimeEntity = new RefreshTimeEntity();
                    refreshTimeEntity.setTableId(tableEntity.getId());
                    refreshTimeEntity.setLastRefreshTime(new Date());
                    refreshTimeEntity.setTenantId(tenantId);
                    refreshTimeService.save(refreshTimeEntity);
                }else{
                    refreshTimeEntity.setLastRefreshTime(new Date());
                    refreshTimeService.updateById(refreshTimeEntity);
                }
            }catch(Exception e) {
                logger.info("修改刷新时间失败，", e);
            }
        } catch (Exception e) {
            logger.info("查询主数据集失败，", e);
            return CommonResponse.error("查询主数据集失败");
        } finally {
            redissonLocker.unlock(lockKey);
            if(null != threadPool) {
                threadPool.shutdown();
            }
        }

        return CommonResponse.success("执行成功");
    }


	/**
	 * 查询当前空间的所有数据
	 *
	 * @param indexName
	 * @param creatorSpace
	 * @return
	 */
	private Long queryDataSize(String indexName, Long creatorSpace) {
        String dataCountSql = "SELECT COUNT(*) FROM report_"+indexName+" WHERE creator_space = "+creatorSpace;
        Long dataCount = tableService.queryCountBySql(dataCountSql);
		return dataCount;
	}

	class MainDatasetCallable implements Callable<List<Map<String, Object>>> {
		private String baseHost;
		private AtomicInteger count;
		private RequestAttributes context;
		private TableEntity tableEntity;
		private Long datasetId;
		private String authority;
		private Long tenantId;
		private ConcurrentHashMap<Long, MdProjectVO> projectCache;

		public MainDatasetCallable(String baseHost, AtomicInteger count, RequestAttributes context, Long datasetId, TableEntity tableEntity,
								   String authority, Long tenantId, ConcurrentHashMap<Long, MdProjectVO> projectCache) {
			this.baseHost = baseHost;
			this.count = count;
			this.context = context;
			this.tableEntity = tableEntity;
			this.datasetId = datasetId;
			this.authority = authority;
			this.tenantId = tenantId;
			this.projectCache = projectCache;
		}

		@Override
		public List<Map<String, Object>> call() throws Exception {
		    InvocationInfoProxy.setTenantid(tenantId);
            context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
            RequestContextHolder.setRequestAttributes(context);
            List<Map<String, Object>> irList = new ArrayList<>();

			Map<String, Object> resp = new HashMap<>();
			List<JSONObject> mainDatasetList = new ArrayList<>();

			// 查询主数据集
			DatasetEntity datasetEntity = datasetService.selectById(datasetId);

            // 通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(datasetEntity.getMdProjectId().longValue());
            if(null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", datasetEntity.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", datasetEntity.getMdProjectId().toString(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);

//                CommonResponse<MdProjectVO> projectResponse = projectApi.queryDetail(datasetEntity.getMdProjectId());
                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("主数据集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", datasetId, response.getMsg());
                }
            }
            if(null != projectVO) {
                String mainDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";
                JSONObject mainParamJson = new JSONObject();
                JSONObject mainQueryParam = new JSONObject();
                mainQueryParam.put("tenantId", tenantId);
                String sql = "SELECT dsq.* FROM (" + datasetEntity.getSqlContent() + " ) dsq WHERE dsq.tenantId =#{tenantId}";

                mainParamJson.put("sqlContent", JdkBase64Util.encode(sql));
                mainParamJson.put("datasetType", "1");
                mainParamJson.put("params", mainQueryParam);

                String mainResponseStr = ReferHttpClientUtils.postByJson(mainDatasetUrl, JSON.toJSONString(mainParamJson));

                CommonResponse<List<JSONObject>> mainResponse = JSON.parseObject(mainResponseStr, CommonResponse.class);
                if (mainResponse.isSuccess()) {
                    mainDatasetList = mainResponse.getData();

                    // 查询子数据集
                    List<DatasetEntity> childDatasetList = datasetService.queryChildrenByParentId(datasetEntity.getId());

                    // 先根据创建者空间查询当前空间的所有数据,然后删除之
                    Long resultSize = queryDataSize(tableEntity.getIndexName(), tenantId);
                    Long deleteTimes = resultSize / BATCH_SIZE  +  (resultSize % BATCH_SIZE == 0 ? 0L : 1L);
                    logger.info("本次待删除数据总数：{}，每次删除量：{}，执行删除次数：{}", resultSize, BATCH_SIZE, deleteTimes);
                    for (int i = 0; i < deleteTimes; i++) {
                        String deleteSql = "DELETE FROM report_" + tableEntity.getIndexName() + " WHERE creator_space = " + tenantId + " LIMIT " + BATCH_SIZE;
                        tableService.executeSql(deleteSql);
                        logger.info("第{}次执行 索引-{} 删除操作，参数-creator_space：{}", i, tableEntity.getIndexName(), tenantId);
                    }

                    if (mainDatasetList != null && mainDatasetList.size() > 0) {
                    	logger.info("本次待插入数据总数：{}", mainDatasetList.size());
                        // 查询数据列
                        Map<String, Object> paramMap = new HashMap<>();
                        paramMap.put("tableId", tableEntity.getId());
                        paramMap.put("tenantId", tenantId);
                        List<ColumnEntity> columnEntities = columnService.queryTenantColumnList(paramMap);
                        Map<String, String> columnMap = new HashMap<>();
                        Map<String, ColumnEntity> columnEntityMap = new HashMap<>();
                        List<String> columnProperties = new ArrayList<>();
                        if (columnEntities != null && columnEntities.size() > 0) {
                            for (ColumnEntity columnEntity : columnEntities) {
                                if(StringUtils.isNotBlank(columnEntity.getProperty())) {
                                    //属性为空的列认为是复核列
                                    if (StringUtils.isBlank(columnEntity.getFormula())) {
                                        columnEntityMap.put(columnEntity.getProperty(), columnEntity);
                                        columnMap.put(columnEntity.getProperty(), columnEntity.getType());
                                        columnProperties.add(columnEntity.getProperty());
                                    } else {
                                        columnProperties.add(columnEntity.getProperty());
                                        columnMap.put(columnEntity.getProperty(), columnEntity.getFormula());
                                    }
                                }
                            }
                        }

                        try {
                            for (JSONObject mainDataset : mainDatasetList) {
                                if (childDatasetList != null && childDatasetList.size() > 0) {
                                    ExecutorService threadPool = null;
                                    if(CollectionUtil.isNotEmpty(childDatasetList)) {
                                        threadPool = Executors.newFixedThreadPool(childDatasetList.size());
                                    }
                                    logger.info("----------------子线程池数量：{}",childDatasetList.size());
                                    List<Future<List<JSONObject>>> futureList = new ArrayList<>();
                                    for (DatasetEntity childDataset : childDatasetList) {
                                        Callable<List<JSONObject>> childCallable = new ChildDatasetCallable(baseHost, childDataset, mainDataset,
                                                context, tenantId, authority, projectCache);
                                        Future<List<JSONObject>> childFuture = threadPool.submit(childCallable);
                                        futureList.add(childFuture);
                                    }

                                    for (Future<List<JSONObject>> future : futureList) {
                                        List<JSONObject> childList = future.get(3000, TimeUnit.SECONDS);
                                        if (childList != null && !childList.isEmpty()) {
                                            JSONObject childObject = childList.get(0);
                                            logger.info("----------------子线程查询数据：{}",childObject);
                                            if(null != childObject) {
                                                for (Map.Entry<String, Object> entry : childObject.entrySet()) {
                                                    mainDataset.put(entry.getKey(), entry.getValue());
                                                }
                                            }
                                        }
                                    }
                                    if(null != threadPool) {
                                        //线程池关闭
                                        threadPool.shutdown();
                                    }
                                }

                                Map<String, Object> dataMap = new HashMap<>();
                                //排序
                                dataMap.put("creator_space", tenantId);
                                dataMap.put("data_sequence", count.incrementAndGet());
                                String columnType = null;
                                for(String key : columnProperties) {
                                    columnType = columnMap.get(key);
                                    if ("time".equals(columnType)) {
                                        if(null != mainDataset.get(key)) {
                                            dataMap.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                            mainDataset.put(key, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(mainDataset.get(key)));
                                        }
                                    } else if ("string".equals(columnType)
                                            || "number".equals(columnType)
                                            || "decimal".equals(columnType)) {
                                        dataMap.put(key, mainDataset.get(key));
                                        mainDataset.put(key, mainDataset.get(key));
                                    } else if ("pic".equals(columnType)) {

                                    } else { // 计算公式
                                        dataMap.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                        mainDataset.put(key, CalculatorUtils.getResult(columnType, mainDataset));
                                    }
                                }
                                for (String key : columnProperties) {
                                    columnType = columnMap.get(key);
                                    if ("pic".equals(columnType)) {
                                        if(StringUtils.isNotEmpty(columnEntityMap.get(key).getFormatter())){
                                            ScriptEngineManager manager = new ScriptEngineManager();
                                            ScriptEngine engine = manager.getEngineByName("js");
                                            try {
                                                engine.eval(columnEntityMap.get(key).getFormatter());
                                                if(engine instanceof Invocable) {
                                                    Invocable invoke = (Invocable)engine;    // 调用merge方法，并传入两个参数
                                                    Integer r = (Integer)invoke.invokeFunction("getIconByRowData", mainDataset); //调用了js的aa方法
                                                    logger.info("调用js 方法 ---{}，参数----{}  ，结果 ---{}",columnEntityMap.get(key).getFormatter(),mainDataset,r);
                                                    dataMap.put(key, r);
                                                    mainDataset.put(key, r);
                                                }else {
                                                    dataMap.put(key, 1);
                                                    mainDataset.put(key, 1);
                                                }
                                            } catch (ScriptException | NoSuchMethodException e) {
                                                e.printStackTrace();
                                                logger.info("执行js脚本出错！！！",e);
                                                dataMap.put(key, 1);
                                                mainDataset.put(key, 1);
                                            }
                                        }else {
                                            dataMap.put(key, 1);
                                            mainDataset.put(key, 1);
                                        }
                                    }
                                }
                                irList.add(dataMap);
                            }
                        }catch (Exception e){
                            logger.error("******************* 执行SQL查询异常： *******************", e);
                        }
                    }
                } else {
                    logger.error("******************* 执行SQL查询失败：{}  *******************", mainResponse.getMsg());
                }
            }

			return irList;
		}
	}

	/**
	 * 子线程
	 *
	 * @author Administrator
	 *
	 */
	class ChildDatasetCallable implements Callable<List<JSONObject>> {

		private String baseHost;
		private DatasetEntity childDataset;
		private JSONObject mainDataset;
		private RequestAttributes context;
		private Long tenantId;
		private String authority;
        private ConcurrentHashMap<Long, MdProjectVO> projectCache;

		public ChildDatasetCallable() {
		}

		public ChildDatasetCallable(String baseHost, DatasetEntity childDataset, JSONObject mainDataset,
                                    RequestAttributes context, Long tenantId, String authority, ConcurrentHashMap<Long, MdProjectVO> projectCache) {
			this.baseHost = baseHost;
			this.childDataset = childDataset;
			this.mainDataset = mainDataset;
			this.context = context;
			this.tenantId = tenantId;
			this.authority = authority;
            this.projectCache = projectCache;
		}

		@Override
		public List<JSONObject> call() throws Exception {
			context.setAttribute("authority", authority, RequestAttributes.SCOPE_REQUEST);
			RequestContextHolder.setRequestAttributes(context);

			List<JSONObject> dataList = new ArrayList<JSONObject>();
//			CommonResponse<MdProjectVO> projectResponse = projectApi.queryDetail(childDataset.getMdProjectId());

//            CommonResponse<MdProjectVO> projectResponse = JSONObject.parseObject(projectInfoResp, CommonResponse.class);
//             通过元数据查询该数据集属于哪个项目
            MdProjectVO projectVO = projectCache.get(childDataset.getMdProjectId().longValue());
            logger.info("children-----tenantid--------"+tenantId);
            if(null == projectVO) {
                String url = baseHost + "ejc-metadata-web/api/mdProjectApi/queryDetail";
                Map<String, String> param = new HashMap<>();
                Map<String, String> header = new HashMap<>();
                param.put("id", childDataset.getMdProjectId().toString());
                header.put("authority", authority);
                String projectInfoResp = HttpTookit.get(url, param, header);
                logger.info("查询【id-{}】元数据项目信息结果：{}", childDataset.getMdProjectId(), projectInfoResp);
                CommonResponse<JSONObject> response = JSONObject.parseObject(projectInfoResp, CommonResponse.class);
                if (response.isSuccess()) {
                    projectVO = JSONObject.parseObject(JSONObject.toJSONString(response.getData()), MdProjectVO.class);
                    projectCache.put(projectVO.getId().longValue(), projectVO);
                } else {
                    logger.error("数据子集【id-{}】查询数据失败, 其对应元数据查询失败, 原因：{}", childDataset.getId(), response.getMsg());
                }
            }

			if(null != projectVO) {
                String childDatasetUrl = baseHost + projectVO.getProjectName() + "/common/report/parse";

                JSONObject childParamJson = new JSONObject();
                JSONObject childQueryParam = new JSONObject();
                childQueryParam.put("tenantId", tenantId);
                if (StringUtils.isNotBlank(childDataset.getChildParam())) {
                    String[] childParamArr = childDataset.getChildParam().split(",");
                    for (String childParamStr : childParamArr) {
                        childQueryParam.put(childParamStr, mainDataset.getString(childParamStr));
                    }
                }
                childParamJson.put("sqlContent", JdkBase64Util.encode(childDataset.getSqlContent()));
                childParamJson.put("datasetType", "2");
                childParamJson.put("params", childQueryParam);

                String childResponseStr = ReferHttpClientUtils.postByJson(childDatasetUrl, JSON.toJSONString(childParamJson));

                CommonResponse<List<JSONObject>> childResponse = JSON.parseObject(childResponseStr, CommonResponse.class);
                if (childResponse.isSuccess()) {
                    return childResponse.getData();
                }else {
                    logger.info("请求{}, 执行sql-{}, 参数-{}, 结果-{}",childDatasetUrl, childDataset.getSqlContent(), childQueryParam, JSONObject.toJSONString(childResponse));
                }
            }
			return dataList;
		}

	}


    public static void main(String[] args) {
        String str = "dddddd\\";
        System.out.println(str.replace("\\",""));
    }
}
