package com.ejianc.foundation.report.controller;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.foundation.metadata.api.IMdProjectApi;
import com.ejianc.foundation.metadata.vo.MdProjectVO;
import com.ejianc.foundation.report.bean.DatasetEntity;
import com.ejianc.foundation.report.service.IDatasetService;
import com.ejianc.foundation.report.vo.DatasetVO;
import com.ejianc.foundation.support.vo.ReferShowfieldVO;
import com.ejianc.foundation.support.vo.ReferVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.EnvironmentTools;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import com.ejianc.framework.skeleton.util.JdkBase64Util;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@RestController
@RequestMapping("/dataset/")
public class DatasetController implements Serializable {

	private static final long serialVersionUID = 7563509787599976846L;

	private Logger logger = LoggerFactory.getLogger(this.getClass());

	@Value("${oms.tenantid}")
	private String OMS_TENANT;
	@Value("${refer.base-host:null}")
	private String baseHost;
	
	@Autowired
	private IDatasetService datasetService;
	@Autowired
	private IMdProjectApi projectApi;
	@Autowired
	private EnvironmentTools environmentTools;
	
	/**
	 * 新增或者修改
	 * 
	 * @param datasetVo
     * @return
	 */
	@RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> saveOrUpdate(@RequestBody DatasetVO datasetVo) {
		String checkMsg = this.sqlCheck(datasetVo.getSqlContent());
		if(StringUtils.isNotBlank(checkMsg)) {
			return CommonResponse.error(checkMsg);
		}

		if(datasetVo.getId() != null && datasetVo.getId() > 0) {
			DatasetEntity uniqueBean = datasetService.selectById(datasetVo.getId());
			uniqueBean.setMdProjectId(datasetVo.getMdProjectId());
			uniqueBean.setCode(datasetVo.getCode());
			uniqueBean.setName(datasetVo.getName());
			uniqueBean.setSqlContent(datasetVo.getSqlContent());
			uniqueBean.setChildParam(datasetVo.getChildParam());
			uniqueBean.setSequence(datasetVo.getSequence());
			
			datasetService.saveOrUpdate(uniqueBean, false);
			
			return CommonResponse.success("修改成功");
		} else {
			DatasetEntity saveBean = BeanMapper.map(datasetVo, DatasetEntity.class);
			
			datasetService.saveOrUpdate(saveBean, false);
			
			return CommonResponse.success("保存成功");
		}
	}

	private String sqlCheck(String sqlContent) {
		if(StringUtils.isNotBlank(sqlContent)) {
			try {
				String[] sqlContentArr = sqlContent.split("\\s+");
				for(String field: sqlContentArr) {
					field = field.toLowerCase();
					if("drop".equals(field.toLowerCase()) || "insert".equals(field) || "update".equals(field) || "delete".equals(field)) {
						throw new BusinessException("执行sql中不允许含有drop、insert、update、delete关键字。");
					}
				}
			}catch (Exception e) {}
			if(sqlContent.toLowerCase().indexOf(";") != -1) {
				throw new BusinessException("执行sql中不允许含有 ; 关键字。");
			}
		}
		return null;
	}

	@PostMapping(value = "sqlRunAbleCheck")
	@ResponseBody
	public CommonResponse<String> sqlRunAbleCheck(@RequestBody DatasetVO datasetVo) {
		// 查询主数据集
		DatasetEntity datasetEntity = datasetService.selectById(datasetVo.getId());

		if(StringUtils.isBlank(datasetEntity.getSqlContent())) {
			return CommonResponse.error("请先填写数据集查询SQL再进行调试！");
		}

		String checkMsg = this.sqlCheck(datasetEntity.getSqlContent());
		if(StringUtils.isNotBlank(checkMsg)) {
			return CommonResponse.error(checkMsg);
		}

		if(null == datasetEntity.getMdProjectId()) {
			return CommonResponse.error("调试失败，数据集未关联元数据!");
		}

		// 通过元数据查询该数据集属于哪个项目
		CommonResponse<MdProjectVO> projectVOResp = projectApi.queryDetail(datasetEntity.getMdProjectId());
		if(!projectVOResp.isSuccess()) {
			return CommonResponse.error("调试失败，获取数据集对应元数据信息获取失败，" + projectVOResp.getMsg());
		}

		String mainDatasetUrl = environmentTools.getBaseHost() + projectVOResp.getData().getProjectName() + "/common/report/parse";
		JSONObject mainParamJson = new JSONObject();
		JSONObject mainQueryParam = new JSONObject();
		mainQueryParam.put("tenantId", InvocationInfoProxy.getTenantid());
		String sql = "SELECT count(1) as total FROM (" + datasetEntity.getSqlContent() + " ) dsq WHERE dsq.tenantId =#{tenantId}";

		mainParamJson.put("sqlContent", JdkBase64Util.encode(sql));
		mainParamJson.put("datasetType", "1");
		mainParamJson.put("params", mainQueryParam);

		try {
			String mainResponseStr = ReferHttpClientUtils.postByJson(mainDatasetUrl, JSON.toJSONString(mainParamJson));

			CommonResponse<List<JSONObject>> mainResponse = JSON.parseObject(mainResponseStr, CommonResponse.class);

			if(!mainResponse.isSuccess()) {
				logger.info("请求-【{}】,参数-【{}】,响应结果：【{}】", mainDatasetUrl, JSONObject.toJSONString(mainParamJson), mainResponseStr);
				return CommonResponse.error("调试失败，调用数据集所属项目服务失败，" + mainResponse.getMsg());
			}

			return CommonResponse.success("查询本租户下符合条件数据总条数：" + mainResponse.getData().get(0).get("total"));
		} catch (Exception e) {
			logger.error("请求-【{}】,参数-【{}】异常，", mainDatasetUrl, JSONObject.toJSONString(mainParamJson), e);
			return CommonResponse.error("调试失败，调用数据集所属项目服务失败！");
	}
	}

	/**
	 * 查询待分页的列表
	 * 
	 * @param queryParam
	 * @return
	 */
	@RequestMapping(value = "/pageList", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<JSONObject> pageList(@RequestBody QueryParam queryParam) {
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, OMS_TENANT));
		queryParam.getOrderMap().put("sequence", QueryParam.ASC);
		IPage<DatasetEntity> pageData = datasetService.queryPage(queryParam);
		JSONObject jsonObject = new JSONObject();
		jsonObject.put("data", pageData);
		return CommonResponse.success(jsonObject);
	}
	
	/**
	 * 根据主键ID查询详情
	 * 
	 * @param id
	 * @return
	 */
	@RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
	@ResponseBody
	public CommonResponse<DatasetVO> queryDetail(@RequestParam Long id) {
		DatasetEntity datasetEntity = datasetService.selectById(id);
		if(datasetEntity != null) {
			DatasetVO datasetVo = BeanMapper.map(datasetEntity, DatasetVO.class);
			return CommonResponse.success(datasetVo);
		}
		return null;
	}
	
	/**
	 * 删除
	 * 
	 * @param ids
	 * @return
	 */
	@RequestMapping(value = "/delete", method = RequestMethod.POST)
	@ResponseBody
	public CommonResponse<String> delete(@RequestBody List<Long> ids) {
		datasetService.deleteByIds(ids);
		return CommonResponse.success("删除成功");
	}
	
	@RequestMapping(value = "refDatasetPage", method = RequestMethod.GET)
	@ResponseBody
    public CommonResponse<IPage<DatasetVO>> refDatasetPage(
		@RequestParam(defaultValue = "1") int pageNumber,
		@RequestParam(defaultValue = "10") int pageSize,
		@RequestParam(required=false) String relyCondition,
		@RequestParam(required=false) String searchText,
		@RequestParam(required=false) String condition,
		@RequestParam(required=false) String searchObject
		) {
    	QueryParam queryParam = new QueryParam();
		queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, OMS_TENANT));
		queryParam.getOrderMap().put("sequence", QueryParam.ASC);
		queryParam.setPageSize(pageSize);
		queryParam.setPageIndex(pageNumber);
		queryParam.setSearchText(searchText);
		queryParam.getFuzzyFields().add("name");
		queryParam.getFuzzyFields().add("code");

		IPage<DatasetVO> voPage = null;
		IPage<DatasetEntity> pageData = datasetService.queryPage(queryParam);
        if(pageData!=null){
            voPage = new Page<>();
            voPage.setCurrent(pageData.getCurrent());
            voPage.setPages(pageData.getPages());
            voPage.setTotal(pageData.getTotal());
            voPage.setSize(queryParam.getPageSize());
            voPage.setRecords(BeanMapper.mapList(pageData.getRecords(),DatasetVO.class));
        }
        return CommonResponse.success("查询成功！", voPage);
    }

    /**
	 * 组织-用户参照元数据
	 * 
	 * @param refCode
	 * @return
	 */
	@RequestMapping(value = "/findByCode", method = RequestMethod.GET)
	@ResponseBody
	public CommonResponse<ReferVO> findByCode(@RequestParam String refCode) {
		ReferVO refer = new ReferVO();
		List<ReferShowfieldVO> gridHeaders = new ArrayList<>();
		String referBaseHost = "";
		if(StringUtils.isNotBlank(baseHost)&& !"null".equals(baseHost)){
			referBaseHost = baseHost;
		}else{
			referBaseHost = environmentTools.getBaseHost();
		}
		switch(refCode){
			case "dataset":
				refer.setRefCode("dataset");
				refer.setRefName("数据集");
				refer.setRefType("grid");
				refer.setDataurl(referBaseHost+"ejc-report-web/dataset/refDatasetPage");
//				refer.setDataurl("http://127.0.0.1:8080/"+"ejc-report-web/dataset/refDatasetPage");
				refer.setIdField("id");
				refer.setCodeField("code");
				refer.setNameField("name");
				refer.setId(new Date().getTime());
				refer.setInnerFilter(true);
				refer.setInnerFilterColumns("code,name");
				
				ReferShowfieldVO idHeader = new ReferShowfieldVO();
				idHeader.setId((long) 1);
				idHeader.setName("主键");
				idHeader.setCode("id");
				idHeader.setHidden(true);
				idHeader.setShowOrder(1);
				idHeader.setType("string");
				gridHeaders.add(idHeader);

				ReferShowfieldVO codeHeader = new ReferShowfieldVO();
				codeHeader.setId((long) 2);
				codeHeader.setName("数据集编码");
				codeHeader.setCode("code");
				codeHeader.setHidden(false);
				codeHeader.setShowOrder(2);
				codeHeader.setType("string");
				gridHeaders.add(codeHeader);

				ReferShowfieldVO nameHeader = new ReferShowfieldVO();
				nameHeader.setId((long) 3);
				nameHeader.setName("数据集名称");
				nameHeader.setCode("name");
				nameHeader.setHidden(false);
				nameHeader.setShowOrder(3);
				nameHeader.setType("string");
				gridHeaders.add(nameHeader);

				ReferShowfieldVO urlHeader = new ReferShowfieldVO();
				urlHeader.setId((long) 4);
				urlHeader.setName("服务地址");
				urlHeader.setCode("url");
				urlHeader.setHidden(false);
				urlHeader.setShowOrder(4);
				urlHeader.setType("string");
				gridHeaders.add(urlHeader);
				
				refer.setGridheaders(gridHeaders);
				break;
		}
		
		return CommonResponse.success(refer);
	}
	
}
