package com.ejianc.business.pay.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.enums.ConfirmStatusEnum;
import com.ejianc.business.pay.mapper.PayMapper;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.business.pay.vo.PayVO;
import com.ejianc.business.payer.bean.PayerDetailEntity;
import com.ejianc.business.payer.service.IpayerDetailService;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 风险金缴纳表
 *
 * @author baipengyan
 */
@Service("payService")
public class PayServiceImpl extends BaseServiceImpl<PayMapper, PayEntity> implements IPayService {

	private final SessionManager sessionManager;
	private final IpayerDetailService payerDetailService;

	public PayServiceImpl(SessionManager sessionManager, IpayerDetailService payerDetailService) {
		this.sessionManager = sessionManager;
		this.payerDetailService = payerDetailService;
	}

	/**
	 * 获取历史版本缴纳金额
	 *
	 * @param userId 用户id
	 * @param orgId  组织id
	 *
	 * @return 历史版本缴纳金额
	 */
	@Override
	public BigDecimal fetchLastPayTaxMny(Long userId, Long orgId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.select("ifnull(sum(pay_tax_mny),0) as lastPayTaxMny")
				.eq("user_id", userId)
				.eq("org_id", orgId);
		return (BigDecimal) super.getMap(queryWrapper).get("lastPayTaxMny");
	}


	/**
	 * 保存前校验：一个员工+员工明细只可以有一个未生效的单据
	 *
	 * @param userId 用户id
	 * @param orgId  组织id
	 */
	@Override
	public void checkBeforeSave(Long userId, Long orgId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.select("ifnull(count(*),0) as count")
				.eq("user_id", userId)
				.eq("org_id", orgId)
				.ne("confirm_status", ConfirmStatusEnum.CONFIRM.getCode());
		if (1 == (Long) super.getMap(queryWrapper).get("count")) {
			throw new BusinessException("一个员工的员工明细只可以有一个未生效的单据");
		}
	}


	/**
	 * 获取当前人员缴纳明细的最大缴纳日期
	 *
	 * @param userId 用户id
	 * @param orgId  组织id
	 *
	 * @return 最大缴纳日期
	 */
	@Override
	public Date fetchMaxPayDate(Long userId, Long orgId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.select("max(pay_date) as payDate")
				.eq("user_id", userId)
				.eq("org_id", orgId)
				.eq("confirm_status", ConfirmStatusEnum.CONFIRM.getCode());
		if (super.getMap(queryWrapper) == null) {
			return null;
		}
		return (Date) super.getMap(queryWrapper).get("payDate");
	}


	/**
	 * 是否是最新单据
	 *
	 * @param billId 单据id
	 * @param userId 用户id
	 * @param orgId  组织id
	 *
	 * @return 是否是最新单据
	 */
	@Override
	public boolean isLatestBill(Long billId, Long userId, Long orgId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("user_id", userId)
				.eq("org_id", orgId)
				.orderByDesc("pay_date");
		List<PayEntity> list = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(list)) {
			return list.get(0).getId().equals(billId);
		}
		return true;
	}


	/**
	 * 风险金缴纳：批量确认
	 *
	 * @param pays 要确认的单据信息
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public void batchConfirm(List<PayVO> pays) {
		List<PayEntity> payEntities = BeanMapper.mapList(pays, PayEntity.class);
		UserContext userContext = sessionManager.getUserContext();

		// 缴纳明细转Map<Long, PayEntity>
		Map<Long, PayEntity> payMap = payEntities.stream().collect(Collectors.toMap(PayEntity::getSourceBillId, Function.identity()));

		// 员工明细ids
		ArrayList<Long> ids = new ArrayList<>();

		for (Map.Entry<Long, PayEntity> entry : payMap.entrySet()) {
			PayEntity pay = entry.getValue();

			pay.setConfirmStatus(ConfirmStatusEnum.CONFIRM.getCode());
			pay.setConfirmStatusName(ConfirmStatusEnum.CONFIRM.getName());
			pay.setConfirmDate(new Date());
			pay.setConfirmUserId(userContext.getUserId());
			pay.setConfirmUserCode(userContext.getUserCode());
			pay.setConfirmUserName(userContext.getUserName());

			// 员工明细ids
			ids.add(pay.getSourceBillId());
		}

		// 批量查询员工明细信息
		Collection<PayerDetailEntity> payerDetails = payerDetailService.listByIds(ids);

		// 更新员工明细表：是否分期、分期金额、分期数、每期缴纳金额、实际已缴纳金额、实际已缴纳比例、剩余缴纳金额
		for (PayerDetailEntity pd : payerDetails) {
			PayEntity pay = payMap.get(pd.getId());
			pd.setPeriodizationFlag(Integer.valueOf(pay.getInstallmentFlag()));
			pd.setPeriodizationMny(pay.getInstallmentTaxMny());
			pd.setPeriodizationNum(pay.getInstallmentNum());
			pd.setEachPeriodizationMny(pay.getPayTaxMny());
			// 实际缴纳金额=员工缴纳+员工退还+员工调动+员工调入
			pd.setActualPayerMny(pay.getTotalTaxMny());
			pd.setActualPayerScale(pay.getPayScale());
			pd.setRemainderPayerMny(pay.getRemainPayTaxMny());
		}

		// 执行批量更新员工明细表
		payerDetailService.saveOrUpdateBatch(payerDetails, 10);

		// 执行批量更新
		super.saveOrUpdateBatch(payEntities, 10);
	}
}
