package com.ejianc.business.repay.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.enums.ConfirmStatusEnum;
import com.ejianc.business.pay.enums.InstallmentFlagEnum;
import com.ejianc.business.pay.enums.PayTypeEnum;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.business.payer.bean.PayerDetailEntity;
import com.ejianc.business.payer.service.IpayerDetailService;
import com.ejianc.business.repay.bean.RepayEntity;
import com.ejianc.business.repay.enums.RepayTypeEnum;
import com.ejianc.business.repay.mapper.RepayMapper;
import com.ejianc.business.repay.service.IRepayService;
import com.ejianc.business.repay.vo.RepayVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 风险金退还表
 *
 * @author baipengyan
 */
@Service("repayService")
public class RepayServiceImpl extends BaseServiceImpl<RepayMapper, RepayEntity> implements IRepayService {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private static final String PAY_BILL_CODE = "EJCBT202206000055";
	private static final String TEN_THOUSAND = "10000";

	private final SessionManager sessionManager;
	private final IpayerDetailService payerDetailService;

	private final IPayService payService;

	public RepayServiceImpl(SessionManager sessionManager, IpayerDetailService payerDetailService, IPayService payService) {
		this.sessionManager = sessionManager;
		this.payerDetailService = payerDetailService;
		this.payService = payService;
	}

	/**
	 * 是否是最新单据
	 *
	 * @param billId 单据id
	 * @param userId 用户id
	 * @param orgId  组织id
	 *
	 * @return 是否是最新单据
	 */
	@Override
	public boolean isLatestBill(Long billId, Long userId, Long orgId) {
		QueryWrapper<RepayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("user_id", userId)
				.eq("org_id", orgId)
				.orderByDesc("repay_date");
		List<RepayEntity> list = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(list)) {
			return list.get(0).getId().equals(billId);
		}
		return true;
	}

	/**
	 * 风险金退还：批量确认
	 *
	 * @param repays 要确认的单据信息
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public void batchConfirm(List<RepayVO> repays) {

		List<RepayEntity> repayEntities = BeanMapper.mapList(repays, RepayEntity.class);
		UserContext userContext = sessionManager.getUserContext();

		ArrayList<RepayEntity> repayEntityList = new ArrayList<>();

		// 退还
		List<RepayEntity> repayList = repayEntities.stream().filter(repay -> RepayTypeEnum.REPAY.getCode().equals(repay.getType())).collect(Collectors.toList());
		if (CollectionUtils.isNotEmpty(repayList)) {
			// 缴纳明细转Map<Long, RepayEntity>
			Map<Long, RepayEntity> repayMap = repayList.stream().collect(Collectors.toMap(RepayEntity::getSourceBillId, Function.identity()));

			// 员工明细ids
			ArrayList<Long> ids = new ArrayList<>();

			for (Map.Entry<Long, RepayEntity> entry : repayMap.entrySet()) {
				RepayEntity repay = entry.getValue();

				repay.setConfirmStatus(ConfirmStatusEnum.CONFIRM.getCode());
				repay.setConfirmStatusName(ConfirmStatusEnum.CONFIRM.getName());
				repay.setConfirmDate(new Date());
				repay.setConfirmUserId(userContext.getUserId());
				repay.setConfirmUserCode(userContext.getUserCode());
				repay.setConfirmUserName(userContext.getUserName());
				repayEntityList.add(repay);

				// 员工明细ids
				ids.add(repay.getSourceBillId());
			}

			// 批量查询员工明细信息
			Collection<PayerDetailEntity> payerDetails = payerDetailService.listByIds(ids);

			// 更新员工明细表：是否分期、分期金额、分期数、每期缴纳金额、实际已缴纳金额、实际已缴纳比例、剩余缴纳金额
			for (PayerDetailEntity pd : payerDetails) {
				RepayEntity repay = repayMap.get(pd.getId());
				// 实际缴纳金额=员工缴纳+员工退还+员工调动+员工调入
				BigDecimal actualPayerMny = pd.getActualPayerMny().subtract(repay.getRepayTaxMny(), new MathContext(2, RoundingMode.HALF_UP));
				pd.setActualPayerMny(actualPayerMny);
				pd.setActualPayerScale(actualPayerMny.divide(pd.getPayerMny(), 2, RoundingMode.HALF_UP));
				pd.setRemainderPayerMny(pd.getPayerMny().subtract(actualPayerMny));
			}

			// 执行批量更新员工明细表
			payerDetailService.saveOrUpdateBatch(payerDetails, 10);
		}


		// 调动
		List<RepayEntity> callInList = repayEntities.stream().filter(repay -> RepayTypeEnum.CALL_IN.getCode().equals(repay.getType())).collect(Collectors.toList());
		if (CollectionUtils.isNotEmpty(callInList)) {
			ArrayList<PayEntity> pays = new ArrayList<>();
			ArrayList<PayerDetailEntity> pds = new ArrayList<>();

			// 缴纳明细转Map<Long, RepayEntity>
			Map<Long, RepayEntity> repayMap = callInList.stream().collect(Collectors.toMap(RepayEntity::getSourceBillId, Function.identity()));

			// 员工明细ids
			ArrayList<Long> ids = new ArrayList<>();

			for (Map.Entry<Long, RepayEntity> entry : repayMap.entrySet()) {
				RepayEntity repay = entry.getValue();

				repay.setConfirmStatus(ConfirmStatusEnum.CONFIRM.getCode());
				repay.setConfirmStatusName(ConfirmStatusEnum.CONFIRM.getName());
				repay.setConfirmDate(new Date());
				repay.setConfirmUserId(userContext.getUserId());
				repay.setConfirmUserCode(userContext.getUserCode());
				repay.setConfirmUserName(userContext.getUserName());
				repayEntityList.add(repay);

				// 员工明细ids
				ids.add(repay.getSourceBillId());

				// 查询新单位的员工明细信息
				QueryWrapper<PayerDetailEntity> pdWrapper = new QueryWrapper<>();
				pdWrapper.eq("user_id", repay.getUserId());
				pdWrapper.eq("payer_org_id", repay.getTransferOrgId());
				PayerDetailEntity npd = payerDetailService.getOne(pdWrapper);
				if (npd == null) {
					logger.error("员工明细不存在，userId={},payerOrgId={}", repay.getUserId(), repay.getTransferOrgId());
					throw new BusinessException("员工明细不存在，请联系管理员！");
				}
				// 新单位员工明细历史缴纳金额
				BigDecimal actualPayerMny1 = npd.getActualPayerMny();
				BigDecimal actualPayerMny = actualPayerMny1.add(repay.getRepayTaxMny(), new MathContext(2, RoundingMode.HALF_UP));
				npd.setActualPayerMny(actualPayerMny);
				npd.setActualPayerScale(actualPayerMny.divide(npd.getPayerMny(), 2, RoundingMode.HALF_UP));
				npd.setRemainderPayerMny(npd.getPayerMny().subtract(actualPayerMny, new MathContext(2, RoundingMode.HALF_UP)));

				// 在新单位生成一条此员工缴纳的单据信息，且单据状态为审核通过，确认状态为已确认,缴纳类型为调入
				PayEntity pay = new PayEntity();
				pay.setOrgId(repay.getTransferOrgId());
				pay.setOrgCode(repay.getTransferOrgCode());
				pay.setOrgName(repay.getTransferOrgName());
				pay.setEmployeeId(repay.getEmployeeId());
				pay.setEmployeeName(repay.getEmployeeName());
				pay.setDepartmentId(repay.getDepartmentId());
				pay.setDepartmentCode(repay.getDepartmentCode());
				pay.setDepartmentName(repay.getDepartmentName());
				pay.setCommitDate(new Date());
				pay.setCommitUserName(userContext.getUserName());
				pay.setCommitUserCode(userContext.getUserCode());
				pay.setEffectiveDate(new Date());
				pay.setBillCode(PAY_BILL_CODE);
				pay.setBillName("风险金缴纳");
				pay.setBillType("风险抵押金");
				pay.setPayerDetailId(repay.getPayerDetailId());
				pay.setPayerDetailCode(repay.getPayerDetailCode());
				pay.setPayerDetailName(repay.getPayerDetailName());
				pay.setUserId(repay.getUserId());
				pay.setUserCode(repay.getUserCode());
				pay.setUserName(repay.getUserName());
				pay.setSourceUserId(npd.getUserId());
				pay.setSourceUserCode(npd.getPhone());
				pay.setSourceUserName(npd.getName());
				pay.setSourceBillId(npd.getId());
				pay.setSourceBillCode(npd.getPayerCode());
				pay.setSourceBillName(npd.getPayerName());
				pay.setSourceBillType(null);
				pay.setSourceOrgId(npd.getPayerOrgId());
				pay.setSourceOrgCode(npd.getPayerOrgCode());
				pay.setSourceOrgName(npd.getPayerOrgName());
				pay.setSourcePeriodizationFlag(npd.getPeriodizationFlag());
				pay.setSourcePayerMny(npd.getPayerMny());
				pay.setSourceActualPayerMny(npd.getActualPayerMny());
				pay.setSourcePayScale(npd.getActualPayerScale());
				pay.setSourceRemainderPayerMny(npd.getRemainderPayerMny());
				pay.setSourcePeriodizationMny(npd.getPeriodizationMny());
				pay.setSourcePeriodizationNum(npd.getPeriodizationNum());
				pay.setType(PayTypeEnum.CALL_IN.getCode());
				pay.setTypeName(PayTypeEnum.CALL_IN.getName());
				pay.setPayerMny(npd.getPayerMny());
				pay.setPayTaxMny(repay.getRepayTaxMny());
				pay.setLastPayTaxMny(actualPayerMny1);
				pay.setTotalTaxMny(npd.getActualPayerMny());
				pay.setPayScale(npd.getActualPayerScale());
				pay.setRemainPayTaxMny(npd.getRemainderPayerMny());
				String code = npd.getPeriodizationFlag() != null ? String.valueOf(npd.getPeriodizationFlag()) : "0";
				pay.setInstallmentFlag(code);
				pay.setInstallmentFlagName(InstallmentFlagEnum.getEnumByCode(code).getName());
				pay.setPayDate(new Date());
				pay.setMemo(repay.getMemo());

				pay.setConfirmDate(new Date());
				pay.setConfirmUserId(userContext.getUserId());
				pay.setConfirmUserName(userContext.getUserName());
				pay.setConfirmUserCode(userContext.getUserCode());


				pay.setBillState(BillStateEnum.PASSED_STATE.getBillStateCode());
				pay.setBillStateName(BillStateEnum.PASSED_STATE.getDescription());
				pay.setConfirmStatus(ConfirmStatusEnum.CONFIRM.getCode());
				pay.setConfirmStatusName(ConfirmStatusEnum.CONFIRM.getName());
				pay.setType(PayTypeEnum.CALL_IN.getCode());
				pay.setTypeName(PayTypeEnum.CALL_IN.getName());

				pays.add(pay);
				pds.add(npd);
			}

			// 执行批量更新员工明细表-新的
			payerDetailService.saveOrUpdateBatch(pds, 10);

			// 新增缴纳申请
			payService.saveOrUpdateBatch(pays, 10);

			// 批量查询员工明细信息
			Collection<PayerDetailEntity> payerDetails = payerDetailService.listByIds(ids);

			// 更新员工明细表：实际已缴纳金额、实际已缴纳比例、剩余缴纳金额
			for (PayerDetailEntity pd : payerDetails) {
				RepayEntity repay = repayMap.get(pd.getId());
				// 实际缴纳金额=员工缴纳+员工退还+员工调动+员工调入
				BigDecimal actualPayerMny = pd.getActualPayerMny().subtract(repay.getRepayTaxMny(), new MathContext(2, RoundingMode.HALF_UP));
				pd.setActualPayerMny(actualPayerMny);
				pd.setActualPayerScale(actualPayerMny.divide(pd.getPayerMny(), 2, RoundingMode.HALF_UP));
				pd.setRemainderPayerMny(pd.getPayerMny().subtract(actualPayerMny, new MathContext(2, RoundingMode.HALF_UP)));
			}

			// 执行批量更新员工明细表-旧的
			payerDetailService.saveOrUpdateBatch(payerDetails, 10);
		}

		// 执行批量更新
		super.saveOrUpdateBatch(repayEntityList, 10);
	}


	/**
	 * 校验：一个员工+员工明细只可以有一个未生效的单据（退还+调动+缴纳）
	 *
	 * @param userId 用户id
	 * @param orgId  组织id
	 * @param info   缴纳/退还
	 */
	@Override
	public CommonResponse<String> checkBeforeSave(Long userId, Long orgId, String info) {
		// 查询未生效的缴纳单
		QueryWrapper<PayEntity> payWrapper = new QueryWrapper<>();
		payWrapper.select("ifnull(count(*),0) as count")
				.eq("user_id", userId)
				.eq("org_id", orgId)
				.notIn("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		Long pay = (Long) payService.getMap(payWrapper).get("count");
		if (pay != 0) {
			return CommonResponse.error("该员工有未生效的缴纳单据，不支持" + info);
		}

		// 查询未生效的退还+调动
		QueryWrapper<RepayEntity> repayWrapper = new QueryWrapper<>();
		repayWrapper.select("ifnull(count(*),0) as count")
				.eq("user_id", userId)
				.eq("org_id", orgId)
				.notIn("bill_state", Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode()));
		Long repay = (Long) super.getMap(repayWrapper).get("count");
		if (repay != 0) {
			return CommonResponse.error("该员工有未生效的退还单据，不支持" + info);
		}
		return CommonResponse.success("校验成功");
	}

	/**
	 * 校验是否可以调动
	 *
	 * @param userId        用户id
	 * @param transferOrgId 调动组织id
	 *
	 * @return 是否可以调动
	 */
	@Override
	public Boolean checkTransferOrg(Long userId, Long transferOrgId) {
		QueryWrapper<PayerDetailEntity> pdWrapper = new QueryWrapper<>();
		pdWrapper.eq("user_id", userId);
		pdWrapper.eq("payer_org_id", transferOrgId);
		PayerDetailEntity npd = payerDetailService.getOne(pdWrapper);
		return npd != null;
	}
}
