package com.ejianc.business.pay.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.pay.bean.PayEntity;
import com.ejianc.business.pay.enums.ConfirmStatusEnum;
import com.ejianc.business.pay.mapper.PayMapper;
import com.ejianc.business.pay.service.IPayService;
import com.ejianc.business.pay.vo.PayVO;
import com.ejianc.business.payer.bean.PayerDetailEntity;
import com.ejianc.business.payer.service.IPayerDetailService;
import com.ejianc.business.repay.bean.RepayEntity;
import com.ejianc.business.repay.service.IRepayService;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 风险金缴纳表
 *
 * @author baipengyan
 */
@Service("payService")
public class PayServiceImpl extends BaseServiceImpl<PayMapper, PayEntity> implements IPayService {

	private final SessionManager sessionManager;
	private final IPayerDetailService payerDetailService;

	@Autowired
	private IRepayService repayService;

	public PayServiceImpl(SessionManager sessionManager, IPayerDetailService payerDetailService) {
		this.sessionManager = sessionManager;
		this.payerDetailService = payerDetailService;
	}


	/**
	 * 获取当前人员缴纳明细的最大缴纳日期
	 *
	 * @param sourceBillId 来源单据id
	 *
	 * @return 最大缴纳日期
	 */
	@Override
	public Date fetchMaxPayDate(Long sourceBillId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.select("max(pay_date) as payDate")
				.eq("source_bill_id", sourceBillId)
				.eq("confirm_status", ConfirmStatusEnum.CONFIRM.getCode());
		if (super.getMap(queryWrapper) == null) {
			return null;
		}
		return (Date) super.getMap(queryWrapper).get("payDate");
	}


	/**
	 * 是否是最新单据
	 *
	 * @param billId       单据id
	 * @param sourceBillId 来源单据id
	 *
	 * @return 是否是最新单据
	 */
	@Override
	public boolean isLatestBill(Long billId, Long sourceBillId) {
		QueryWrapper<PayEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("source_bill_id", sourceBillId)
				.orderByDesc("pay_date");
		List<PayEntity> list = super.list(queryWrapper);
		if (CollectionUtils.isNotEmpty(list)) {
			return list.get(0).getId().equals(billId);
		}
		return true;
	}


	/**
	 * 风险金缴纳：批量确认
	 *
	 * @param pays 要确认的单据信息
	 */
	@Override
	@Transactional(rollbackFor = Exception.class)
	public void batchConfirm(List<PayVO> pays) {
		List<PayEntity> payEntities = BeanMapper.mapList(pays, PayEntity.class);
		UserContext userContext = sessionManager.getUserContext();

		// 缴纳明细转Map<Long, PayEntity>
		Map<Long, PayEntity> payMap = payEntities.stream().collect(Collectors.toMap(PayEntity::getSourceBillId, Function.identity()));

		// 员工明细ids
		ArrayList<Long> ids = new ArrayList<>();

		for (Map.Entry<Long, PayEntity> entry : payMap.entrySet()) {
			PayEntity pay = entry.getValue();

			pay.setConfirmStatus(ConfirmStatusEnum.CONFIRM.getCode());
			pay.setConfirmStatusName(ConfirmStatusEnum.CONFIRM.getName());
			pay.setConfirmDate(new Date());
			pay.setConfirmUserId(userContext.getUserId());
			pay.setConfirmUserCode(userContext.getUserCode());
			pay.setConfirmUserName(userContext.getUserName());

			// 员工明细ids
			ids.add(pay.getSourceBillId());
		}

		// 批量查询员工明细信息
		Collection<PayerDetailEntity> payerDetails = payerDetailService.listByIds(ids);

		// 更新员工明细表：是否分期、分期金额、分期数、每期缴纳金额、实际已缴纳金额、实际已缴纳比例、剩余缴纳金额
		for (PayerDetailEntity pd : payerDetails) {
			PayEntity pay = payMap.get(pd.getId());
			pd.setPeriodizationFlag(Integer.parseInt(pay.getInstallmentFlag()));
			if (Integer.parseInt(pay.getInstallmentFlag()) == 1) {
				pd.setPeriodizationMny(pay.getInstallmentTaxMny().setScale(2, RoundingMode.HALF_UP));
				pd.setPeriodizationNum(pay.getInstallmentNum());
				pd.setEachPeriodizationMny(pay.getInstallmentTaxMny().divide(new BigDecimal(pay.getInstallmentNum()), 2, RoundingMode.HALF_UP));
			}
			// 实际缴纳金额=员工缴纳+员工退还+员工调动+员工调入
			BigDecimal actually = pay.getPayTaxMny().add(pd.getActualPayerMny());
			pd.setActualPayerMny(actually.setScale(2, RoundingMode.HALF_UP));
			pd.setActualPayerScale(actually.divide(pd.getPayerMny(), 2, RoundingMode.HALF_UP));
			pd.setRemainderPayerMny(pd.getPayerMny().subtract(actually).setScale(2, RoundingMode.HALF_UP));
		}

		// 执行批量更新员工明细表
		payerDetailService.saveOrUpdateBatch(payerDetails, 10);

		// 执行批量更新
		super.saveOrUpdateBatch(payEntities, 10);
	}

	@Override
	public  CommonResponse<String> back(PayVO payVO) {
		PayEntity entity = selectById(payVO.getId());
		QueryWrapper<PayEntity> payWrapper = new QueryWrapper<>();
		payWrapper.select("ifnull(count(*),0) as count")
				.eq("source_bill_id", entity.getSourceBillId())
				.ne("confirm_status", ConfirmStatusEnum.CONFIRM.getCode());
		Long pay = (Long) getMap(payWrapper).get("count");
		if (pay != 0) {
			return CommonResponse.error("该员工有待确认/确认中的单据，不支持撤回");
		}

		// 查询未生效的退还+调动
		QueryWrapper<RepayEntity> repayWrapper = new QueryWrapper<>();
		repayWrapper.select("ifnull(count(*),0) as count")
				.eq("source_bill_id", entity.getSourceBillId())
				.ne("confirm_status", ConfirmStatusEnum.CONFIRM.getCode());
		Long repay = (Long) repayService.getMap(repayWrapper).get("count");
		if (repay != 0) {
			return CommonResponse.error("该员工有待确认/确认中单据，不支持撤回");
		}

		entity.setConfirmStatus(ConfirmStatusEnum.CONFIRMING.getCode());
		entity.setConfirmStatusName(ConfirmStatusEnum.CONFIRMING.getName());
		entity.setConfirmDate(null);
		entity.setConfirmUserId(null);
		entity.setConfirmUserCode(null);
		entity.setConfirmUserName(null);

        // 批量查询员工明细信息
        PayerDetailEntity pd = payerDetailService.selectById(entity.getSourceBillId());

        // 更新员工明细表：是否分期、分期金额、分期数、每期缴纳金额、实际已缴纳金额、实际已缴纳比例、剩余缴纳金额
		pd.setPeriodizationFlag(Integer.parseInt(entity.getInstallmentFlag()));
		if (Integer.parseInt(entity.getInstallmentFlag()) == 1) {
			pd.setPeriodizationMny(null);
			pd.setPeriodizationNum(null);
			pd.setEachPeriodizationMny(null);
		}
		// 实际缴纳金额=员工缴纳+员工退还+员工调动+员工调入
		pd.setActualPayerMny(ComputeUtil.safeSub(pd.getActualPayerMny(), entity.getPayTaxMny()));
		pd.setActualPayerScale(ComputeUtil.safeDiv(pd.getActualPayerMny(), pd.getPayerMny()));
		pd.setRemainderPayerMny(ComputeUtil.safeSub(pd.getPayerMny(), pd.getActualPayerMny()));

        // 执行更新员工明细表
		payerDetailService.saveOrUpdate(pd);

        // 执行更新
        saveOrUpdate(entity);
        return CommonResponse.success("撤回成功");
	}
}
