package com.ejianc.business.capital.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.capital.bean.CapitalRealPaymentApplyEntity;
import com.ejianc.business.capital.service.ICapitalRealPaymentApplyService;
import com.ejianc.business.capital.vo.CapitalRealPaymentApplyVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * 资本金实缴申请
 *
 * @author CJ
 * @Description: 资本金实缴申请
 * @date 2021/10/15 17:28
 */
@RestController
@RequestMapping(value = "/capitalRealPayApply/")
public class CapitalRealPayApplyController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICapitalRealPaymentApplyService capitalRealPaymentApplyService;

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final static String BILL_CODE_RULL_CODE = "SDDJS_CAPITAL_APPLY01";

    /**
     * 分页查询资本金实缴申请
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<CapitalRealPaymentApplyVO>> pageList(@RequestBody QueryParam queryParam) {

        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }
        List<Long> orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        queryParam.getParams().put("applyOrgId", new Parameter(QueryParam.IN, orgIds));

        IPage<CapitalRealPaymentApplyEntity> page = capitalRealPaymentApplyService.queryPage(queryParam, false);

        IPage<CapitalRealPaymentApplyVO> result = new Page<>();
        result.setTotal(page.getTotal());
        result.setRecords(BeanMapper.mapList(page.getRecords(), CapitalRealPaymentApplyVO.class));
        result.setCurrent(page.getCurrent());
        result.setSize(page.getSize());
        result.setPages(page.getPages());


        return CommonResponse.success("分页查询成功！", result);
    }

    /**
     * 资本金实缴申请保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<CapitalRealPaymentApplyVO> saveOrUpdate(@RequestBody CapitalRealPaymentApplyVO vo) {
        CapitalRealPaymentApplyEntity saveEntity = null;
        UserContext userContext = sessionManager.getUserContext();

        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, CapitalRealPaymentApplyEntity.class);
            //生成编码
            CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULL_CODE, InvocationInfoProxy.getTenantid());
            if(!billCodeResp.isSuccess()) {
                return CommonResponse.error("保存失败，生成单据编码失败！");
            }
            saveEntity.setBillCode(billCodeResp.getData());

            //默认为启用状态
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
            saveEntity.setCreateUserName(userContext.getUserName());

        } else {
            saveEntity = capitalRealPaymentApplyService.selectById(vo.getId());
            saveEntity.setApplyAmt(vo.getApplyAmt());
            saveEntity.setCaptailId(vo.getCaptailId());
            saveEntity.setCaptailName(vo.getCaptailName());
            saveEntity.setApplyReason(vo.getApplyReason());
            saveEntity.setTotalPaidCapital(vo.getTotalPaidCapital());

            saveEntity.setModifyUserName(userContext.getUserName());
        }

        capitalRealPaymentApplyService.saveOrUpdate(saveEntity, false);

        return CommonResponse.success("保存成功！", BeanMapper.map(saveEntity, CapitalRealPaymentApplyVO.class));
    }

    /**
     * 查询指定的资本金实缴申请详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<CapitalRealPaymentApplyVO> queryDetail(@RequestParam(value = "id") Long id) {
        CapitalRealPaymentApplyVO resp = null;

        CapitalRealPaymentApplyEntity dbEntity = capitalRealPaymentApplyService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, CapitalRealPaymentApplyVO.class);
        }

        return CommonResponse.success("查询详情成功！", resp);
    }

    /**
     * 资本金实缴申请删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<CapitalRealPaymentApplyVO> vos) {
        capitalRealPaymentApplyService.removeByIds(vos.stream().map(CapitalRealPaymentApplyVO::getId).collect(Collectors.toList()));

        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        param.getFuzzyFields().add("billCode");
        param.getFuzzyFields().add("captailName");
        param.getFuzzyFields().add("organizationName");
        param.getParams().put("tenant_id",new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        param.setPageIndex(1);
        param.setPageSize(-1);
        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("导出失败, 查询组织信息失败！");
        }
        List<Long> orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        param.getParams().put("applyOrgId", new Parameter(QueryParam.IN, orgIds));

        List<CapitalRealPaymentApplyEntity> list = capitalRealPaymentApplyService.queryList(param);

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", list);
        ExcelExport.getInstance().export("Org-export.xlsx", beans, response);
    }
}
