package com.ejianc.business.capital.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.capital.bean.CapitalRecRegistryEntity;
import com.ejianc.business.capital.bean.CapitalRecRegistrySubEntity;
import com.ejianc.business.capital.service.ICapitalRecRegistryService;
import com.ejianc.business.capital.vo.CapitalRecRegistrySubVO;
import com.ejianc.business.capital.vo.CapitalRecRegistryVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2021/10/19 10:42
 */
@RestController
@RequestMapping(value = "/capitalRecRegistry/")
public class CapitalRecRegistryController {

    @Autowired
    private ICapitalRecRegistryService capitalRecRegistryService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IBillCodeApi billCodeApi;

    private final static String BILL_CODE_RULL_CODE = "SDDJS_CAPITAL_REC_REGISTRY01";

    @Autowired
    private SessionManager sessionManager;

    /**
     * 分页查询资本金收款登记
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<CapitalRecRegistryVO>> pageList(@RequestBody QueryParam queryParam) {
        queryParam.getFuzzyFields().add("billCode");
        queryParam.getFuzzyFields().add("captailNames");
        queryParam.getFuzzyFields().add("sddjsOrgName");
        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
        }
        List<Long> orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        queryParam.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));

        IPage<CapitalRecRegistryEntity> page = capitalRecRegistryService.queryPage(queryParam, false);

        IPage<CapitalRecRegistryVO> result = new Page<>();
        result.setTotal(page.getTotal());
        result.setRecords(BeanMapper.mapList(page.getRecords(), CapitalRecRegistryVO.class));
        result.setCurrent(page.getCurrent());
        result.setSize(page.getSize());
        result.setPages(page.getPages());

        return CommonResponse.success("分页查询成功！", result);
    }

    /**
     * 资本金收款登记保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<CapitalRecRegistryVO> saveOrUpdate(@RequestBody CapitalRecRegistryVO vo) {
        CapitalRecRegistryEntity saveEntity = null;
        UserContext userContext = sessionManager.getUserContext();

        if(CollectionUtils.isEmpty(vo.getSubList())) {
            return CommonResponse.error("保存失败，实缴收款信息不能为空！");
        }

        List<Long> capitalIdList = new ArrayList<>();
        BigDecimal curTotalRecAmt = BigDecimal.ZERO;
        StringBuilder captailNames = new StringBuilder();
        for(CapitalRecRegistrySubVO sub : vo.getSubList()) {
            if(!"del".equals(sub.getRowState())) {
                if(null != sub.getCurPayAmt()) {
                    if(capitalIdList.contains(sub.getCaptailId())) {
                        return CommonResponse.error("保存失败，出资人重复!");
                    }
                    captailNames.append(sub.getCaptailName()).append(",");
                    curTotalRecAmt = curTotalRecAmt.add(sub.getCurPayAmt());
                    if(CollectionUtils.isNotEmpty(sub.getAttachIds())) {
                        sub.setAttachFileIds(StringUtils.join(sub.getAttachIds(), ","));
                    }
                } else {
                    //本次收款金额为空的行删除
                    sub.setRowState("del");
                }
            }
        }
        vo.setCurTotalRecAmt(curTotalRecAmt);
        vo.setCaptailNames(captailNames.substring(0, captailNames.length()-1));

        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, CapitalRecRegistryEntity.class);
            //生成编码
            CommonResponse<String> billCodeResp = billCodeApi.getCodeBatchByRuleCode(BILL_CODE_RULL_CODE, InvocationInfoProxy.getTenantid());
            if(!billCodeResp.isSuccess()) {
                return CommonResponse.error("保存失败，生成单据编码失败！");
            }
            saveEntity.setBillCode(billCodeResp.getData());

            //默认为启用状态
            saveEntity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
            saveEntity.setCreateUserName(userContext.getUserName());

        } else {
            saveEntity = capitalRecRegistryService.selectById(vo.getId());
            saveEntity.setSubList(BeanMapper.mapList(vo.getSubList(), CapitalRecRegistrySubEntity.class));
            saveEntity.setSddjsOrgId(vo.getSddjsOrgId());
            saveEntity.setSddjsOrgName(vo.getSddjsOrgName());
            saveEntity.setCurrencyTypeId(vo.getCurrencyTypeId());
            saveEntity.setCurrencyTypeName(vo.getCurrencyTypeName());
            saveEntity.setRemark(vo.getRemark());
            saveEntity.setRegisterCapital(vo.getRegisterCapital());
            saveEntity.setCurTotalRecAmt(vo.getCurTotalRecAmt());
            saveEntity.setCaptailNames(vo.getCaptailNames());

            saveEntity.setModifyUserName(userContext.getUserName());
            saveEntity.setUpdateUserCode(userContext.getUserCode());
        }

        capitalRecRegistryService.saveOrUpdate(saveEntity, false);

        return CommonResponse.success("保存成功！", BeanMapper.map(saveEntity, CapitalRecRegistryVO.class));
    }

    /**
     * 查询指定的资本金收款登记详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<CapitalRecRegistryVO> queryDetail(@RequestParam(value = "id") Long id) {
        CapitalRecRegistryVO resp = null;

        CapitalRecRegistryEntity dbEntity = capitalRecRegistryService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, CapitalRecRegistryVO.class);
            if(CollectionUtils.isNotEmpty(resp.getSubList())) {
                resp.getSubList().forEach(sub -> {
                    if(StringUtils.isNotBlank(sub.getAttachFileIds())) {
                        sub.setAttachIds(Arrays.asList(sub.getAttachFileIds().split(",")).stream().map(item -> Long.valueOf(item)).collect(Collectors.toList()));
                    } else {
                        sub.setAttachFileIds(null);
                    }
                });
            }
        }

        return CommonResponse.success("查询详情成功！", resp);
    }

    /**
     * 资本金收款登记删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<CapitalRecRegistryVO> vos) {
        for(CapitalRecRegistryVO vo : vos) {
            capitalRecRegistryService.removeById(vo.getId(), false);
        }
        return CommonResponse.success("删除成功！");
    }

    /**
     * @Description 导出
     * @param param
     * @Return void
     */
    @RequestMapping(value = "/excelExport", method = RequestMethod.POST)
    @ResponseBody
    public void excelExport(@RequestBody QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        param.getFuzzyFields().add("billCode");
        param.getFuzzyFields().add("captailNames");
        param.getFuzzyFields().add("sddjsOrgName");
        param.setPageIndex(1);
        param.setPageSize(-1);
        //查询本下
        CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId());
        if(!childOrgResp.isSuccess()) {
            throw new BusinessException("导出失败, 查询组织信息失败！");
        }
        List<Long> orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        param.getParams().put("orgId", new Parameter(QueryParam.IN, orgIds));

        List<CapitalRecRegistryEntity> list = capitalRecRegistryService.queryList(param);
        List<CapitalRecRegistryVO> vos = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(list)) {
            CapitalRecRegistryVO tmp = null;
            for(CapitalRecRegistryEntity e : list) {
                tmp = BeanMapper.map(e, CapitalRecRegistryVO.class);
                tmp.setCreateTimeStr(DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", e.getCreateTime()));
                tmp.setBillStateName(BillStateEnum.getEnumByStateCode(e.getBillState()).getDescription());
                vos.add(tmp);
            }
        }

        Map<String, Object> beans = new HashMap<>();
        beans.put("records", vos);
        ExcelExport.getInstance().export("capitalRecRegistryExport.xlsx", beans, response);
    }

}
