package com.ejianc.business.capital.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.capital.bean.CapitalRealPaymentApplyEntity;
import com.ejianc.business.capital.bean.CapitalRecRegistryEntity;
import com.ejianc.business.capital.bean.CapitalRecRegistrySubEntity;
import com.ejianc.business.capital.service.ICapitalRecRegistryService;
import com.ejianc.business.capital.util.MoneyUtils;
import com.ejianc.business.sddjsorg.bean.OrgShareholderEntity;
import com.ejianc.business.sddjsorg.bean.SddjsOrgEntity;
import com.ejianc.business.sddjsorg.service.ISddjsOrgService;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2021/10/19 19:31
 */
@Service(value = "capitalRecRegistry")
public class CapitalRecRegistryBpmServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private SessionManager sessionManager;

    @Autowired
    private ICapitalRecRegistryService capitalRecRegistryService;

    @Autowired
    private ISddjsOrgService sddjsOrgService;

    /**
     * 提交审批
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        CapitalRecRegistryEntity e = capitalRecRegistryService.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();

        e.setCommitDate(new Date());
        e.setCommitUserCode(userContext.getUserCode());
        e.setCommitUserName(userContext.getUserName());
        capitalRecRegistryService.saveOrUpdate(e, false);

        return CommonResponse.success("资本金收款登记审批回调处理成功！");
    }

    /**
     * 直审/审批通过
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {
        CapitalRecRegistryEntity e = capitalRecRegistryService.selectById(billId);

        if(state == 1) {
            UserContext userContext = sessionManager.getUserContext();

            e.setCommitDate(new Date());
            e.setCommitUserCode(userContext.getUserCode());
            e.setCommitUserName(userContext.getUserName());
            capitalRecRegistryService.saveOrUpdate(e, false);
        }

        if(CollectionUtils.isNotEmpty(e.getSubList())) {
            Map<Long, CapitalRecRegistrySubEntity> subMap = e.getSubList().stream().collect(Collectors.toMap(item -> item.getId(), item -> item));

            //累加回写【组织机构查询】组织机构信息【股东信息】对应行上的“实缴金额”、“实缴金额(大写)”
            SddjsOrgEntity sddjsOrg = sddjsOrgService.selectById(e.getSddjsOrgId());
            if(CollectionUtils.isNotEmpty(sddjsOrg.getOrgShareholderList())) {
                for(OrgShareholderEntity item : sddjsOrg.getOrgShareholderList()) {
                    if(null != subMap.get(item.getId())) {
                        //实缴金额
                        item.setPaidCapital(item.getPaidCapital().add(subMap.get(item.getId()).getPaidCapital()));
                        //实缴金额大写
                        item.setPaidCapitalBig(MoneyUtils.getMnyUpCase(item.getPaidCapital()));
                    }
                }

                sddjsOrgService.saveOrUpdate(sddjsOrg, false);
            } else {
                return CommonResponse.error("资本金收款登记审批回调处理失败，组织机构["+e.getSddjsOrgId()+"]的股东信息为空");
            }
        }


        return CommonResponse.success("资本金收款登记审批回调处理成功！");
    }

    /**
     * 撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审/撤回前回调校验
     *  校验是否已经存在该公司更新的【资本金收款登记】，
     *  如果是则不允许弃审，并提示：操作失败！该公司已经存在新的【资本金收款登记】，不允许取消审批。
     *
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        CapitalRecRegistryEntity e = capitalRecRegistryService.selectById(billId);

        logger.info("资本金收款登记单据[billId-{}],[billTypeCode-{}]执行弃审前校验！", billId, billTypeCode);
        //校验是否已经存在该公司更新的【资本金收款登记】单据
        QueryWrapper<CapitalRecRegistryEntity> query = new QueryWrapper<>();
        query.eq("sddjs_org_id", e.getSddjsOrgId());
        query.eq("dr", BaseVO.DR_UNDELETE);
        query.gt("create_time", e.getCreateTime());

        List<CapitalRecRegistryEntity> list = capitalRecRegistryService.list(query);
        if(CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.error("操作失败！该公司已经存在新的[资本金收款登记],不允许取消审批。");
        }

        return CommonResponse.success("资本金收款登记审批回调处理成功!");
    }

    /**
     * 撤回回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 弃审回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 弃审、撤回回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public CommonResponse<String> doBpmBack(Long billId, Integer state, String billTypeCode) {
        CapitalRecRegistryEntity e = capitalRecRegistryService.selectById(billId);
        logger.info("资本金收款登记单据-【{}】执行撤回逻辑", JSONObject.toJSONString(e));

        //清空生效时间/提交人相关信息
        e.setEffectiveDate(null);
        e.setCommitDate(null);
        e.setCommitUserCode(null);
        e.setCommitUserName(null);
        capitalRecRegistryService.saveOrUpdate(e, false);

        //累减回写【组织机构查询】组织机构信息【股东信息】对应行上的“实缴金额”、“实缴金额(大写)”
        if(CollectionUtils.isNotEmpty(e.getSubList())) {
            Map<Long, CapitalRecRegistrySubEntity> subMap = e.getSubList().stream().collect(Collectors.toMap(item -> item.getId(), item -> item));
            SddjsOrgEntity sddjsOrg = sddjsOrgService.selectById(e.getSddjsOrgId());
            if(CollectionUtils.isNotEmpty(sddjsOrg.getOrgShareholderList())) {
                for(OrgShareholderEntity item : sddjsOrg.getOrgShareholderList()) {
                    if(null != subMap.get(item.getId())) {
                        //实缴金额
                        item.setPaidCapital(item.getPaidCapital().subtract(subMap.get(item.getId()).getPaidCapital()));
                        //实缴金额大写
                        item.setPaidCapitalBig(MoneyUtils.getMnyUpCase(item.getPaidCapital()));
                    }
                }

                sddjsOrgService.saveOrUpdate(sddjsOrg, false);
            } else {
                return CommonResponse.error("资本金收款登记审批回调处理失败，组织机构["+e.getSddjsOrgId()+"]的股东信息为空");
            }
        }


        return CommonResponse.success("执行资本金收款登记单据撤回逻辑成功! ");
    }

}
