package com.ejianc.business.capital.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.capital.bean.CapitalRealPaymentApplyEntity;
import com.ejianc.business.capital.service.ICapitalRealPaymentApplyService;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.billState.service.ICommonBusinessService;
import com.ejianc.framework.skeleton.template.BaseVO;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

/**
 * 资本金实缴申请审批回调服务
 *
 * @author CJ
 * @Description:
 * @date 2021/10/15 17:55
 */
@Service(value = "capitalRealPaymentApply")
public class CapitalRealPayApplyBpmServiceImpl implements ICommonBusinessService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICapitalRealPaymentApplyService capitalRealPaymentApplyService;

    @Autowired
    private SessionManager sessionManager;

    /**
     * 提交审批
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterSubmitProcessor(Long billId, Integer state, String billTypeCode) {
        CapitalRealPaymentApplyEntity e = capitalRealPaymentApplyService.selectById(billId);
        UserContext userContext = sessionManager.getUserContext();

        e.setCommitDate(new Date());
        e.setCommitUserCode(userContext.getUserCode());
        e.setCommitUserName(userContext.getUserName());
        capitalRealPaymentApplyService.saveOrUpdate(e, false);

        return CommonResponse.success("资本金实缴申请审批回调处理成功！");
    }

    /**
     * 直审/审批通过
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterApprovalProcessor(Long billId, Integer state, String billTypeCode) {

        if(state == 1) {
            CapitalRealPaymentApplyEntity e = capitalRealPaymentApplyService.selectById(billId);
            UserContext userContext = sessionManager.getUserContext();

            e.setCommitDate(new Date());
            e.setCommitUserCode(userContext.getUserCode());
            e.setCommitUserName(userContext.getUserName());
            capitalRealPaymentApplyService.saveOrUpdate(e, false);
        }

        return CommonResponse.success("资本金实缴申请审批回调处理成功！");
    }

    /**
     * 撤回
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> beforeAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return bpmBackCheck(billId, state, billTypeCode);
    }

    /**
     * 弃审/撤回前回调校验
     *  查询按照“公司名称+出资人名称”维度，
     *  是否存在更新的【资本金实缴申请】，如果是则不允许弃审
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    private CommonResponse<String> bpmBackCheck(Long billId, Integer state, String billTypeCode) {
        CapitalRealPaymentApplyEntity e = capitalRealPaymentApplyService.selectById(billId);

        logger.info("资本金收款登记单据[billId-{}],[billTypeCode-{}]执行弃审前校验！", billId, billTypeCode);
        //校验是否已经存在该公司+出资人更新的【资本金实缴申请】单据
        QueryWrapper<CapitalRealPaymentApplyEntity> query = new QueryWrapper<>();
        query.eq("sddjs_org_id", e.getSddjsOrgId());
        query.eq("captail_id", e.getCaptailId());
        query.eq("dr", BaseVO.DR_UNDELETE);
        query.gt("create_time", DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", e.getCreateTime()));

        List<CapitalRealPaymentApplyEntity> list = capitalRealPaymentApplyService.list(query);
        if(CollectionUtils.isNotEmpty(list)) {
            return CommonResponse.success("公司[" + e.getSddjsOrgName() + "]的出资方["+e.getCaptailName()+"]已经存在新的[资本金实缴]单据，不允许取消审批。");
        }

        return CommonResponse.success("资本金收款登记审批回调处理成功!");
    }

    /**
     * 撤回回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterHasBpmBack(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 弃审回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    @Override
    public CommonResponse<String> afterAbstainingProcessor(Long billId, Integer state, String billTypeCode) {
        return doBpmBack(billId, state, billTypeCode);
    }

    /**
     * 弃审、撤回回调处理
     *
     * @param billId
     * @param state
     * @param billTypeCode
     * @return
     */
    private CommonResponse<String> doBpmBack(Long billId, Integer state, String billTypeCode) {
        CapitalRealPaymentApplyEntity e = capitalRealPaymentApplyService.selectById(billId);
        logger.info("资本金实缴申请单据-【{}】执行撤回逻辑", JSONObject.toJSONString(e));

        //清空生效时间/提交人相关信息
        e.setEffectiveDate(null);
        e.setCommitDate(null);
        e.setCommitUserCode(null);
        e.setCommitUserName(null);
        capitalRealPaymentApplyService.saveOrUpdate(e, false);

        return CommonResponse.success("执行资本金实缴申请单据撤回逻辑成功! ");
    }

}
