package com.ejianc.business.orgcenter.controller;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.orgcenter.bean.DirectorEntity;
import com.ejianc.business.orgcenter.service.IDirectorService;
import com.ejianc.business.orgcenter.vo.DirectorVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2021/9/28 16:25
 */
@RestController
@RequestMapping(value = "/director/")
public class DirectorController {

    @Autowired
    private IDirectorService directorService;

    /**
     * 分页查询董高监人员列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "pageList")
    public CommonResponse<IPage<DirectorVO>> pageList(@RequestBody QueryParam queryParam) {
        queryParam.getOrderMap().put("createTime", "desc");
        IPage<DirectorEntity> page = directorService.queryPage(queryParam, false);

        IPage<DirectorVO> result = new Page<>();
        result.setTotal(page.getTotal());
        result.setRecords(BeanMapper.mapList(page.getRecords(), DirectorVO.class));
        result.setCurrent(page.getCurrent());
        result.setSize(page.getSize());
        result.setPages(page.getPages());


        return CommonResponse.success("分页查询成功！", result);
    }

    /**
     * 董监高人员保存/更新
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<DirectorVO> saveOrUpdate(@RequestBody DirectorVO vo) {
        DirectorEntity saveEntity = null;

        //编号重复校验
        QueryWrapper<DirectorEntity> query = new QueryWrapper();
        query.eq("bill_code", vo.getBillCode());
        if(null != vo.getId()) {
            query.ne("id", vo.getId());
        }
        List<DirectorEntity> billCodeEntityList = directorService.list(query);
        if(CollectionUtils.isNotEmpty(billCodeEntityList)) {
            return CommonResponse.error("保存失败，人员编号重复！");
        }

        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, DirectorEntity.class);
            //默认为启用状态
            saveEntity.setStatus(DirectorVO.DIRECTOR_STATUS_VALID);
        } else {
            saveEntity = directorService.selectById(vo.getId());

            saveEntity.setDirectorName(vo.getDirectorName());
            saveEntity.setBillCode(vo.getBillCode());
            saveEntity.setOrgName(vo.getOrgName());
            saveEntity.setPostName(vo.getPostName());
        }

        directorService.saveOrUpdate(saveEntity, false);

        return CommonResponse.success("保存成功！", BeanMapper.map(saveEntity, DirectorVO.class));
    }

    @PostMapping(value = "saveEmployeeRefers")
    public CommonResponse<String> saveEmployeeRefers(@RequestBody List<DirectorVO> vos) {
        List<Long> idList = new ArrayList<>();
        List<String> billCodeList = new ArrayList<>();
        Map<Long, DirectorVO> voMap = new HashMap<>();
        vos.stream().forEach(vo -> {
            idList.add(vo.getId());
            voMap.put(vo.getId(), vo);
            billCodeList.add(vo.getBillCode());
        });

        //编号重复校验
        QueryWrapper<DirectorEntity> repeatQuery = new QueryWrapper();
        repeatQuery.in("bill_code", billCodeList);
        List<DirectorEntity> billCodeEntityList = directorService.list(repeatQuery);
        if(CollectionUtils.isNotEmpty(billCodeEntityList)) {
            for(DirectorEntity dire : billCodeEntityList) {
                if(null == voMap.get(dire.getId()) || !(voMap.get(dire.getId())).equals(dire.getBillCode())) {
                    return CommonResponse.error("保存失败，已存在编号为" + dire.getBillCode() + "的人员！");
                }
            }
        }

        QueryWrapper<DirectorEntity> query = new QueryWrapper<>();
        query.in("id", idList);
        List<DirectorEntity> saveList = directorService.list(query);

        //如果其中有人员已经存在，则更新信息
        DirectorVO tmp= null;
        if(CollectionUtils.isNotEmpty(saveList)) {
            for(DirectorEntity directorEntity : saveList) {
                tmp = voMap.get(directorEntity.getId());
                directorEntity.setPostName(tmp.getPostName());
                directorEntity.setOrgName(tmp.getOrgName());
                directorEntity.setBillCode(tmp.getBillCode());
                directorEntity.setDirectorName(tmp.getDirectorName());
                vos.remove(tmp);
            }
        }

        if(CollectionUtils.isNotEmpty(vos)) {
            saveList.addAll(BeanMapper.mapList(vos, DirectorEntity.class));
        }

        directorService.saveOrUpdateBatch(saveList, vos.size());
        return CommonResponse.success("保存成功！");
    }

    /**
     * 查询指定的董监高人员详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<DirectorVO> queryDetail(@RequestParam(value = "id") Long id) {
        DirectorVO resp = null;

        DirectorEntity dbEntity = directorService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, DirectorVO.class);
        }

        return CommonResponse.success("查询详情成功！", resp);
    }

    /**
     * 董监高人员删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<DirectorVO> vos) {
        directorService.batchDelByIds(vos.stream().map(DirectorVO::getId).collect(Collectors.toList()));

        return CommonResponse.success("删除成功！");
    }

    /**
     * 人员起停用状态修改
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<String> changeStatus(@RequestBody DirectorVO vo) {
        DirectorEntity dbEntity = directorService.selectById(vo.getId());
        if(null != dbEntity) {
            dbEntity.setStatus(vo.getStatus());
            directorService.saveOrUpdate(dbEntity, false);
        }

        return CommonResponse.success("操作成功！");
    }


    /**
     * 内部人员档案信息同步
     *
     * @return
     */
    @PostMapping(value = "syncEmployeeDetails")
    public CommonResponse<String> syncEmployeeDetails() {
        directorService.syncEmployeeDetails();
        return CommonResponse.success("同步成功！");
    }


    @RequestMapping(value = "queryRefList", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<IPage<DirectorVO>> queryRefList(
    		@RequestParam(defaultValue = "1") int pageNumber, @RequestParam(defaultValue = "10") int pageSize,
    		@RequestParam(required=false) String searchText, @RequestParam(required=false) String condition
    		, @RequestParam(required=false) String searchObject) {
        QueryParam param = new QueryParam();

    	if(StringUtils.isNotBlank(searchObject)){
    		JSONObject search = JSONObject.parseObject(searchObject);
    		for(String key : search.keySet()){
    			param.getParams().put(key,new Parameter(QueryParam.LIKE,search.get(key).toString()));
    		}
    	}
    	param.setPageSize(pageSize);
    	param.setPageIndex(pageNumber);
        if(StringUtils.isNotBlank(searchText)){
        	param.setSearchText(searchText);
            /** 模糊搜索配置字段示例 */
            List<String> fuzzyFields = param.getFuzzyFields();
            fuzzyFields.add("directorName");
            fuzzyFields.add("orgName");
            fuzzyFields.add("postName");
            fuzzyFields.add("billCode");
        }
        
        IPage<DirectorEntity> page = directorService.queryPage(param, false);
        IPage<DirectorVO> result = new Page<>();
        result.setTotal(page.getTotal());
        result.setRecords(BeanMapper.mapList(page.getRecords(), DirectorVO.class));
        result.setCurrent(page.getCurrent());
        result.setSize(page.getSize());
        result.setPages(page.getPages());
        return CommonResponse.success("查询仓库数据成功！", result);
    }

}
