package com.ejianc.business.orgcenter.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.orgcenter.bean.AdministrativeAreaEntity;
import com.ejianc.business.orgcenter.service.IAdministrativeAreaService;
import com.ejianc.business.orgcenter.vo.AdministrativeAreaVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.framework.core.util.ResultAsTree;
import com.ejianc.framework.skeleton.template.BaseVO;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 行政地区控制器
 *
 * @author CJ
 * @Description:
 * @date 2021/10/12 10:46
 */
@RestController
@RequestMapping("/administrativeArea/")
public class AdministrativeAreaController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());
    @Autowired
    private IAdministrativeAreaService administrativeAreaService;

    @Autowired
    private CacheManager cacheManager;

    // 所有地区缓存
    private final static String SDDJS_AREA_CACHE_KEY = "SDDJS_area_cache_key";
    /**
     * 行政地区懒加载
     *
     * @param pid
     * @return
     */
    @GetMapping(value = "queryList")
    public CommonResponse<List<AdministrativeAreaVO>> queryList(@RequestParam(value = "pid", required = false) Long pid,
                                                               @RequestParam(value = "billCode", required = false) String billCode,
                                                               @RequestParam(value = "areaName", required = false) String areaName) {
        List<AdministrativeAreaVO> listMap = administrativeAreaService.queryList(pid, billCode, areaName);
        if(CollectionUtils.isNotEmpty(listMap)) {
            listMap.stream().forEach(item -> {if(item.isLeaf()) {item.setChildren(null);}});
        }
        return CommonResponse.success("查询成功！", listMap);
    }

    /**
     * 行政地区信息保存
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "saveOrUpdate")
    public CommonResponse<AdministrativeAreaVO> saveOrUpdate(@RequestBody AdministrativeAreaVO vo) {
        //编号重复校验
        QueryWrapper<AdministrativeAreaEntity> query = new QueryWrapper();
        query.eq("bill_code", vo.getBillCode());
        if(null != vo.getId()) {
            query.ne("id", vo.getId());
        }
        List<AdministrativeAreaEntity> billCodeEntityList = administrativeAreaService.list(query);
        if(CollectionUtils.isNotEmpty(billCodeEntityList)) {
            return CommonResponse.error("保存失败，编码重复！");
        }

        AdministrativeAreaEntity saveEntity = null;
        if(null == vo.getId()) {
            saveEntity = BeanMapper.map(vo, AdministrativeAreaEntity.class);
            saveEntity.setStatus(AdministrativeAreaVO.STATUS_VALID);
        } else {
            saveEntity = administrativeAreaService.selectById(vo.getId());

            saveEntity.setAreaName(vo.getAreaName());
            saveEntity.setBillCode(vo.getBillCode());
            saveEntity.setPid(vo.getPid());
            saveEntity.setpName(vo.getpName());
            saveEntity.setDescription(vo.getDescription());
        }

        administrativeAreaService.saveOrUpdate(saveEntity);
        refreshCache();
        return CommonResponse.success("保存成功！");
    }

    /**
     * 查询行政地区详情
     *
     * @param id
     * @return
     */
    @GetMapping(value = "queryDetail")
    public CommonResponse<AdministrativeAreaVO> queryDetail(@RequestParam(value = "id") Long id) {
        AdministrativeAreaVO resp = null;

        AdministrativeAreaEntity dbEntity = administrativeAreaService.selectById(id);
        if(null != dbEntity) {
            resp = BeanMapper.map(dbEntity, AdministrativeAreaVO.class);
        }
        return CommonResponse.success("查询成功！", resp);
    }

    /**
     * 行政地区删除
     *
     * @param vos
     * @return
     */
    @PostMapping(value = "delete")
    public CommonResponse<String> delete(@RequestBody List<AdministrativeAreaVO> vos) {
        List<Long> ids = vos.stream().map(AdministrativeAreaVO::getId).collect(Collectors.toList());
        //查询是否删除节点是否有子节点，如有则不允许删除
        QueryWrapper<AdministrativeAreaEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("dr", BaseVO.DR_UNDELETE);
        queryWrapper.in("pid", ids);
        List<AdministrativeAreaEntity> childList = administrativeAreaService.list(queryWrapper);
        if(CollectionUtils.isNotEmpty(childList)) {
            return CommonResponse.error("删除数据存在子级节点，不允许删除！");
        }
        administrativeAreaService.batchDelByIds(ids);
        refreshCache();
        return CommonResponse.success("删除成功！");
    }

    /**
     * 行政地区起停用状态改变
     *
     * @param vo
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<String> changeStatus(@RequestBody AdministrativeAreaVO vo) {
        AdministrativeAreaEntity dbEntity = administrativeAreaService.selectById(vo.getId());
        if(null != dbEntity) {
            dbEntity.setStatus(vo.getStatus());
            administrativeAreaService.saveOrUpdate(dbEntity, false);
        }
        refreshCache();
        return CommonResponse.success("操作成功！");
    }

    /**
     * 按照下了级联数据格式查询所有的行政地区
     *
     * @param pid
     * @return
     */
    @GetMapping(value = "loadAreaCascader")
    public List<Map<String, Object>> loadAreaCascader(@RequestParam(value = "pid", required = false) Long pid) {
        List<Map> listMap = administrativeAreaService.queryAll(pid);
        List<Map<String, Object>> resp = ResultAsTree.createTreeData(listMap);

        return resp;
    }

    @PostMapping(value = "areaExcelImport")
    public CommonResponse<JSONObject> areaExcelImport(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<JSONObject> successList = new ArrayList<>();
        List<JSONObject> errorList = new ArrayList<>();
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if(isFailed) {
            return CommonResponse.error("文件格式不合法");
        }else{
            List<List<String>> result = ExcelReader.readExcel(mf);
            if(result != null && result.size() > 0) {
                if(result.size()>10000){
                    return CommonResponse.error("地区数据超过10000条，请分批上传！");
                }
                Map<String, JSONObject> areaBeanMap = new HashMap<>();
                JSONObject parent = null;
                AdministrativeAreaEntity parentEntity = null;
                for(int i = 0;i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    JSONObject area = new JSONObject();
                    area.put("id", IdWorker.getId());
                    area.put("billCode", datas.get(0));
                    area.put("areaName", datas.get(1));
                    area.put("pName", datas.get(2));
                    area.put("description", datas.get(3));

                    if(StringUtils.isBlank(datas.get(0))){
                        area.put("err", "行政地区编码为空！");
                        errorList.add(area);
                        continue;
                    }
                    if(StringUtils.isBlank(datas.get(1))){
                        area.put("err", "行政地区名称为空！");
                        errorList.add(area);
                        continue;
                    }

                    if(datas.get(0).length() > 50) {
                        area.put("err", "行政地区编码长度限制为50字以内！");
                        errorList.add(area);
                        continue;
                    }
                    if(datas.get(1).length() > 50) {
                        area.put("err", "行政地区名称长度限制为50字以内！");
                        errorList.add(area);
                        continue;
                    }
                    if(StringUtils.isNotBlank(datas.get(3)) && datas.get(3).length() > 200) {
                        area.put("err", "行政地区描述长度限制为200字以内！");
                        errorList.add(area);
                        continue;
                    }

                    QueryParam queryParam = new QueryParam();
                    queryParam.getParams().put("billCode", new Parameter(QueryParam.EQ, area.get("billCode")));
                    List<AdministrativeAreaEntity> list = administrativeAreaService.queryList(queryParam, false);
                    if(CollectionUtils.isNotEmpty(list)) {
                        area.put("err", "行政地区编码重复！");
                        errorList.add(area);
                        continue;
                    }

                    if(null != area.get("pName") && StringUtils.isNotBlank(area.get("pName").toString())) {
                        //检测上级行政地区是否存在
                        parent = areaBeanMap.get(area.get("pName"));
                        if(null != parent) {
                            area.put("pid", parent.get("id"));
                            area.put("pName", parent.get("areaName"));
                            area.put("level", Integer.valueOf(parent.get("level").toString()) + 1);
                        } else {
                            QueryParam parentQueryParam = new QueryParam();
                            parentQueryParam.getParams().put("p_name", new Parameter(QueryParam.EQ, area.get("pName")));
                            List<AdministrativeAreaEntity> parentList = administrativeAreaService.queryList(queryParam, false);
                            if(CollectionUtils.isEmpty(parentList)) {
                                area.put("err", "上级行政地区在系统中不存在！");
                                errorList.add(area);
                                continue;
                            } else {
                                parentEntity = parentList.get(0);
                                area.put("pid", parentEntity.getId());
                                area.put("pName", parentEntity.getAreaName());
                                area.put("level", parentEntity.getLevel() + 1);
                            }
                        }
                    } else {
                        area.put("level", 1);
                    }

                    //设置为启用状态
                    area.put("status", AdministrativeAreaVO.STATUS_VALID);
                    successList.add(area);
                    areaBeanMap.put(area.get("areaName").toString(), area);
                }
            }
        }
        JSONObject json = new JSONObject();
        json.put("successList", successList);
        json.put("errorList", errorList);
        return CommonResponse.success(json);
    }

    @PostMapping(value = "downloadImportTemplate")
    public void downloadImportTemplate(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "administrativeArea-import.xlsx", "行政地区导入模板");
    }

    @PostMapping(value = "saveAreaExcelImport")
    public CommonResponse<String> saveAreaExcelImport(@RequestBody List<JSONObject> importData) {
        if(ListUtil.isEmpty(importData)){
            return CommonResponse.error("导入的数据为空！");
        }
        List<AdministrativeAreaEntity> saveData = BeanMapper.mapList(importData, AdministrativeAreaEntity.class);
        administrativeAreaService.saveOrUpdateBatch(saveData, saveData.size(), false);
        refreshCache();
        return CommonResponse.success("导入成功！");
    }

    @GetMapping("refreshCache")
    public void refreshCache(){
        Thread t = new Thread(() -> {
            logger.info("开始刷新地址缓存.......");
            List<JSONObject> data = queryArea();
            if(ListUtil.isNotEmpty(data)){
                cacheManager.set(SDDJS_AREA_CACHE_KEY,JSONObject.toJSONString(data));
            }else {
                cacheManager.set(SDDJS_AREA_CACHE_KEY,"[]");
            }
            logger.info("刷新地址缓存完成.......");
        });
        t.start();
    }

    @GetMapping(value = "queryAreaCascader")
    public JSONArray queryAreaCascader() {
        JSONArray data = JSONObject.parseArray(cacheManager.get(SDDJS_AREA_CACHE_KEY).toString());
        logger.info(data.toJSONString());
        return data;
    }

    public List<JSONObject> queryArea() {
        QueryParam param = new QueryParam();
        param.getParams().put("status",new Parameter(QueryParam.EQ,1));
        param.getParams().put("pid",new Parameter(QueryParam.EQ,null));
        List<AdministrativeAreaEntity> listMap = administrativeAreaService.queryList(param);
        List<JSONObject> resp = createTreeData(listMap);
        return resp;
    }


    public List<JSONObject> createTreeData(List<AdministrativeAreaEntity> list) {
        List<JSONObject> resp = new ArrayList<>();

        for(AdministrativeAreaEntity item:list) {
            JSONObject object = JSONObject.parseObject(JSONObject.toJSONString(item));
            object.put("value",item.getId()+"");
            object.put("title",item.getAreaName());
            object.put("label",item.getAreaName());
            resp.add(object);
        }
        for (JSONObject parent : resp) {
            setChildren(parent);
        }
        return resp;
    }

    private void setChildren(JSONObject parent){
        List<JSONObject> children = new ArrayList<>();

        QueryParam param = new QueryParam();
        param.getParams().put("status",new Parameter(QueryParam.EQ,1));
        param.getParams().put("pid",new Parameter(QueryParam.EQ,parent.getString("id")));
        List<AdministrativeAreaEntity> list = administrativeAreaService.queryList(param);
        if(ListUtil.isNotEmpty(list)){
            for(AdministrativeAreaEntity item:list) {
                JSONObject child = JSONObject.parseObject(JSONObject.toJSONString(item));
                child.put("value",item.getId()+"");
                child.put("title",item.getAreaName());
                child.put("label",item.getAreaName());
                children.add(child);
            }
        }
        if(ListUtil.isNotEmpty(children)){
            parent.put("children",children);
            for (JSONObject child : children) {
                setChildren(child);
            }
        }
    }

}
