package com.ejianc.business.projectapply.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.projectapply.bean.ExecutiveAgentEntity;
import com.ejianc.business.projectapply.bean.ProjectApplyEntity;
import com.ejianc.business.projectapply.service.IExecutiveAgentService;
import com.ejianc.business.projectapply.service.IProjectApplyService;
import com.ejianc.business.projectapply.vo.ExecutiveAgentVO;
import com.ejianc.business.projectapply.vo.ProjectApplyVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.CamelAndUnderLineConverter;
import com.ejianc.framework.core.util.ExcelExport;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 项目信息
 *
 * @author CJ
 * @Description: 项目申请控制器
 * @date 2021/9/6 10:10
 */
@RestController
@RequestMapping("/projectInfo/")
public class ProjectInfoController {

    @Autowired
    private IProjectApplyService projectApplyService;

    @Autowired
    private IExecutiveAgentService executiveAgentService;
    
    @Autowired
    private IOrgApi orgApi;

    private final String reg = "(?:')|(?:--)|(/\\*(?:.|[\\n\\r])*?\\*/)|(\\b(from|concat|updatexml|select|update|union|and|or|delete|insert|trancate|char|into|substr|ascii|declare|exec|count|master|drop|execute)\\b)";

    private boolean isValidKey(String key) {
        return !Pattern.compile(reg, 2).matcher(key).find();
    }

    /**
     * 分页查询项目申请列表
     *
     * @param queryParam
     * @return
     */
    @PostMapping(value = "queryList")
    public CommonResponse<IPage<ProjectApplyVO>> queryList(@RequestBody QueryParam queryParam) {
    	Parameter range = queryParam.getParams().get("range");
        queryParam.getParams().remove("range");
    	Parameter orgIdParam = queryParam.getParams().get("orgId");
        queryParam.getParams().remove("orgId");
        List<Long> orgIds =new ArrayList<>();
        queryParam.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        Long orgId = (orgIdParam!=null&&orgIdParam.getValue()!=null)?Long.parseLong(orgIdParam.getValue().toString()):InvocationInfoProxy.getOrgId();
        if("selfAndBelow".equals(range.getValue())) {
            //查询本下
            CommonResponse<List<OrgVO>> childOrgResp = orgApi.findChildrenByParentId(orgId);
            if(!childOrgResp.isSuccess()) {
                throw new BusinessException("分页列表查询失败, 查询组织信息失败！");
            }
            orgIds = childOrgResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList());
        } else {
            //查询本级
            orgIds.add(orgId);
        }

        queryParam.getParams().put("orgIds", new Parameter(QueryParam.IN, orgIds));

        Map<String, Parameter> params = queryParam.getParams();
        Map<String, Parameter> newP = new HashedMap();
        Parameter parameter = null;
        if(null != params && !params.isEmpty()) {
            for(String key : params.keySet()) {
                parameter = params.get(key);
                if(isValidKey(key) && isValidKey(parameter.getType())) {
                    if("createTime".equals(key)) {
                        String[] rangeArr = parameter.getValue().toString().split(",");
                        rangeArr[0] += " 00:00:00";
                        rangeArr[1] += " 23:59:59";
                        parameter.setValue(StringUtils.join(rangeArr, ","));
                    }
                    newP.put(key.replaceAll("[A-Z]", "_$0").toLowerCase(), parameter);
                }
            }
        }
        //默认查询审批通过的数据
        newP.put("bill_state", new Parameter(QueryParam.IN, StringUtils.joinWith(",", BillStateEnum.COMMITED_STATE.getBillStateCode(), BillStateEnum.PASSED_STATE.getBillStateCode())));

        long count = projectApplyService.dynamicCountPage(queryParam.getSearchText(), newP);
        IPage<ProjectApplyVO> page = new Page<>();
        if(count > 0) {
            long pageSize = queryParam.getPageSize();
            long pageNum = queryParam.getPageIndex();
            long startLine = (pageNum - 1 < 0 ? 0 : pageNum - 1) * pageSize;

            Map<String, String> orderMap = queryParam.getOrderMap();
            String orderBy = null;
            if(null != orderMap && !orderMap.isEmpty()) {
                StringBuilder sp = new StringBuilder(" ");
                for(String key : orderMap.keySet()) {
                    if(isValidKey(key) && isValidKey(orderMap.get(key))) {
                        sp.append("  a.").append(CamelAndUnderLineConverter.humpToLine(key)).append(" ").append(orderMap.get(key));
                    }
                }
                orderBy = sp.toString();
            }

            List<ProjectApplyVO> records = projectApplyService.dynamicPageList(queryParam.getSearchText(), newP, startLine, pageSize, orderBy);

            page.setCurrent(pageNum);
            page.setSize(pageSize);
            page.setTotal(count);
            page.setRecords(records);
        }
        return CommonResponse.success("分页查询成功！", page);
    }

    /**
     * 查询项目拆分详情
     *
     * @param projectId
     * @return
     */
    @GetMapping(value = "queryExecutiveAgentList")
    public CommonResponse<List<ExecutiveAgentVO>> queryExecutiveAgentList(@RequestParam(value = "projectId") Long projectId) {
    	QueryWrapper<ExecutiveAgentEntity> queryWrapper = new QueryWrapper<>();
    	queryWrapper.eq("dr", 0);
    	queryWrapper.eq("project_apply_id", projectId);
		List<ExecutiveAgentEntity> list = executiveAgentService.list(queryWrapper);
		List<ExecutiveAgentVO> backList = new ArrayList<>();
		if(list!=null&&list.size()>0){
			backList = BeanMapper.mapList(list, ExecutiveAgentVO.class);
		}
        return CommonResponse.success("查询详情成功！", backList);
    }

    /**
     * 变更启用状态
     *
     * @param dataList
     * @return
     */
    @PostMapping(value = "changeStatus")
    public CommonResponse<String> changeStatus(@RequestBody List<ProjectApplyVO> dataList) {
    	List<OrgVO> orgVOs = new ArrayList<>();
    	List<Long> projectIds = new ArrayList<>();
    	if(dataList!=null&&dataList.size()>0){
    		for(ProjectApplyVO vo : dataList){
    			projectIds.add(vo.getId());
    		}
    	}else{
    		return CommonResponse.error("数据不能为空");
    	}
    	List<ProjectApplyEntity> list = (List<ProjectApplyEntity>) projectApplyService.listByIds(projectIds);
    	if(!(list!=null&&list.size()>0)){
    		return CommonResponse.error("数据不能为空");
    	}
    	Integer status = dataList.get(0).getStatus();
    	//TODO
        for(ProjectApplyEntity entity : list){
    		entity.setStatus(status);
			CommonResponse<OrgVO> data = orgApi.detailById(entity.getProjectDepartmentId());
			if(data.isSuccess()&&data.getData()!=null){
				orgVOs.add(data.getData());
			}else{
				throw new BusinessException("查询["+entity.getCnName()+"]对应的项目部信息失败");
			}
        }
        projectApplyService.saveOrUpdateBatch(list, list.size(), false);
        
        //同步项目部的启停用信息
        for(OrgVO orgVO : orgVOs){
            /**
             * 项目状态：0-未开工（项目部状态：启用），1-在建（项目部状态：启用），2-停工（项目部状态：停用），3-已完工（项目部状态：停用）
             */
        	if(status.equals(ProjectApplyVO.PROJECT_STATUS_NO_START)||status.equals(ProjectApplyVO.PROJECT_STATUS_UNDERCONSTRUCTION)){
        		orgVO.setState(1);//启用
        	}else{
        		orgVO.setState(0);//停用
        	}
        }
        orgApi.saveProOrgList(orgVOs);
        return CommonResponse.success("改变状态成功！");
    }

    /**
     * 项目信息列表导出
     *
     * @param queryParam
     * @param response
     */
    @PostMapping(value = "excelExport")
    public void excelExport(@RequestBody QueryParam queryParam, HttpServletResponse response) {
        List<ProjectApplyVO> voList = null;
        Map<Integer, String> statusMap = new HashMap<>();
        statusMap.put(0, "未开工");
        statusMap.put(1, "在建");
        statusMap.put(2, "停用");
        statusMap.put(3, "已完工");

        queryParam.setPageIndex(1);
        queryParam.setPageSize(10000);
        IPage<ProjectApplyVO> page = queryList(queryParam).getData();
        voList = page.getRecords();
        voList.forEach(vo -> {
            vo.setSplitTypeDisplay(vo.getProjectSplit() ? "拆分" : "非拆分");
            vo.setStatusDisplay(statusMap.get(vo.getStatus()));
        });

        Map<String, Object> beans = new HashMap<String, Object>();
        beans.put("records", voList);
        ExcelExport.getInstance().export("sddjsProjectInfo-export.xlsx", beans, response);
    }
}
