package com.ejianc.foundation.share.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.HttpTookit;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author CJ
 * @Description:
 * @date 2022/1/15 14:31
 */
@Component
public class FileUtil {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static String baseHost;

    private static final FileUtil instance = new FileUtil();

    private FileUtil() {}

    public static FileUtil getInstance() {
        return instance;
    }

    /**
     * 文件批量下载
     *
     * @param fileIds 待下载附件列表
     * @param mustSuc 是否必须下载成功
     * @return
     */
    public Map<String, InputStream> batchDownFileFlow(List<Long> fileIds, boolean mustSuc) {
        Map<String, InputStream> resp = new HashMap<>();

        String url = baseHost + "ejc-file-web/attachment/batchdownflow";
        Map<String, String> params = new HashMap();
        params.put("fileIds", StringUtils.join(fileIds, ","));
        String downloadFileStr = null;

        try {
            //下载附件
            downloadFileStr = ReferHttpClientUtils.getAndHeader(url, params);
            logger.info("根据url-{}，参数-{}", url, JSONObject.toJSONString(params));

            //解析附件信息
            Map<String, String> respMap = JSONObject.parseObject(downloadFileStr, new TypeReference<Map<String, String>>(){});
            if(null == respMap.get("data")) {
                logger.error("根据附件Id下载附件为空", JSONObject.toJSONString(fileIds));
                return new HashMap<>();
            }

            // 文件较大，内存溢出修改  edit by yqls 20220531
//            Map<String, byte[]> fileDataMap = JSONObject.parseObject(JSON.parseObject(respMap.get("data")).toJSONString(), new TypeReference<Map<String, byte[]>>(){});
//            for(String fileName : fileDataMap.keySet()) {
//                resp.put(fileName, new ByteArrayInputStream(fileDataMap.get(fileName)));
//            }
            //将附件根据附件名 转换为 以附件业务类型为键，列表存放同一业务类型的附件 的集合
            JSONObject json = JSON.parseObject(respMap.get("data"));
            for(String key : json.keySet()){
                resp.put(key, new ByteArrayInputStream(json.getBytes(key)));
            }

        } catch (Exception e) {
            logger.error("根据附件Id下载附件异常-{}", JSONObject.toJSONString(fileIds), e);
            if(mustSuc) {
                throw new BusinessException("下载附件异常，", e);
            }
        }
        return resp;
    }

    /**
     * 获取到请求中附件信息，并上传至文件中心，返回附件Id列表
     *
     * @param multipartRequest 请求Req对象
     * @param fileSourceTypeMapping 附件对应业务类型
     * @param billType 附件单据类型
     * @param authority 上下文信息
     * @return
     */
    public Map<String, List<Long>> handleReqFile(MultipartHttpServletRequest multipartRequest, Map<String, String> fileSourceTypeMapping, String billType, String authority, String sourceId) {
        Map<String, List<Long>> resp = new HashMap<>();

        // 获取file框
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        // 文件列表
        Map<String, Map<String, Map<String, InputStream>>> fileMaps = new HashMap<>();
        Map<String, InputStream> fileItemMap = null;

        try {
            // 文件类型判断
            for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
                MultipartFile mf = entity.getValue();
                String originalFilename = mf.getOriginalFilename();
                if(null == fileMaps.get(fileSourceTypeMapping.get(originalFilename))) {
                    fileMaps.put(fileSourceTypeMapping.get(originalFilename), new HashMap<>());
                }
                fileItemMap = new HashMap<>();
                fileItemMap.put(originalFilename, mf.getInputStream());

                fileMaps.get(fileSourceTypeMapping.get(originalFilename)).put(originalFilename, fileItemMap);
            }

            fileMaps.keySet().forEach(sourceType -> {
                List<AttachmentVO> attachs = upFile(sourceType, sourceId, billType, fileMaps.get(sourceType), authority);
                if(CollectionUtils.isNotEmpty(attachs)) {
                    resp.put(sourceType, attachs.stream().map(item -> item.getId()).collect(Collectors.toList()));
                }
            });
        } catch (Exception e) {
            throw new BusinessException("将请求中附件信息上传至文件中心异常: ", e);
        }

        return resp;
    }

    /**
     * 上传附件入口
     *
     * @param sourceType 业务类型
     * @param sourceId 业务单据Id
     * @param billType 单据类型编码
     * @param files 文件
     * @param authority 上下文信息
     * @return
     */
    public List<AttachmentVO> upFile(String sourceType, String sourceId, String billType, Map<String, Map<String, InputStream>> files, String authority){
        logger.info("获取认证信息--"+authority);
        List<AttachmentVO> attachs = new ArrayList<>();

        if(StringUtils.isNotBlank(authority)) {
            InvocationInfoProxy.setExtendAttribute("authority", authority);
            Map<String,String> params = new HashMap<>();
            if(StringUtils.isNotBlank(sourceId)) {
                params.put("sourceId", sourceId);
            }
            if(StringUtils.isNotBlank(sourceType)) {
                params.put("sourceType", sourceType);
            }
            params.put("billType", billType);
            Map<String,String> headers = new HashMap<>();
            headers.put("authority", authority);

            try {
                //保存新的模板文件
                logger.info("文件上传参数：params-{}, headers-{}, 文件数量-{}", params, headers, files.size());
                String fileUploadRespStr = HttpTookit.postFiles(baseHost +"ejc-file-web/attachment/upload", params, headers, files, HttpTookit.connTimeout, HttpTookit.readTimeout);
                logger.info("文件上传响应结果：{}", fileUploadRespStr);

                CommonResponse<List<AttachmentVO>> attachmentResp = JSONObject.parseObject(fileUploadRespStr, CommonResponse.class);
                logger.info("上传文件返回信息: {}", JSONObject.toJSONString(attachmentResp));

                if(!attachmentResp.isSuccess()) {
                    logger.error("保存模板文件失败sourceId-{}, sourceType-{}, billType-{}, 原因：{}",
                            sourceId, sourceType, billType, attachmentResp.getMsg());
                    return null;
                }

                attachs = JSONObject.parseArray(JSONObject.toJSONString(attachmentResp.getData()), AttachmentVO.class);
            } catch (Exception e) {
                logger.error("保存模板文件失败sourceId-{}, sourceType-{}, billType-{}, 原因：",
                        sourceId, sourceType, billType, e);
                return null;
            }
            logger.info("模板文件保存成功：{}", JSONObject.toJSONString(attachs));
        }

        return attachs;
    }

    /**
     * 文件上传
     *
     * @param url 文件上传Url
     * @param params 文件上传参数
     * @param headers 请求头参数
     * @param inputStream 文件
     * @param fileName 文件名称
     * @return
     * @throws Exception
     */
    public static String postFile(String url, Map<String, String> params, Map<String, String> headers, InputStream inputStream, String fileName) throws Exception {
        String charset = "UTF-8";
        HttpClient client = null;

        HttpPost post = new HttpPost(url);
        HttpServletRequest request = null;
        String result = "";
        Integer connTimeout = 10000;
        Integer readTimeout = 10000;

        try {
            MultipartEntityBuilder builder = MultipartEntityBuilder.create().setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            builder.setCharset(Charset.forName("UTF-8")).addBinaryBody("file",
                    inputStream, ContentType.MULTIPART_FORM_DATA.withCharset("UTF-8"), fileName);
            for(String key : params.keySet()) {
                builder.addPart(key, new StringBody(params.get(key), ContentType.MULTIPART_FORM_DATA.withCharset("UTF-8")));
            }
            HttpEntity entity = builder.build();
            post.setEntity(entity);

            RequestConfig.Builder customReqConf = RequestConfig.custom();
            if (connTimeout != null) {
                customReqConf.setConnectTimeout(connTimeout);
            }

            if (readTimeout != null) {
                customReqConf.setSocketTimeout(readTimeout);
            }

            post.setConfig(customReqConf.build());
            if (null != headers) {
                String key = null;
                Iterator it = headers.keySet().iterator();

                while(it.hasNext()) {
                    key = (String)it.next();
                    post.addHeader(key, (String)headers.get(key));
                }
            } else {
                request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
                post.addHeader("authority", request.getHeader("authority"));
                post.addHeader("ejc-token", request.getHeader("ejc-token"));
            }

            HttpResponse res;
            if (url.startsWith("https")) {
                client = HttpTookit.createSSLInsecureClient();
                res = ((HttpClient)client).execute(post);
            } else {
                PoolingHttpClientConnectionManager cm = new PoolingHttpClientConnectionManager();
                cm.setMaxTotal(128);
                cm.setDefaultMaxPerRoute(128);
                client = HttpClients.custom().setConnectionManager(cm).build();
                res = ((HttpClient)client).execute(post);
            }

            result = IOUtils.toString(res.getEntity().getContent(), charset);
        } finally {
            post.releaseConnection();
            if (url.startsWith("https") && client != null && client instanceof CloseableHttpClient) {
                ((CloseableHttpClient)client).close();
            }
        }

        return result;
    }

    public String getBaseHost() {
        return baseHost;
    }

    @Value("${common.env.base-host}")
    public void setBaseHost(String baseHost) {
        FileUtil.baseHost = baseHost;
    }
}
