package com.ejianc.foundation.share.controller;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.ejianc.framework.core.response.Parameter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.foundation.share.bean.EquipmentCategoryEntity;
import com.ejianc.foundation.share.service.IEquipmentCategoryService;
import com.ejianc.foundation.share.service.IEquipmentService;
import com.ejianc.foundation.share.vo.EquipmentCategoryVO;
import com.ejianc.foundation.share.vo.EquipmentVO;
import com.ejianc.foundation.share.vo.MaterialCategoryImportVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelReader;
import com.ejianc.framework.core.util.FileUtils;
import com.ejianc.framework.core.util.ImportTemplate;
import com.ejianc.framework.core.util.ResultAsTree;
import com.ejianc.support.idworker.util.IdWorker;

@SuppressWarnings("rawtypes")
@RestController
@RequestMapping("/equipment/category/")
public class EquipmentCategoryController implements Serializable {

    private static final long serialVersionUID = 1L;
    private static final String EQUIPMENT_CATEGORY_BILL_CODE = "SUPPORT_EQUIPMENT_CATEGORY";

    @Value("${oms.tenantid}")
    private String OMS_TENANT;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IEquipmentCategoryService equipmentCategoryService;
    @Autowired
    private IEquipmentService equipmentService;
    @Autowired
    private IBillCodeApi billCodeApi;

    /**
     * 新增或修改
     *
     * @param equipmentCategoryVo
     * @return
     */
    @RequestMapping(value = "/saveOrUpdate", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> saveOrUpdate(@RequestBody EquipmentCategoryVO equipmentCategoryVo) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if (equipmentCategoryVo.getId() != null && equipmentCategoryVo.getId() > 0) {
            if (StringUtils.isNoneBlank(equipmentCategoryVo.getCode())) {
                List<EquipmentCategoryEntity> codeData = equipmentCategoryService.queryByCode(equipmentCategoryVo.getCode());
                if (codeData != null && codeData.size() > 0) {
                    if (!equipmentCategoryVo.getId().equals(codeData.get(0).getId())) {
                        throw new BusinessException("分类编码已存在,请重新填写");
                    }
                }
            } else {
                throw new BusinessException("分类编码不能为空");
            }
            EquipmentCategoryVO uniqueBean = equipmentCategoryService.queryDetail(equipmentCategoryVo.getId());
            uniqueBean.setCode(equipmentCategoryVo.getCode());
            uniqueBean.setName(equipmentCategoryVo.getName());
            uniqueBean.setDescription(equipmentCategoryVo.getDescription());
            uniqueBean.setSequence(equipmentCategoryVo.getSequence());
            uniqueBean.setUpdateUserCode(InvocationInfoProxy.getUsercode());
            uniqueBean.setUpdateTime(equipmentCategoryVo.getUpdateTime());
            uniqueBean.setSubjectId(equipmentCategoryVo.getSubjectId());
            uniqueBean.setSubjectName(equipmentCategoryVo.getSubjectName());
            uniqueBean.setSequence(equipmentCategoryVo.getSequence());
            EquipmentCategoryEntity entity = BeanMapper.map(uniqueBean, EquipmentCategoryEntity.class);
            equipmentCategoryService.update(entity);
            return CommonResponse.success("修改成功");
        } else {
            if (StringUtils.isNoneBlank(equipmentCategoryVo.getCode())) {
                List<EquipmentCategoryEntity> codeData = equipmentCategoryService.queryByCode(equipmentCategoryVo.getCode());
                if (codeData != null && codeData.size() > 0) {
                    throw new BusinessException("分类编码已存在,请重新填写");
                }
            } else {
                CommonResponse<String> codeBatchByRuleCode = billCodeApi.getCodeBatchByRuleCode(EQUIPMENT_CATEGORY_BILL_CODE, tenantId);
                if (codeBatchByRuleCode.isSuccess()) {
                    equipmentCategoryVo.setCode(codeBatchByRuleCode.getData());
                }
            }
            EquipmentCategoryEntity entity = BeanMapper.map(equipmentCategoryVo, EquipmentCategoryEntity.class);
            entity.setId(IdWorker.getId());
            if (entity.getParentId() != null && entity.getParentId() > 0) {
                EquipmentCategoryVO uniqueBean = equipmentCategoryService.queryDetail(entity.getParentId());
                entity.setInnerCode(uniqueBean.getInnerCode() + "|" + entity.getId());
            } else {
                entity.setInnerCode(entity.getId().toString());
            }
            entity.setEnabled(1);
            entity.setCreateUserCode(InvocationInfoProxy.getUsercode());
            entity.setCreateTime(new Date());
            entity.setTenantId(InvocationInfoProxy.getTenantid());
            equipmentCategoryService.save(entity);
            return CommonResponse.success("保存成功");
        }
    }

    /**
     * 查询列表
     *
     * @param queryParam
     * @return
     */
    @RequestMapping(value = "/queryList", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> queryList(@RequestBody QueryParam queryParam) {
        List<EquipmentCategoryEntity> dataList = equipmentCategoryService.queryList(queryParam);
        List<Map> resultMapList = BeanMapper.mapList(dataList, Map.class);
        resultMapList.forEach(l -> {
            if (l.get("leaf").equals(false)) {
                l.put("children", new ArrayList<>());
            }
        });
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("data", ResultAsTree.createTreeData(resultMapList));
        return CommonResponse.success(jsonObject);
    }


    /**
     * @Description queryList 查询列表
     * @param queryParam
     * @Return com.ejianc.framework.core.response.CommonResponse<java.lang.String>
     */
    @RequestMapping(value = "/queryListData", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<List<Map<String, Object>>> queryListData(@RequestBody QueryParam queryParam) {
        /** 模糊搜索配置字段示例 */
        List<EquipmentCategoryEntity> dataList = equipmentCategoryService.queryList(queryParam);
        List<Map> resultMapList = BeanMapper.mapList(dataList, Map.class);
        resultMapList.forEach(l -> {
            if (l.get("leaf").equals(false)) {
                l.put("children", new ArrayList<>());
            }
        });
        List<Map<String, Object>> treeData = ResultAsTree.createTreeData(resultMapList);
        return CommonResponse.success(treeData);
    }


    /**
     * 根据主键查询设备分类详情
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/queryDetail", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<EquipmentCategoryVO> queryDetail(@RequestParam Long id) {
        EquipmentCategoryVO equipmentCategoryVo = equipmentCategoryService.queryDetail(id);
        return CommonResponse.success(equipmentCategoryVo);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @RequestMapping(value = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> delete(@RequestBody List<Long> ids) {
        if (ListUtil.isNotEmpty(ids)) {
            for (Long id : ids) {
                EquipmentCategoryVO equipmentCategoryVO = equipmentCategoryService.queryDetail(id);
                //分类下有设备了不允许删除
                List<EquipmentVO> equipmentList = equipmentService.queryListByCategoryId(id);
                if (equipmentList != null && equipmentList.size() > 0) {
                    return CommonResponse.error("分类【" + equipmentCategoryVO.getName() + "】下有设备了，不允许删除");
                }
                //如果分类有子节点并且所有子节点也包含在删除项内，允许删除
                List<EquipmentCategoryVO> children = equipmentCategoryService.queryListByPid(id);
                if(children != null && children.size() > 0) {
                    for (EquipmentCategoryVO child : children) {
                        if(!ids.contains(child.getId())){
                            return CommonResponse.error("分类【" + equipmentCategoryVO.getName() + "】下有子分类不允许删除");
                        }
                    }
                }
            }
        } else {
            return CommonResponse.error("删除失败：主键为空!");
        }
        equipmentCategoryService.delete(ids);
        return CommonResponse.success("删除成功");
    }

    /**
     * 更新父节点状态，同时更新子节点状态
     *
     * @return
     */
    @RequestMapping(value = "/enabled/update", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> updateEnabled(@RequestBody EquipmentCategoryVO equipmentCategoryVo) {
        EquipmentCategoryVO condition = equipmentCategoryService.queryDetail(equipmentCategoryVo.getId());
        equipmentCategoryService.updateEnabled(equipmentCategoryVo.getEnabled(), condition.getTenantId(), condition.getInnerCode());
        return CommonResponse.success("更新成功");
    }

    /**
     * 同步平台设备分类
     *
     * @return
     */
    @RequestMapping(value = "/sync", method = RequestMethod.GET)
    @ResponseBody
    public CommonResponse<String> syncEquipmentCategoryList() {
        Long tenantId = InvocationInfoProxy.getTenantid();
        if (OMS_TENANT.equals(tenantId.toString())) {
            return CommonResponse.error("平台运营人员不能操作此功能");
        }
        equipmentCategoryService.insertCategoryListFromPlatform();
        return CommonResponse.success("同步成功");
    }

    /**
     * 设备分类树
     *
     * @param searchText
     * @return
     */
    @RequestMapping(value = "/queryEquipmentRefTree", method = RequestMethod.GET)
    @ResponseBody
    public List<Map<String, Object>> queryEquipmentRefTree(@RequestParam(required = false) String searchText) {
        List<EquipmentCategoryVO> dataList = null;
        try {
            Map<String, Object> params = new HashMap<>();
            if (StringUtils.isNotBlank(searchText)) {
                params.put("searchText", searchText);
            }
            params.put("tenantId", InvocationInfoProxy.getTenantid());
            dataList = equipmentCategoryService.queryListTree(params);
            List<Map> map = BeanMapper.mapList(dataList, Map.class);
            return ResultAsTree.createTreeData(map);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 参照异步（懒加载）树----设备分类树
     *
     * @param condition
     * @param searchText
     * @return
     */
    @RequestMapping(value = "/lazyEquipmentCategoryTree", method = RequestMethod.GET)
    @ResponseBody
    public List<Map<String, Object>> lazyEquipmentCategoryTree(@RequestParam(required = false) String condition,
                                                               @RequestParam(required = false) Long pid, @RequestParam(required = false) String searchText) {
        List<EquipmentCategoryVO> dataList = null;
        try {
            Map<String, Object> params = new HashMap<>();
            if (StringUtils.isNotBlank(searchText)) {
                params.put("searchText", searchText);
                params.put("tenantId", InvocationInfoProxy.getTenantid());
                if(StringUtils.isNotBlank(condition)){
                    JSONObject con = JSONObject.parseObject(condition);
                    Object categoryIdObj = con.get("categoryId");
                    if (null != categoryIdObj) {
                        List<Long> asList = new ArrayList<>();
                        String categoryId = ((String) categoryIdObj);
                        if (categoryId.indexOf(",") > 0) {
                            String[] strings = categoryId.split(",");
                            for (String str : strings) {
                                asList.add(Long.valueOf(str));
                            }
                        } else {
                            asList.add(Long.valueOf(categoryId));
                        }
                        List<EquipmentCategoryVO> categoryList = new ArrayList<>();
                        for(Long parentId : asList){
                            categoryList.addAll(equipmentCategoryService.queryListByPid(parentId));
                        }
                        asList = categoryList.stream().map(EquipmentCategoryVO::getId).collect(Collectors.toList());
                        params.put("ids", asList);
                    }
                }
                dataList = equipmentCategoryService.queryListTree(params);
            } else {
                dataList = equipmentCategoryService.queryListByPid(pid);
                if (StringUtils.isNotBlank(condition) && pid == null) {
                    /** 处理condition */
                    JSONObject con = JSONObject.parseObject(condition);
                    Object categoryIdObj = con.get("categoryId");
                    if (null != categoryIdObj) {
                        List<Long> asList = new ArrayList<>();
                        String categoryId = ((String) categoryIdObj);
                        if (categoryId.indexOf(",") > 0) {
                            String[] strings = categoryId.split(",");
                            for (String str : strings) {
                                asList.add(Long.valueOf(str));
                            }
                        } else {
                            asList.add(Long.valueOf(categoryId));
                        }
                        dataList = equipmentCategoryService.queryCategoryListByIds(asList);
                    }
                }
            }
            List<Map> map = BeanMapper.mapList(dataList, Map.class);
            return ResultAsTree.createTreeData(map);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/6/30 设备分类导入模板下载
     * @Description download
     * @Param [request, response]
     * @Return void
     */
    @RequestMapping(value = "/download")
    @ResponseBody
    public void download(HttpServletRequest request, HttpServletResponse response) {
        ImportTemplate.initialize(response);
        ImportTemplate.templetdownload(request, "material-category-import.xlsx", "设备分类导入模板");
    }

    @RequestMapping(value = "/excelImport", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<JSONObject> excelImport(HttpServletRequest request) {
        MultipartHttpServletRequest multipartRequest = (MultipartHttpServletRequest) request;
        Map<String, MultipartFile> fileMap = multipartRequest.getFileMap();
        boolean isFailed = false;
        MultipartFile mf = null;
        List<MaterialCategoryImportVO> successList = new ArrayList<>();
        List<MaterialCategoryImportVO> processList = new ArrayList<>();
        List<MaterialCategoryImportVO> errorList = new ArrayList<>();
        for (Map.Entry<String, MultipartFile> entity : fileMap.entrySet()) {
            mf = entity.getValue();
            String originalFileName = mf.getOriginalFilename();
            String extName = null;
            originalFileName = originalFileName.replaceAll("\\/|\\/|\\||:|\\?|\\*|\"|<|>|\\p{Cntrl}", "_");
            originalFileName.replaceAll("00.", "");
            extName = FileUtils.getFileExt(originalFileName, false);
            if (!"xls".equals(extName) && !"xlsx".equals(extName)) {
                isFailed = true;
                break;
            }
        }
        if (isFailed) {
            return CommonResponse.error("文件格式不合法");
        } else {
            List<List<String>> result = ExcelReader.readExcel(mf);
            if (result != null && result.size() > 0) {
                if (result.size() > 10000) {
                    return CommonResponse.error("分类数据超过10000条，请分批上传！");
                }
                Map<String, List<MaterialCategoryImportVO>> mapChildren = new HashMap<>();
                Map<String, MaterialCategoryImportVO> codeToData = new HashMap<>();
                Map<String, String> codeNotExistData = new HashMap<>();
                for (int i = 0; i < result.size(); i++) {
                    List<String> datas = result.get(i);
                    MaterialCategoryImportVO materialVo = new MaterialCategoryImportVO();
                    materialVo.setId(IdWorker.getId());
                    materialVo.setCode(datas.get(0));
                    materialVo.setParentCode(datas.get(1));
                    materialVo.setName(datas.get(2));
                    if (StringUtils.isBlank(datas.get(0))) {
                        materialVo.setDescription("分类编码为空！");
                        errorList.add(materialVo);
                        continue;
                    }
                    if (StringUtils.isBlank(datas.get(2))) {
                        materialVo.setDescription("分类名称为空！");
                        errorList.add(materialVo);
                        continue;
                    }
                    //通过设备分类Code获取设备分类
                    List<EquipmentCategoryEntity> list = equipmentCategoryService.queryByCode(String.valueOf(datas.get(0)));
                    if (ListUtil.isEmpty(list)) {
                        //通过设备分类name获取设备分类
                        list = equipmentCategoryService.queryByName(String.valueOf(datas.get(2)));
                        if (ListUtil.isNotEmpty(list)) {
                            materialVo.setDescription("分类名称重复！");
                            errorList.add(materialVo);
                            continue;
                        }
                        if (StringUtils.isBlank(datas.get(1))) {
                            materialVo.setInnerCode(materialVo.getId().toString());
                        }
                        materialVo.setEnabled(1);
                        materialVo.setCreateUserCode(InvocationInfoProxy.getUsercode());
                        materialVo.setCreateTime(new Date());
                        materialVo.setTenantId(InvocationInfoProxy.getTenantid());
                        processList.add(materialVo);
                        codeToData.put(materialVo.getCode(), materialVo);
                        if (StringUtils.isBlank(materialVo.getParentCode())) {
                            List<MaterialCategoryImportVO> list1 = mapChildren.get("null");
                            if (ListUtil.isEmpty(list1)) {
                                list1 = new ArrayList<>();
                                mapChildren.put("null", list1);
                            }
                            list1.add(materialVo);
                        } else {
                            List<MaterialCategoryImportVO> list1 = mapChildren.get(materialVo.getParentCode());
                            if (ListUtil.isEmpty(list1)) {
                                list1 = new ArrayList<>();
                                mapChildren.put(materialVo.getParentCode(), list1);
                            }
                            list1.add(materialVo);
                        }
                    } else {
                        materialVo.setDescription("分类编码重复！");
                        errorList.add(materialVo);
                    }
                }
                if (ListUtil.isNotEmpty(processList)) {
                    processList.forEach(p -> {
                        if (StringUtils.isNotEmpty(p.getParentCode())) {
                            MaterialCategoryImportVO parentVo = codeToData.get(p.getParentCode());
                            if (parentVo == null) {
                                if (StringUtils.isNotEmpty(codeNotExistData.get(p.getParentCode()))) {
                                    p.setDescription("父分类编码不存在");
                                    errorList.add(p);
                                } else {
                                    List<EquipmentCategoryEntity> list = equipmentCategoryService.queryByCode(p.getParentCode());
                                    if (ListUtil.isEmpty(list)) {
                                        codeNotExistData.put(p.getParentCode(), p.getParentCode());
                                        p.setDescription("父分类编码不存在");
                                        errorList.add(p);
                                    } else {
                                        p.setInnerCode(list.get(0).getInnerCode() + "|" + p.getId());
                                        p.setParentId(list.get(0).getId());
                                        successList.add(p);
                                    }
                                }
                            } else {
                                if (StringUtils.isEmpty(parentVo.getInnerCode())) {
                                    setInnerCodeAndParentId(parentVo, codeToData, codeNotExistData);
                                    if (StringUtils.isEmpty(parentVo.getInnerCode())) {
                                        p.setDescription("父分类编码不存在");
                                        errorList.add(p);
                                    } else {
                                        p.setInnerCode(parentVo.getInnerCode() + "|" + p.getId());
                                        p.setParentId(parentVo.getId());
                                        successList.add(p);
                                    }
                                } else {
                                    p.setInnerCode(parentVo.getInnerCode() + "|" + p.getId());
                                    p.setParentId(parentVo.getId());
                                    successList.add(p);
                                }
                            }
                        } else {
                            successList.add(p);
                        }
                    });
                }
            }
        }
        JSONObject json = new JSONObject();
        json.put("successList", successList);
        json.put("errorList", errorList);
        return CommonResponse.success(json);
    }

    private void setInnerCodeAndParentId(MaterialCategoryImportVO vo, Map<String, MaterialCategoryImportVO> codeToData, Map<String, String> codeNotExistData) {
        MaterialCategoryImportVO parentVo = codeToData.get(vo.getParentCode());
        if (parentVo == null) {
            if (StringUtils.isNotEmpty(codeNotExistData.get(vo.getParentCode()))) {
                vo.setDescription("父分类编码不存在");
            } else {
                List<EquipmentCategoryEntity> list = equipmentCategoryService.queryByCode(vo.getParentCode());
                if (ListUtil.isEmpty(list)) {
                    codeNotExistData.put(vo.getParentCode(), vo.getParentCode());
                    vo.setDescription("父分类编码不存在");
                } else {
                    vo.setInnerCode(list.get(0).getInnerCode() + "|" + vo.getId());
                    vo.setParentId(list.get(0).getId());
                }
            }
        } else {
            if (StringUtils.isEmpty(parentVo.getInnerCode())) {
                setInnerCodeAndParentId(parentVo, codeToData, codeNotExistData);
                if (StringUtils.isEmpty(parentVo.getInnerCode())) {
                    vo.setDescription("父分类编码不存在");
                } else {
                    vo.setInnerCode(parentVo.getInnerCode() + "|" + vo.getId());
                    vo.setParentId(parentVo.getId());
                }
            } else {
                vo.setInnerCode(parentVo.getInnerCode() + "|" + vo.getId());
                vo.setParentId(parentVo.getId());
            }

        }
    }

    @RequestMapping(value = "/saveImportExcelEquipmentCategory", method = RequestMethod.POST)
    @ResponseBody
    public CommonResponse<String> saveImportExcelEquipmentCategory(@RequestBody List<MaterialCategoryImportVO> importVOS) {
        if (ListUtil.isEmpty(importVOS)) {
            return CommonResponse.error("导入的数据为空！");
        }
        List<EquipmentCategoryEntity> toSave = new ArrayList<>();
        importVOS.forEach(importVO -> toSave.add(BeanMapper.map(importVO, EquipmentCategoryEntity.class)));
        equipmentCategoryService.insertBatch(toSave);
        return CommonResponse.success("保存成功！");
    }
}
