package com.ejianc.business.signaturemanage.service.impl;

import cn.hutool.core.io.FileUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.signaturemanage.bean.SignMgrEntity;
import com.ejianc.business.signaturemanage.bean.SignMgrPreviewEntity;
import com.ejianc.business.signaturemanage.enums.DelFlagEnum;
import com.ejianc.business.signaturemanage.enums.WatermarkSchemeEnum;
import com.ejianc.business.signaturemanage.service.ISignMgrPreviewService;
import com.ejianc.business.signaturemanage.service.ISignMgrService;
import com.ejianc.business.signaturemanage.service.ISignatureCommonService;
import com.ejianc.business.signaturemanage.utils.UploadFile;
import com.ejianc.business.signaturemanage.utils.UploadFileCenter;
import com.ejianc.business.signaturemanage.utils.WatermarkConfigConvert;
import com.ejianc.business.signaturemanage.vo.ContractVO;
import com.ejianc.business.signaturemanage.vo.SignMgrPreviewVO;
import com.ejianc.business.signaturemanage.vo.WatermarkConfigVO;
import com.ejianc.business.signaturemanage.vo.WatermarkVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import net.qiyuesuo.sdk.SDKClient;
import net.qiyuesuo.sdk.api.ContractService;
import net.qiyuesuo.sdk.bean.contract.*;
import net.qiyuesuo.sdk.bean.document.DocumentCreateByUrl;
import net.qiyuesuo.sdk.bean.document.DownloadDocRequest;
import net.qiyuesuo.sdk.common.exception.PrivateAppException;
import net.qiyuesuo.sdk.impl.ContractServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author baipengyan
 * @date 2022/5/18
 * @description
 */
@Service("signatureCommonService")
public class SignatureCommonServiceImpl implements ISignatureCommonService {
	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final Map<String, Function<WatermarkVO, File>> watermarkMap = new HashMap<>();
	@Value("${common.env.base-host}")
	private String BASE_HOST; // 域名
	@Value("${ejc.fileServerUrl}")
	private String FILE_HOST; // 文件中心
	@Autowired
	private IAttachmentApi attachmentApi; // 文件信息
	@Autowired
	private UploadFileCenter upload; // 上传文件中心
	@Autowired
	private UploadFile uploadFile; // 上传文件工具类
	@Autowired
	private ISignMgrPreviewService signMgrPreviewService; // 已签章文件附件信息
	@Autowired
	private ISignMgrService signMgrService; // 签章服务
	@Autowired
	private WatermarkConfigConvert watermarkConfigConvert;

	@Value("${qiyuesuo.client.url}")
	private String url;// 私有化开放平台请求地址
	@Value("${qiyuesuo.client.accessKey}")
	private String accessKey;// 私有化开放平台申请的token
	@Value("${qiyuesuo.client.accessSecret}")
	private String accessSecret;// 私有化开放平台申请的secret

	@PostConstruct
	public void watermarkDispatcher() {
		// 加水印方案：契约锁
		watermarkMap.put(WatermarkSchemeEnum.QI_YUE_SUO.getCode(), watermarkVO -> {
			try {
				return addWatermarkIfQiyuesuo(watermarkVO);
			} catch (IOException ignored) {
			}
			return null;
		});

		// 其他方案
	}

	/**
	 * 合同添加水印
	 *
	 * @param watermarkVO 合同文件添加水印参数
	 */
	@Override
	public File addWatermark(WatermarkVO watermarkVO) {
		Function<WatermarkVO, File> res = watermarkMap.get(watermarkVO.getScheme());
		Assert.notNull(res, "合同添加水印的方案不存在，请联系管理员!");
		return res.apply(watermarkVO);
	}

	/**
	 * 删除临时文件
	 *
	 * @param file 要删除的文件
	 */
	@Override
	@Async(value = "commonTask")
	public void cleanFile(File file) throws IOException {
		Files.delete(Paths.get(file.getPath()));
	}

	private SDKClient getSdkClient() {
		// 初始化client
		SDKClient client = new SDKClient(url, accessKey, accessSecret);
		// 开启防止重放攻击
		client.enableNonce();
		return client;
	}

	private File addWatermarkIfQiyuesuo(WatermarkVO watermarkVO) throws IOException {
		logger.info("合同文件添加水印，入参：{}", JSON.toJSONString(watermarkVO, SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat));

		CommonResponse<AttachmentVO> res = attachmentApi.queryDetail(String.valueOf(watermarkVO.getFileId()));
		if (!res.isSuccess()) {
			logger.error("调用文件中心接口，获取附件失败！原因：{}", res.getMsg());
			throw new BusinessException("调用文件中心接口，获取附件失败！");
		}

		AttachmentVO attachmentVO = res.getData();
		logger.info("合同文件详情：{}", JSON.toJSONString(attachmentVO, SerializerFeature.WriteMapNullValue, SerializerFeature.PrettyFormat));
		String contractUrl = FILE_HOST + attachmentVO.getFilePath();
		Assert.hasText(contractUrl, "合同文件网络路径不能为空！");
		String fullName = attachmentVO.getFileName();
		String contractTitle = fullName.substring(0, fullName.lastIndexOf("."));
		Assert.hasText(contractTitle, "合同文件名称不能为空！");
		String contractFileType = fullName.substring(fullName.lastIndexOf(".") + 1);
		Assert.hasText(contractFileType, "合同文件类型不能为空！");
		if(contractTitle.length() < 3) {
			//文件名称长度小于3时，创建临时文件会报错
			contractTitle = contractTitle + "-" + DateFormatUtil.formatDate(DateFormatUtil.PATTERN_ISO_ON_DATE, new Date());

		}

		SDKClient sdkClient = getSdkClient();
		ContractService contractService = new ContractServiceImpl(sdkClient);


		DocumentCreateByUrl documentCreateByUrl = new DocumentCreateByUrl(contractUrl, contractTitle, contractFileType, null);
		// 创建合同文档
		Long documentId = null;
		try {
			documentId = contractService.createDocumentByUrl(documentCreateByUrl);
		} catch (PrivateAppException e) {
			throw new BusinessException(e.getMessage());
		}
		logger.info("合同文档id--{}", documentId);

		// 水印参数组装
		// List<WaterMarkContent> waterMarks = new ArrayList<>();
		for (WatermarkConfigVO vo : watermarkVO.getWatermarkConfigs()) {
			WaterMarkContent waterMarkContent = new WaterMarkContent();
			waterMarkContent.setDocumentId(documentId);
			waterMarkContent.setType(WaterMarkType.valueOf(vo.getType()));
			waterMarkContent.setContent(vo.getContent());
			waterMarkContent.setImageBase64(vo.getImageBase64());
			waterMarkContent.setFontSize(vo.getFontSize());
			waterMarkContent.setColor(vo.getColor());
			waterMarkContent.setRotateAngle(vo.getRotateAngle());
			waterMarkContent.setTransparency(vo.getTransparency());
			waterMarkContent.setLocation(WaterMarkLocation.valueOf(vo.getLocation()));
			try {
				contractService.addWatermark(waterMarkContent);
			} catch (PrivateAppException e) {
				throw new BusinessException("调用契约锁加水印接口失败，原因：{} " + e.getMessage());
			}
			// waterMarks.add(waterMarkContent);
		}

		// 下载合同文档
		DownloadDocRequest request = new DownloadDocRequest();

		File file = File.createTempFile(contractTitle, ".pdf");
		OutputStream out = FileUtil.getOutputStream(file);
		request.setDocumentId(documentId);
		request.setContact(contractTitle);
		request.setOutputStream(out);
		try {
			contractService.downloadDoc(request);
		} catch (PrivateAppException e) {
			throw new BusinessException("调用契约锁下载合同文档接口失败，原因：{} " + e.getMessage());
		}
		Path target = Paths.get(file.getParent() + "/" + contractTitle + ".pdf");
		Files.move(file.toPath(), target, StandardCopyOption.REPLACE_EXISTING);
		return new File(String.valueOf(target));
	}


	/**
	 * 水印文件上传文件中心
	 *
	 * @param file        水印文件
	 * @param authority   上下文
	 * @param watermarkVO 合同文件添加水印参数
	 *
	 * @return AttachmentVO 水印附件
	 */
	@Override
	public AttachmentVO fetchAttachment(File file, String authority, WatermarkVO watermarkVO) {
		CommonResponse<List<AttachmentVO>> res = upload.uploadFile(file, authority, watermarkVO.getSourceType(), watermarkVO.getBillId(), watermarkVO.getBillType(), "false", BASE_HOST);
		if (!res.isSuccess() || res.getData() == null) {
			logger.error("水印文件上传文件中心失败，失败原因：{}，请求文件中心入参：WatermarkVO--{}", res.getMsg(), JSON.toJSONString(watermarkVO, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
			throw new BusinessException("水印文件上传文件中心失败，失败原因：" + res.getMsg());
		}
		return JSON.parseObject(JSON.toJSONString(res.getData().get(0)), AttachmentVO.class);
	}


	/**
	 * @param fileId     要加水印的文件id
	 * @param billId     业务id
	 * @param billCode   合同编码
	 * @param billType   单据类型编码
	 * @param sourceType 业务类型编码
	 *
	 * @return WatermarkVO 水印配置
	 */
	@Override
	public WatermarkVO watermarkConfigConvert(Long fileId, Long billId, String billCode, String billType, String sourceType) {
		return watermarkConfigConvert.convert(fileId, billId, billCode, billType, sourceType);
	}


	/**
	 * 根据合同获取文件中心已签章的合同信息
	 *
	 * @param contracts 合同
	 *
	 * @return List<SignMgrPreviewVO> 已签章的合同信息
	 */
	@Override
	public List<SignMgrPreviewVO> fetchSignedContract(List<ContractVO> contracts) {
		// 校验合同状态
		List<Long> contractIds = contracts.stream().map(ContractVO::getContractId).collect(Collectors.toList());
		contracts.forEach(contract -> {
			LambdaQueryWrapper<SignMgrEntity> sm = new LambdaQueryWrapper<>();
			sm.eq(SignMgrEntity::getDelFlag, DelFlagEnum.NORMAL.getDelFlag());
			sm.eq(SignMgrEntity::getBillId, contract.getContractId());
			SignMgrEntity entity = signMgrService.getOne(sm);
			Assert.notNull(entity, "PM系统合同id--" + contract.getContractId() + "，查询不到签章记录！");
			// 设置第三方合同id
			contract.setThirdContractId(entity.getSourceBillId());

			ContractDetail detail = signMgrService.detail(entity.getSourceBillId(), false);
			Assert.notNull(detail, "PM系统合同id--" + contract.getContractId() + "契约锁合同id--" + entity.getSourceBillId() + "，查询不到合同信息！");
			if (!ContractStatus.COMPLETE.equals(detail.getStatus())) {
				throw new BusinessException("PM系统合同id--" + contract.getContractId() + "契约锁合同id--" + entity.getSourceBillId() + "，合同状态不是已签章！");
			}
		});

		// 先去查询签章中心已签章的合同附件信息
		LambdaQueryWrapper<SignMgrPreviewEntity> spWrapper = new LambdaQueryWrapper<>();
		spWrapper.eq(SignMgrPreviewEntity::getDelFlag, 0);
		spWrapper.in(SignMgrPreviewEntity::getBillId, contractIds);
		List<SignMgrPreviewVO> sps = BeanMapper.mapList(signMgrPreviewService.list(spWrapper), SignMgrPreviewVO.class);

		// 未命中，则去下载对应已签章合同文件，上传文件中心，写入数据库并返回附件信息
		if (CollectionUtils.isEmpty(sps)) {
			sps.addAll(this.syncSignedContract(contracts));
		} else {
			contractIds.removeAll(sps.stream().map(SignMgrPreviewVO::getBillId).collect(Collectors.toList()));
			if (CollectionUtils.isNotEmpty(contractIds)) {
				sps.addAll(this.syncSignedContract(contracts));
			}
		}

		return sps;
	}

	/**
	 * 同步已签章的合同附件信息
	 *
	 * @param contracts 合同
	 *
	 * @return List<SignMgrPreviewVO> 已签章的合同附件信息
	 */
	private List<SignMgrPreviewVO> syncSignedContract(List<ContractVO> contracts) {
		// 同步获取合同附件信息
		List<SignMgrPreviewEntity> res = new ArrayList<>();
		SignMgrPreviewEntity spv = new SignMgrPreviewEntity();

		contracts.forEach(contract -> {
			try {
				// 从契约锁下载已签章的合同文件
				File file = File.createTempFile(contract.getContractName(), ".pdf");
				OutputStream out = Files.newOutputStream(file.toPath());
				signMgrService.downloadHasSignedFile(contract.getThirdContractId(), out);

				Path target = Paths.get(file.getParent() + "/" + contract.getContractName() + ".pdf");
				Files.move(file.toPath(), target, StandardCopyOption.REPLACE_EXISTING);
				File file1 = new File(target.toString());
				// 获取authority
				String authority = ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest().getHeader("authority");
				// 文件上传
				// fixme: 修复上传
				// AttachmentVO attachmentVO = uploadFile.upload(authority, file1, contract.getContractId(), contract.getBillType(), contract.getSourceType(), "true");
				CommonResponse<List<AttachmentVO>> result = upload.uploadFile(file1, authority, contract.getSourceType(), contract.getContractId(), contract.getBillType(), "true", BASE_HOST);
				if (!result.isSuccess() || result.getData() == null) {
					logger.error("水印文件上传文件中心失败，失败原因：{}，请求文件中心入参：contract--{}", result.getMsg(), JSON.toJSONString(contract, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
					throw new BusinessException("水印文件上传文件中心失败，失败原因：" + result.getMsg());
				}
				// 合同附件信息
				AttachmentVO attachment = JSON.parseObject(JSON.toJSONString(result.getData().get(0)), AttachmentVO.class);

				spv.setDelFlag(DelFlagEnum.NORMAL.getDelFlag());
				spv.setBillId(attachment.getSourceId());
				spv.setBillType(attachment.getBillType());
				spv.setSourceType(attachment.getSourceType());
				spv.setFileId(attachment.getId());
				spv.setFileName(attachment.getFileName());
				spv.setFilePath(attachment.getFilePath());
				spv.setFileSize(attachment.getFileSize());
				spv.setOnlinePath(attachment.getOnlinePath());
				spv.setTruePath(attachment.getTruePath());
				res.add(spv);

				// 删除文件
				this.cleanFile(file1);
			} catch (IOException e) {
				throw new BusinessException(e.getMessage());
			}
		});
		Assert.notEmpty(res, "同步合同附件信息失败！");

		// 批量保存
		signMgrPreviewService.saveBatch(res, 5);
		return BeanMapper.mapList(res, SignMgrPreviewVO.class);
	}

	private List<AttachmentVO> syncDownloadAndUploadSignedContract(List<ContractVO> contracts) {
		List<AttachmentVO> attachments = new ArrayList<>();
		contracts.forEach(contract -> {
			try {
				// 从契约锁下载已签章的合同文件
				File file = File.createTempFile(contract.getContractName(), ".pdf");
				OutputStream out = Files.newOutputStream(file.toPath());
				signMgrService.downloadHasSignedFile(contract.getThirdContractId(), out);

				Path target = Paths.get(file.getParent() + "/" + contract.getContractName() + ".pdf");
				Files.move(file.toPath(), target, StandardCopyOption.REPLACE_EXISTING);
				File file1 = new File(target.toString());
				// 获取authority
				String authority = ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest().getHeader("authority");
				// 文件上传
				// AttachmentVO attachmentVO = uploadFile.upload(authority, file1, contract.getContractId(), contract.getBillType(), contract.getSourceType(), "true");
				CommonResponse<List<AttachmentVO>> res = upload.uploadFile(file1, authority, contract.getSourceType(), contract.getContractId(), contract.getBillType(), "true", BASE_HOST);
				if (!res.isSuccess() || res.getData() == null) {
					logger.error("水印文件上传文件中心失败，失败原因：{}，请求文件中心入参：contract--{}", res.getMsg(), JSON.toJSONString(contract, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
					throw new BusinessException("水印文件上传文件中心失败，失败原因：" + res.getMsg());
				}
				// 合同附件信息
				attachments.add(JSON.parseObject(JSON.toJSONString(res.getData().get(0)), AttachmentVO.class));
				// 删除文件
				this.cleanFile(file1);
			} catch (IOException e) {
				throw new BusinessException(e.getMessage());
			}
		});
		return attachments;
	}
}
