package com.ejianc.framework.skeleton.refer.serialize;

import java.io.IOException;
import java.util.Arrays;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.skeleton.refer.annotation.ReferSerialTransfer;
import com.ejianc.framework.skeleton.refer.util.ReferObjectUtil;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.introspect.AnnotatedMethod;
import com.fasterxml.jackson.databind.ser.ContextualSerializer;

/**
 * 将String转换成json{id:xx,code:xx,name:xx}
 * <note>将String{id,id}转换成JsonArray[{id:xx,code:xx,name:xx},{id:xx,code:xx,name:
 * xx}]</note>
 * 
 * @author guominga 2020年02月13日
 *
 */

public class ReferSerializer extends JsonSerializer<String> implements ContextualSerializer {
	private final static org.slf4j.Logger Log = LoggerFactory.getLogger(ReferSerializer.class);

	/**
	 * 实体引用对应code
	 */
	private String referCode;

	private String[] referCodes;

	private String[] extraFileds;

	private static final String DEFDOC_REFER_CODE = "defdoc";

	public ReferSerializer(String referCode, String[] referCodes, String[] extraFileds) {
		super();
		this.referCode = referCode;
		this.referCodes = referCodes;
		this.extraFileds = extraFileds;
	}

	// 必须要保留无参构造方法
	public ReferSerializer() {
		super();
	}

	@Override
	public void serialize(String value, JsonGenerator jgen, SerializerProvider provider)
			throws IOException, JsonProcessingException {
		String[] ids = StringUtils.isEmpty(value) ? null : value.split(",");
		if (ids != null && ids.length > 0) {
			boolean isSingle = ids.length == 1 && value.indexOf(",") == -1;
			JSONArray jsonArray = null;
			try {
				// 单个参照code
				jsonArray = ReferObjectUtil.getReferEntityValue(Arrays.asList(ids), referCode, referCodes, extraFileds);
			} catch (Exception e) {
				Log.error("referCode:" + referCode + " id：" + value + "转换出错，--" + e.getMessage(), e);
			}
			if (jsonArray != null && jsonArray.size() > 0) {
				// 单选
				if (isSingle) {
					jgen.writeObject(jsonArray.get(0));
				}
				// 多选
				else {
					jgen.writeObject(jsonArray);
				}
			} else {
				JSONObject result = new JSONObject();
				result.put("id", value);
				result.put("code", value);
				result.put("name", value);
				// 单选
				if (isSingle) {
					jgen.writeObject(result);
				}
				// 多选
				else {
					jsonArray = new JSONArray();
					jsonArray.add(result);
					jgen.writeObject(jsonArray);
				}

			}
		} else {
			jgen.writeObject(null);
		}

	}

	@Override
	public JsonSerializer<?> createContextual(SerializerProvider serializerProvider, BeanProperty beanProperty)
			throws JsonMappingException {
		if (beanProperty != null) { // 为空直接跳过
			if (Objects.equals(beanProperty.getType().getRawClass(), String.class)) { // 非String类直接跳过
				AnnotatedMethod method = (AnnotatedMethod) beanProperty.getMember();
				ReferSerialTransfer referSerialTransfer = method.getAnnotated()
						.getDeclaredAnnotation(ReferSerialTransfer.class);
				if (referSerialTransfer != null) { // 如果能得到注解，就将注解的value传入ReferSerializer
					// xg modified on 2017-5-5 自定义档案参照特殊处理
					String referCode = referSerialTransfer.referCode();
					if (StringUtils.isNoneBlank(referCode)) {
						if (DEFDOC_REFER_CODE.equalsIgnoreCase(referCode)
								|| referCode.toLowerCase().startsWith(DEFDOC_REFER_CODE)) {
							referCode = DEFDOC_REFER_CODE;
						}
					}
					return new ReferSerializer(referCode, referSerialTransfer.referCodes(), referSerialTransfer.extraFileds());
				}
			}
			return serializerProvider.findValueSerializer(beanProperty.getType(), beanProperty);
		}
		return serializerProvider.findNullValueSerializer(beanProperty);
	}

}
