package com.ejianc.framework.skeleton.template;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.CamelAndUnderLineConverter;
import com.ejianc.framework.mq.common.MqMessage;
import com.ejianc.framework.skeleton.template.es.EsEntity;
import com.ejianc.framework.skeleton.template.es.IBaseEsQueryDao;
import com.ejianc.framework.skeleton.template.es.TableInfoTool;

/**
 * 集成mybatis plus的serviceImpl，各业务service都集成之。
 * @author guominga
 *
 * @param <M>
 * @param <T>
 */
public class BaseServiceImpl<M extends BaseCrudMapper<T>, T> extends ServiceImpl<M, T> implements IBaseService<T> {
	
	@Autowired
	private RabbitTemplate rabbitTemplate;
	@Autowired
	private IBaseEsQueryDao<T> baseEsQueryDao;
	
	@Value("${spring.application.name}")
	private String applicationName;

	/**
	 * 增加或修改对象
	 * 
	 * @param entity 实体对象
	 * @param syncEs 是否同步es
	 */
	@Transactional
	public boolean saveOrUpdate(T entity, boolean syncEs) {
		//增加数据
		boolean isSuccess = super.saveOrUpdate(entity);
		//插入mq
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entity.getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateOne");
			esEntity.setContent(JSON.toJSONString(entity));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}
	
	/**
	 * 批量增加或修改对象
	 * 
	 * @param entityList 实体对象list
	 * @param batchSize 一次批量增加或修改对象数量
	 * @param syncEs 是否同步es
	 */
	public boolean saveOrUpdateBatch(List<T> entityList, int batchSize, boolean syncEs) {
		boolean isSuccess = super.saveOrUpdateBatch(entityList, batchSize);
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entityList.get(0).getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateBatch");
			esEntity.setContent(JSON.toJSONString(entityList));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}
	
	/**
	 * 根据主键ID删除对象
	 * 
	 * @param id 主键id
	 * @param syncEs 是否同步es
	 */
	public boolean removeById(Serializable id, boolean syncEs) {
		boolean isSuccess = false;
		T bean = super.getById(id);
		if(bean != null) {
			isSuccess = super.removeById(id);
			if(isSuccess && syncEs) {
				TableInfo tableInfo = TableInfoTool.initTableInfo(bean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteById");
				esEntity.setContent(id.toString());
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}
	
	/**
	 * 根据map删除对象
	 * 
	 * @param columnMap 参数map
	 * @param syncEs 是否同步es
	 */
	public boolean removeByMap(Map<String, Object> columnMap, boolean syncEs) {
		boolean isSuccess = false;
		Collection<T> list = super.listByMap(columnMap);
		if(list != null && list.size() > 0) {
			isSuccess = super.removeByMap(columnMap);
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}
	
	/**
	 * 根据wrapper条件删除对象
	 * 
	 * @param wrapper 参数条件
	 * @param syncEs 是否同步es
	 */
	public boolean remove(Wrapper<T> wrapper, boolean syncEs){
		boolean isSuccess = false;
		List<T> list = super.list(wrapper);
		if(list != null && list.size() > 0) {
			isSuccess = super.remove(wrapper);
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}

	/**
	 * 根据主键Idlist删除满足条件的对象
	 * 
	 * @param idList 参数条件
	 * @param syncEs 是否同步es
	 */
	public boolean removeByIds(Collection<? extends Serializable> idList, boolean syncEs) {
		boolean isSuccess = false;
		Collection<T> list = super.listByIds(idList);
		if(list != null && list.size() > 0) {
			isSuccess = super.removeByIds(idList);
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}
	
	/**
	 * 根据wrapper修改对象
	 * 
	 * @param entity 实体对象
	 * @param updateWrapper 条件
	 * @param syncEs 是否同步es
	 */
	public boolean update(T entity, Wrapper<T> updateWrapper, boolean syncEs) {
		//增加数据
		boolean isSuccess = super.update(entity, updateWrapper);
		//插入mq
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entity.getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateOne");
			esEntity.setContent(JSON.toJSONString(entity));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}

	/**
	 * 根据查询对象查询list
	 * 
	 * @param queryParam 查询参数
	 * @param isEs 是否查询es
	 * @return
	 */
	@Override
	public List<T> queryList(QueryParam queryParam, boolean isEs) {
		List<T> list = new ArrayList<T>();
		if(isEs) { //查询es
			list = baseEsQueryDao.selectList(queryParam);
		}else{ //查询数据库
			QueryWrapper<T> queryWrapper = new QueryWrapper<>();
			Map<String, Parameter> params = queryParam.getParams();
			for(Map.Entry<String, Parameter> entry: params.entrySet()){
				Parameter param = entry.getValue();
				String type = param.getType();
				switch(type) {
					case QueryParam.EQ:
						queryWrapper.eq(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						break;
					case QueryParam.NE:
						queryWrapper.ne(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						break;
					case QueryParam.LIKE:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.like(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.NOT_LIKE:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.LIKE_LEFT:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.LIKE_RIGHT:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
				}
			}
			
			Map<String, String> orderMap = queryParam.getOrderMap();
			for(String key : orderMap.keySet()){
				String value = orderMap.get(key);
				if(QueryParam.ASC.equals(value.toLowerCase())){
					queryWrapper.orderByAsc(CamelAndUnderLineConverter.humpToLine(key));
				}else if(QueryParam.DESC.equals(value.toLowerCase())){
					queryWrapper.orderByDesc(CamelAndUnderLineConverter.humpToLine(key));
				}
			}
			list = this.list(queryWrapper);
		}
		return list;
	}

	/**
	 * 查询分页对象
	 * 
	 * @param queryParam 查询参数
	 * @param isEs 是否查询es
	 * @return
	 */
	@Override
	public IPage<T> queryPage(QueryParam queryParam, boolean isEs) {
		IPage<T> page = new Page<>();
		page.setCurrent(queryParam.getPageIndex());
		page.setSize(queryParam.getPageSize());

		if(isEs) {
			page = baseEsQueryDao.selectPage(page, queryParam);
		}else{
			QueryWrapper<T> queryWrapper = new QueryWrapper<>();
			Map<String, Parameter> params = queryParam.getParams();
			for(Map.Entry<String, Parameter> entry: params.entrySet()){
				Parameter param = entry.getValue();
				String type = param.getType();
				switch(type) {
					case QueryParam.EQ:
						queryWrapper.eq(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						break;
					case QueryParam.NE:
						queryWrapper.ne(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						break;
					case QueryParam.LIKE:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.like(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.NOT_LIKE:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.LIKE_LEFT:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
					case QueryParam.LIKE_RIGHT:
						if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
							queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
						}
						break;
				}
			}
			Map<String, String> orderMap = queryParam.getOrderMap();
			for(String key : orderMap.keySet()){
				String value = orderMap.get(key);
				if(QueryParam.ASC.equals(value.toLowerCase())){
					queryWrapper.orderByAsc(CamelAndUnderLineConverter.humpToLine(key));
				}else if(QueryParam.DESC.equals(value.toLowerCase())){
					queryWrapper.orderByDesc(CamelAndUnderLineConverter.humpToLine(key));
				}
			}
			page = this.page(page, queryWrapper);
		}
		return page;
	}
}
