package com.ejianc.framework.skeleton.template.es;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.xcontent.XContentType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.skeleton.refer.common.mapper.CommonReferMapper;

/**
 * Title: EsCudService
 * 
 * Description: es增删该操作
 * 
 * @author guominga
 * 
 * @date 2020年2月15日
 */
@Repository
public class BaseEsCudDao implements IBaseEsCudDao {

	@Autowired(required=false)
	private RestHighLevelClient client;
	@Autowired
	private CommonReferMapper commonReferMapper;

	/**
	 * 新增和修改操作
	 * 
	 * @param index: 对应的索引库 
	 * @param entity：索引实体
	 */
	public void insertOrUpdateOne(String index, EsEntity entity) {
		String content = entity.getContent();
		if(StringUtils.isNotBlank(content)) {
			JSONObject jsonObject = JSON.parseObject(content);
			IndexRequest request = new IndexRequest(index);
			request.id(jsonObject.getString("id"));
			request.source(jsonObject, XContentType.JSON);
			try {
				client.index(request, RequestOptions.DEFAULT);
				
				//修改对应表数据同步es状态成功
				commonReferMapper.updateDataStatus(index, jsonObject.getString("id"));
			} catch (Exception e) {
				throw new BusinessException("索引新增/修改失败，MSG：" + e.getMessage());
			}
		}
	}

	/**
	 * 批量插入或修改
	 * 
	 * @param index: 索引库 
	 * @param entity：索引实体
	 */
	public void insertOrUpdateBatch(String index, EsEntity entity) {
		String content = entity.getContent();
		BulkRequest request = new BulkRequest();
		if(StringUtils.isNotBlank(content)) {
			JSONArray jsonArray = JSON.parseArray(content);
			for(int i=0;i<jsonArray.size();i++) {
				JSONObject jsonObject = jsonArray.getJSONObject(i);
				request.add(new IndexRequest(index).id(jsonObject.getString("id")).source(jsonObject, XContentType.JSON));
			}
			try {
				client.bulk(request, RequestOptions.DEFAULT);
				
				//修改对应表数据同步es状态成功
				if(StringUtils.isNotBlank(content)) {
					JSONArray dataArray = JSON.parseArray(content);
					for(int i=0;i<dataArray.size();i++) {
						JSONObject dataObject = dataArray.getJSONObject(i);
						commonReferMapper.updateDataStatus(index, dataObject.getString("id"));
					}
				}
			} catch (Exception e) {
				throw new BusinessException("批量插入或修改索引失败，MSG：" + e.getMessage());
			}
		}
	}

	/**
	 * 批量删除
	 * 
	 * @param index:索引库 
	 * @param entity：索引实体
	 */
	public void deleteBatch(String index, EsEntity entity) {
		String content = entity.getContent();
		BulkRequest request = new BulkRequest();
		if(StringUtils.isNotBlank(content)) {
			List<String> ids = JSON.parseArray(content, String.class);
			for(String id:ids) {
				DeleteRequest deleteRequest = new DeleteRequest(index);
				deleteRequest.id(id);
				request.add(deleteRequest);
			}
		}
		try {
			client.bulk(request, RequestOptions.DEFAULT);
		} catch (Exception e) {
			throw new BusinessException("批量删除索引失败，MSG：" + e.getMessage());
		}
	}

	/**
	 * 根据主键ID删除
	 * 
	 * @param index：索引库 
	 * @param entity：索引实体
	 */
	public void deleteById(String index, EsEntity entity) {
		String content = entity.getContent();
		if(StringUtils.isNotBlank(content)) {
			DeleteRequest request = new DeleteRequest(index);
			request.id(content);
			try {
				client.delete(request, RequestOptions.DEFAULT);
			} catch (Exception e) {
				throw new BusinessException("根据主键ID删除失败，MSG：" + e.getMessage());
			}
		}
	}
}
