package com.ejianc.framework.skeleton.template.es;

import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;

/**
 * 
 * ES的dao接口偷懒，就直接复用了苞米豆的了, 只查询封装
 * 
 * @author guominga 2020-02-15
 */
@SuppressWarnings("unchecked")
@Repository
public class BaseEsQueryDao<T> implements IBaseEsQueryDao<T> {
	
	private final static Integer QUERY_TIMEOUT = 60;
	private final static String ASC = "asc";

	@Autowired(required=false)
	private RestHighLevelClient client;
	
	
	private Class<T> getTClass() {
		return (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
	}

	/**
     * 根据 参数map 条件，查询总记录数
     *
     * @param queryParam 查询参数
     */
	@Override
	public Long selectCount(QueryParam queryParam) {
		Long total = 0l;
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		SearchRequest searchRequest = new SearchRequest(index);
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			sourceBuilder.query(QueryBuilders.matchQuery(entry.getKey(), param.getValue()));
		}
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
		searchRequest.source(sourceBuilder);
        try {
        	total = client.search(searchRequest, RequestOptions.DEFAULT).getHits().getTotalHits();
            return total;
        } catch (Exception e) {
        	throw new BusinessException("根据 参数map 条件，查询总记录数索引失败，MSG："+e.getMessage());
        }
	}

	/**
     * 根据 parammap 条件，查询全部记录
     *
     * @param paramMap 参数map
     */
	@Override
	public List<T> selectList(QueryParam queryParam) {
		List<T> list = new ArrayList<>();
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		SearchRequest searchRequest = new SearchRequest(index);
		
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			sourceBuilder.query(QueryBuilders.matchQuery(entry.getKey(), param.getValue()));
		}
		
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			for(Map.Entry<String, String> entry : orderMap.entrySet()) {
				sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
			}
		}
		sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, getTClass()));
            }
            return list;
        } catch (Exception e) {
        	throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e.getMessage());
        }
	}

	/**
     * 根据 map 条件，查询全部记录（并翻页）
     *
     * @param page         分页查询条件（可以为 RowBounds.DEFAULT）
     * @param queryParam  查询参数
     */
	@Override
	public IPage<T> selectPage(IPage<T> page, QueryParam queryParam) {
		//根据class对象获取该实体对应的表名
		TableInfo tableInfo = TableInfoTool.initTableInfo(getTClass());
		String index = tableInfo.getTableName();
		
		//查询当前页记录
		SearchRequest searchRequest = new SearchRequest(index);
		
		SearchSourceBuilder sourceBuilder = new SearchSourceBuilder();
		
		//查询参数
		Map<String, Parameter> paramMap = queryParam.getParams();
		for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			sourceBuilder.query(QueryBuilders.matchQuery(entry.getKey(), param.getValue()));
		}
		
		//排序
		Map<String, String> orderMap = queryParam.getOrderMap();
		if(orderMap != null && !orderMap.isEmpty()) {
			for(Map.Entry<String, String> entry : orderMap.entrySet()) {
				sourceBuilder.sort(new FieldSortBuilder(entry.getKey()).order(ASC.equals(entry.getValue().toLowerCase())?SortOrder.ASC:SortOrder.DESC));
			}
		}
		
		sourceBuilder.from((int)page.getCurrent());
        sourceBuilder.size((int)page.getSize());
        sourceBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
        searchRequest.source(sourceBuilder);
        try {
        	List<T> list = new ArrayList<>();
            SearchResponse response = client.search(searchRequest, RequestOptions.DEFAULT);
            SearchHits hits = response.getHits();
            for (SearchHit hit : hits) {
                String sourceAsString = hit.getSourceAsString();
                list.add(JSON.parseObject(sourceAsString, getTClass()));
            }
            page.setRecords(list);
        } catch (Exception e) {
        	throw new BusinessException("根据 parammap 条件，查询全部记录索引失败，MSG："+e.getMessage());
        }

        //查询总数
        SearchRequest countRequest = new SearchRequest(index);
        SearchSourceBuilder countBuilder = new SearchSourceBuilder();
        //查询参数
        for(Map.Entry<String, Parameter> entry: paramMap.entrySet()){
			Parameter param = entry.getValue();
			countBuilder.query(QueryBuilders.matchQuery(entry.getKey(), param.getValue()));
		}
		countBuilder.timeout(new TimeValue(QUERY_TIMEOUT, TimeUnit.SECONDS)); //设置超时时间
		countRequest.source(countBuilder);
        try {
        	SearchResponse response = client.search(countRequest, RequestOptions.DEFAULT);
            page.setTotal(response.getHits().getTotalHits());
        } catch (Exception e) {
        	throw new BusinessException("根据 参数map 条件，查询总记录数索引失败，MSG："+e.getMessage());
        }
        return page;
	}
}
