package com.ejianc.framework.skeleton.refer.serialize;

import java.io.IOException;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.skeleton.refer.annotation.ReferSerialTransfer;
import com.ejianc.framework.skeleton.refer.util.ReferObjectUtil;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.introspect.AnnotatedMethod;
import com.fasterxml.jackson.databind.ser.ContextualSerializer;

/**
 * 将String转换成json{id:xx,code:xx,name:xx}
 * <note>将String{id,id}转换成JsonArray[{id:xx,code:xx,name:xx},{id:xx,code:xx,name:
 * xx}]</note>
 * 
 * @author guominga 2020年02月13日
 *
 */

public class ReferSerializer extends JsonSerializer<Long> implements ContextualSerializer {
	private final static org.slf4j.Logger Log = LoggerFactory.getLogger(ReferSerializer.class);

	/**
	 * 实体引用对应code
	 */
	private String referCode;

	private static final String DEFDOC_REFER_CODE = "defdoc";

	public ReferSerializer(String referCode) {
		super();
		this.referCode = referCode;
	}

	// 必须要保留无参构造方法
	public ReferSerializer() {
		super();
	}

	@Override
	public void serialize(Long value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
		JSONObject jsonObject = null;
		try {
			// 单个参照code
			jsonObject = ReferObjectUtil.getReferEntityValue(value, referCode);
		} catch (Exception e) {
			Log.error("referCode:" + referCode + " id：" + value + "转换出错，--" + e.getMessage(), e);
		}
		if (jsonObject != null) {
			jgen.writeObject(jsonObject);
		} else {
			JSONObject result = new JSONObject();
			result.put("id", value);
			result.put("code", value);
			result.put("name", value);
			jgen.writeObject(result);
		}

	}

	@Override
	public JsonSerializer<?> createContextual(SerializerProvider serializerProvider, BeanProperty beanProperty) throws JsonMappingException {
		if (beanProperty != null) { // 为空直接跳过
			if (Objects.equals(beanProperty.getType().getRawClass(), Long.class)) { // 非Long类直接跳过
				AnnotatedMethod method = (AnnotatedMethod) beanProperty.getMember();
				ReferSerialTransfer referSerialTransfer = method.getAnnotated().getDeclaredAnnotation(ReferSerialTransfer.class);
				if (referSerialTransfer != null) { // 如果能得到注解，就将注解的value传入ReferSerializer
					String referCode = referSerialTransfer.referCode();
					if (StringUtils.isNoneBlank(referCode)) {
						if (DEFDOC_REFER_CODE.equalsIgnoreCase(referCode) || referCode.toLowerCase().startsWith(DEFDOC_REFER_CODE)) {
							referCode = DEFDOC_REFER_CODE;
						}
					}
					return new ReferSerializer(referCode);
				}
			}
			return serializerProvider.findValueSerializer(beanProperty.getType(), beanProperty);
		}
		return serializerProvider.findNullValueSerializer(beanProperty);
	}

}
