package com.ejianc.framework.skeleton.advice;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.exception.ErrorEnum;
import com.ejianc.framework.core.kit.base.ExceptionUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.fasterxml.jackson.databind.exc.UnrecognizedPropertyException;
/**
 * 通用 Api Controller 全局异常处理
 * 
 * Title: GlobalExceptionHandler Description:
 * 
 * @author guominga
 * @date 2019年3月7日
 */
@RestControllerAdvice
public class GlobalExceptionHandler {
	private Logger logger = LoggerFactory.getLogger(GlobalExceptionHandler.class);
	private static final String SERV_ERROR_MSG = "网络错误，请稍后重试！";
	
	CommonResponse<String> genResponse(String msg, String data){
		CommonResponse<String> ret = CommonResponse.error(ErrorEnum.STATUS_PARAM_ERROR.getErrorCode(),
				ErrorEnum.STATUS_PARAM_ERROR.getErrorMsg()
        		);
        ret.setData(data);
        return ret;
	}
	
	
	@ExceptionHandler(MissingServletRequestParameterException.class)
    public CommonResponse<String> resolveMissingServletRequestParameterException(MissingServletRequestParameterException ex){
    	logger.error(ExceptionUtil.toStringWithRootCause(ex));
    	logger.error("参数{}必须传递",ex.getParameterName() );

        String msg = ex.getMessage();
        return genResponse("参数验证错误", msg);
    }
	
	@ExceptionHandler(UnrecognizedPropertyException.class)
    public CommonResponse<String> resolveUnrecognizedPropertyException(UnrecognizedPropertyException ex){
    	logger.error(ExceptionUtil.toStringWithRootCause(ex));
    	logger.error("参数{}无法解析",ex.getPropertyName() );

        String msg = ex.getMessage();
        return genResponse("参数验证错误", msg);
    }
	
	
	
	@ExceptionHandler(ConstraintViolationException.class)
    public  CommonResponse<String> resolveConstraintViolationException(ConstraintViolationException ex){
		logger.error(ExceptionUtil.toStringWithRootCause(ex));
		String msg = ex.getMessage();;
        Set<ConstraintViolation<?>> constraintViolations = ex.getConstraintViolations();
        if(!CollectionUtils.isEmpty(constraintViolations)){
            StringBuilder msgBuilder = new StringBuilder();
            for(ConstraintViolation<?> constraintViolation :constraintViolations){
                msgBuilder.append(constraintViolation.getMessage()).append(",");
            }
            String errorMessage = msgBuilder.toString();
            if(errorMessage.length()>1){
                errorMessage = errorMessage.substring(0,errorMessage.length()-1);
            }
            msg = errorMessage;
        }
        
        return genResponse("参数验证错误", msg);
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    public CommonResponse<String> resolveMethodArgumentNotValidException(MethodArgumentNotValidException ex){
    	logger.error(ExceptionUtil.toStringWithRootCause(ex));
    	BindingResult result = ex.getBindingResult();
    	List<ObjectError>  objectErrors = result.getAllErrors();
        String msg = ex.getMessage();
        if(!CollectionUtils.isEmpty(objectErrors)) {
            StringBuilder msgBuilder = new StringBuilder();
            msgBuilder.append(result.getObjectName()).append(":");    
            for (ObjectError objectError : objectErrors) {
                msgBuilder.append(objectError.getDefaultMessage()).append(",");
            }
            
            String errorMessage = msgBuilder.toString();
            if (errorMessage.length() > 1) {
                errorMessage = errorMessage.substring(0, errorMessage.length() - 1);
            }
            msg = errorMessage;
        }
       
        return genResponse("参数验证错误", msg);
    }
    
	/**
	 * <p>
	 * 自定义 REST 业务异常
	 * <p>
	 *
	 * @param e
	 *            异常类型
	 * @return
	 */
	@ExceptionHandler(value = Exception.class)
	public CommonResponse<ErrorEnum> exceptionHandler(HttpServletRequest request, Exception e) {
		
		Throwable throwable = getBusinessException(e);
        String errorMsg = null;
		if (!Objects.isNull(throwable)) {
			Integer errorCode = ((BusinessException) throwable).getErrorCode();
            errorMsg = ((BusinessException) throwable).getErrorMsg();
			if (errorCode != null && StringUtils.isNotBlank(ErrorEnum.getErrorMsgByCode(errorCode))) {
				if(StringUtils.isBlank(errorMsg)) {
					errorMsg = ErrorEnum.getErrorMsgByCode(errorCode);
				}
				return CommonResponse.error(errorCode, StringUtils.isNotBlank(errorMsg) ? errorMsg : SERV_ERROR_MSG);
			}
		}
		return CommonResponse.error(StringUtils.isNotBlank(errorMsg) ? errorMsg : SERV_ERROR_MSG);
	}

	/**
     * 若有异常进行嵌套，打印出每个异常的堆栈信息，若包含自定义异常，返回最内部的BusinessException异常。
     * @param e
     * @return
     */
    private Throwable getBusinessException(Throwable e) {
        if (e == null) {
            return null;
        } else if (e instanceof BusinessException) {
            e.printStackTrace();
            logger.error("方法异常[{}]",e.getCause());
            Throwable temp = getBusinessException(e.getCause());
            if (temp == null) {
                return e;
            } else {
                return temp;
            }
        } else if(null != e.getCause()) {
        	logger.error("方法异常[{}]", e.getCause());
            e.printStackTrace();
            return getBusinessException(e.getCause());
        } else {
            logger.error("方法异常[{}]", e.getMessage());
            e.printStackTrace();
            return new BusinessException(null != e.getMessage() ? e.getMessage() : "程序运行异常：", e);
        }
    }
}
