package com.ejianc.framework.skeleton.template;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.ejianc.framework.core.response.ComplexParam;
import com.fasterxml.jackson.databind.ser.Serializers;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.core.toolkit.ReflectionKit;
import com.baomidou.mybatisplus.core.toolkit.TableInfoHelper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.CamelAndUnderLineConverter;
import com.ejianc.framework.mq.common.MqMessage;
import com.ejianc.framework.skeleton.refer.util.ContextUtil;
import com.ejianc.framework.skeleton.template.annotation.SubEntity;
import com.ejianc.framework.skeleton.template.es.EsEntity;
import com.ejianc.framework.skeleton.template.es.IBaseEsQueryDao;
import com.ejianc.framework.skeleton.template.es.TableInfoTool;

/**
 * 集成mybatis plus的serviceImpl，各业务service都集成之。
 * @author guominga
 *
 * @param <M>
 * @param <T>
 */
public class BaseServiceImpl<M extends BaseCrudMapper<T>, T> extends ServiceImpl<M, T> implements IBaseService<T> {
	
	@Autowired
	private RabbitTemplate rabbitTemplate;
	@Autowired
	private IBaseEsQueryDao<T> baseEsQueryDao;

	@Value("${spring.application.name}")
	private String applicationName;

	/**
	 * 增加或修改对象
	 * 
	 * @param entity 实体对象
	 * @param syncEs 是否同步es
	 */
	@Transactional
	public boolean saveOrUpdate(T entity, boolean syncEs) {
		//增加数据--主表
		boolean isSuccess = super.saveOrUpdate(entity);
		//处理子表
		if(isSuccess){
			Class<?> mainClass = entity.getClass();
			TableInfo tableInfo = TableInfoHelper.getTableInfo(mainClass);
			Object pid = ReflectionKit.getMethodValue(mainClass, entity, tableInfo.getKeyProperty());
			List<Field> subFieldsList = new ArrayList<>();
			Field[] fields = mainClass.getDeclaredFields();
			/** 查找字表字段 可能有多个子表*/
			for (Field field : fields) {
				if(field.isAnnotationPresent(SubEntity.class)){
					subFieldsList.add(field);
				}
			}
			if(!ListUtil.isEmpty(subFieldsList)){
				T finalEntity = entity;
				subFieldsList.forEach(field -> {
					/** 字表字段名 */
					String fieldName = field.getName();
					try {
						/** 获取子表get方法*/
						Method m =  mainClass.getDeclaredMethod("get"+fieldName.substring(0,1).toUpperCase()+fieldName.substring(1),  null);
						/** 获取子表数据 */
						List<T> subList = (List<T>)m.invoke(finalEntity, (Object[]) null);
						if(!ListUtil.isEmpty(subList)){
							SubEntity subEntity = field.getAnnotation(SubEntity.class);
							List<T> delSubList = new ArrayList<>();
							List<Object> delSubListIds = new ArrayList<>();
							/** 将主表id赋值给子表*/
							subList.forEach(s->{
								String p = subEntity.pidName();
								if(StringUtils.isBlank(p)){
									throw new BusinessException("字段【"+fieldName+"】注解SubEntity未设置pidName！");
								}
								Class<?> subClass = s.getClass();
								try {
									Method subPid = subClass.getDeclaredMethod("set"+p.substring(0,1).toUpperCase()+p.substring(1), Long.class);
									subPid.invoke(s,pid);
									Method getRowState = subClass.getSuperclass().getDeclaredMethod("getRowState",  null);
									String o = (String)getRowState.invoke(s, (Object[]) null);
									/** 处理子表删除 */
									if(StringUtils.isNotBlank(o)&&"del".equals(o)){
										Method setDr = subClass.getSuperclass().getDeclaredMethod("setDr", Integer.class);
										setDr.invoke(s,1);
										TableInfo subtableInfo = TableInfoHelper.getTableInfo(subClass);
										Object idVal = ReflectionKit.getMethodValue(subClass, s, subtableInfo.getKeyProperty());
										delSubListIds.add(idVal);
										delSubList.add(s);
									}else {
										Method setRowState = subClass.getSuperclass().getDeclaredMethod("setRowState",  String.class);
										setRowState.invoke(s,(String)null);
									}
								} catch (NoSuchMethodException e) {
									throw new BusinessException("类【"+subClass+"】的字段【"+p+"】未设置get方法！");
								} catch (IllegalAccessException e) {
									throw new BusinessException("类【"+subClass+"】的字段【"+p+"】的get方法需要设置成public属性！");
								} catch (InvocationTargetException e) {
									throw new BusinessException("类【"+subClass+"】的字段【"+p+"】的get方法调用失败！");
								}
							});
							/** 查找子表实现服务类 */
							String serviceName = subEntity.serviceName();
							if(StringUtils.isBlank(serviceName)){
								throw new BusinessException("子表字段【"+fieldName+"】注解SubEntity未设置子表实现类服务名！");
							}
							BaseServiceImpl subService = ContextUtil.getBean(serviceName, BaseServiceImpl.class);
							if(subService==null){
								throw new BusinessException("字段【"+fieldName+"】子表没有实现类！");
							}
							/** 批量新增子表数据 */
							subList.removeAll(delSubList);
							if(!ListUtil.isEmpty(subList)){
								subService.saveOrUpdateBatch(subList);
							}
							if(!ListUtil.isEmpty(delSubListIds)){
								subService.removeByIds(delSubListIds);
							}
						}
					} catch (NoSuchMethodException e) {
						throw new BusinessException("子表字段【"+fieldName+"】未设置get方法！");
					} catch (IllegalAccessException e) {
						throw new BusinessException("子表字段【"+fieldName+"】的get方法需要设置成public属性！");
					} catch (InvocationTargetException e) {
						throw new BusinessException("子表字段【"+fieldName+"】的get方法调用失败！");
					}
				});
			}
		}else {
			throw new BusinessException("数据已被您的同事修改，请刷新页面获取最新数据再保存！");
		}
		//插入mq
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entity.getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateOne");
			esEntity.setContent(JSON.toJSONString(entity));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}

	/**
	 * 批量增加或修改对象
	 * 
	 * @param entityList 实体对象list
	 * @param batchSize 一次批量增加或修改对象数量
	 * @param syncEs 是否同步es
	 */
	public boolean saveOrUpdateBatch(List<T> entityList, int batchSize, boolean syncEs) {
		boolean isSuccess = super.saveOrUpdateBatch(entityList, batchSize);
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entityList.get(0).getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateBatch");
			esEntity.setContent(JSON.toJSONString(entityList));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}

	/**
	 * 根据主键ID查询对象
	 *
	 * @param id 主键id
	 */
	@Override
	public T selectById(Serializable id) {
		T entity = baseMapper.selectById(id);
		Class<?> mainClass = entity.getClass();
		Field[] fields = mainClass.getDeclaredFields();
		/** 查找字表字段 可能有多个子表*/
		for (Field field : fields) {
			if(field.isAnnotationPresent(SubEntity.class)){
				SubEntity subEntity = field.getAnnotation(SubEntity.class);
				String serviceName = subEntity.serviceName();
				String fieldName = field.getName();
				if(StringUtils.isBlank(serviceName)){
					throw new BusinessException("子表字段【"+fieldName+"】注解SubEntity未设置子表实现类服务名！");
				}
				BaseServiceImpl subService = ContextUtil.getBean(serviceName, BaseServiceImpl.class);
				if(subService==null){
					throw new BusinessException("字段【"+fieldName+"】子表没有实现类！");
				}
				QueryParam queryParam = new QueryParam();
				queryParam.getParams().put(subEntity.pidName(), new Parameter(QueryParam.EQ, id));
				List<T> subList = subService.queryList(queryParam,false);
				try {
					Class[] cArg = new Class[1];
					cArg[0] = field.getType();
					Method setSubList = mainClass.getDeclaredMethod("set"+fieldName.substring(0,1).toUpperCase()+fieldName.substring(1),cArg);
					setSubList.invoke(entity,subList);
				} catch (NoSuchMethodException e) {
					throw new BusinessException("字段【"+fieldName+"】未定义set方法！");
				} catch (IllegalAccessException e) {
					throw new BusinessException("字段【"+fieldName+"】set方法需要是public属性！");
				} catch (InvocationTargetException e) {
					throw new BusinessException("字段【"+fieldName+"】set方法执行出错！");
				}

			}
		}
		return entity;
	}

	/**
	 * 根据主键ID删除对象
	 * 
	 * @param id 主键id
	 * @param syncEs 是否同步es
	 */
	public boolean removeById(Serializable id, boolean syncEs) {
		boolean isSuccess = false;
		T entity = this.selectById(id);
		if(entity != null) {
			isSuccess = super.removeById(id);
			if(isSuccess){
				Class<?> mainClass = entity.getClass();
				Field[] fields = mainClass.getDeclaredFields();
				/** 查找字表字段 可能有多个子表*/
				for (Field field : fields) {
					if(field.isAnnotationPresent(SubEntity.class)){
						SubEntity subEntity = field.getAnnotation(SubEntity.class);
						String serviceName = subEntity.serviceName();
						String fieldName = field.getName();
						if(StringUtils.isBlank(serviceName)){
							throw new BusinessException("子表字段【"+fieldName+"】注解SubEntity未设置子表实现类服务名！");
						}
						BaseServiceImpl subService = ContextUtil.getBean(serviceName, BaseServiceImpl.class);
						if(subService==null){
							throw new BusinessException("字段【"+fieldName+"】子表没有实现类！");
						}
						try {
							Method getSubList = mainClass.getDeclaredMethod("get"+fieldName.substring(0,1).toUpperCase()+fieldName.substring(1),  null);
							List<BaseEntity> subs =  (List<BaseEntity>)getSubList.invoke(entity, (Object[]) null);
							if(!ListUtil.isEmpty(subs)){
								subService.removeByIds(subs.stream().map(BaseEntity::getId).collect(Collectors.toList()),false);
							}
						} catch (NoSuchMethodException e) {
							throw new BusinessException("字段【"+fieldName+"】未定义set方法！");
						} catch (IllegalAccessException e) {
							throw new BusinessException("字段【"+fieldName+"】set方法需要是public属性！");
						} catch (InvocationTargetException e) {
							throw new BusinessException("字段【"+fieldName+"】set方法执行出错！");
						}

					}
				}
			}
			if(isSuccess && syncEs) {
				TableInfo tableInfo = TableInfoTool.initTableInfo(entity.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteById");
				esEntity.setContent(id.toString());
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}
	
	/**
	 * 根据map删除对象
	 * 
	 * @param columnMap 参数map
	 * @param syncEs 是否同步es
	 */
	public boolean removeByMap(Map<String, Object> columnMap, boolean syncEs) {
		boolean isSuccess = false;
		Collection<T> list = super.listByMap(columnMap);
		if(list != null && list.size() > 0) {
			isSuccess = super.removeByMap(columnMap);
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}
	
	/**
	 * 根据wrapper条件删除对象
	 * 
	 * @param wrapper 参数条件
	 * @param syncEs 是否同步es
	 */
	public boolean remove(Wrapper<T> wrapper, boolean syncEs){
		boolean isSuccess = false;
		List<T> list = super.list(wrapper);
		if(list != null && list.size() > 0) {
			isSuccess = super.remove(wrapper);
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}
		return isSuccess;
	}

	/**
	 * 根据主键Idlist删除满足条件的对象
	 * 
	 * @param idList 参数条件
	 * @param syncEs 是否同步es
	 */
	public boolean removeByIds(Collection<? extends Serializable> idList, boolean syncEs) {
		boolean isSuccess = false;
		for(Serializable id : idList){
			isSuccess = this.removeById(id,syncEs);
			if(!isSuccess){
				throw new BusinessException("删除失败");
			}
		}
		/*
		Collection<T> list = super.listByIds(idList);
		
		if(list != null && list.size() > 0) {
			isSuccess = super.removeByIds(idList);
			if(isSuccess){
				list.forEach(entity->{
					Class<?> mainClass = entity.getClass();
					Field[] fields = mainClass.getDeclaredFields();
					*//** 查找字表字段 可能有多个子表*//*
					for (Field field : fields) {
						if(field.isAnnotationPresent(SubEntity.class)){
							SubEntity subEntity = field.getAnnotation(SubEntity.class);
							String serviceName = subEntity.serviceName();
							String fieldName = field.getName();
							if(StringUtils.isBlank(serviceName)){
								throw new BusinessException("子表字段【"+fieldName+"】注解SubEntity未设置子表实现类服务名！");
							}
							BaseServiceImpl subService = ContextUtil.getBean(serviceName, BaseServiceImpl.class);
							if(subService==null){
								throw new BusinessException("字段【"+fieldName+"】子表没有实现类！");
							}
							try {
								Method getSubList = mainClass.getDeclaredMethod("get"+fieldName.substring(0,1).toUpperCase()+fieldName.substring(1),  null);
								List<BaseEntity> subs =  (List<BaseEntity>)getSubList.invoke(entity, (Object[]) null);
								if(!ListUtil.isEmpty(subs)){
									subService.removeByIds(subs.stream().map(BaseEntity::getId).collect(Collectors.toList()),false);
								}
							} catch (NoSuchMethodException e) {
								throw new BusinessException("字段【"+fieldName+"】未定义set方法！");
							} catch (IllegalAccessException e) {
								throw new BusinessException("字段【"+fieldName+"】set方法需要是public属性！");
							} catch (InvocationTargetException e) {
								throw new BusinessException("字段【"+fieldName+"】set方法执行出错！");
							}

						}
					}
				});
			}
			
			if(isSuccess && syncEs) {
				//循环Collection
				T tableBean = null;
				List<Long> ids = new ArrayList<Long>();
				for(T bean:list) {
					tableBean = bean;
					
					BaseEntity baseEntity = (BaseEntity) bean;
					ids.add(baseEntity.getId());
				}
				
				TableInfo tableInfo = TableInfoTool.initTableInfo(tableBean.getClass());
				String indexName = tableInfo.getTableName();
				EsEntity esEntity = new EsEntity();
				
				esEntity.setIndexName(indexName);
				esEntity.setType("deleteBatch");
				esEntity.setContent(JSON.toJSONString(ids));
				//插入mq
				MqMessage mqMessage = new MqMessage();
				mqMessage.setBody(esEntity);
				rabbitTemplate.convertAndSend(applicationName, mqMessage);
			}
		}*/
		return isSuccess;
	}
	
	/**
	 * 根据wrapper修改对象
	 * 
	 * @param entity 实体对象
	 * @param updateWrapper 条件
	 * @param syncEs 是否同步es
	 */
	public boolean update(T entity, Wrapper<T> updateWrapper, boolean syncEs) {
		//增加数据
		boolean isSuccess = super.update(entity, updateWrapper);
		//插入mq
		if(isSuccess && syncEs) {
			TableInfo tableInfo = TableInfoTool.initTableInfo(entity.getClass());
			String indexName = tableInfo.getTableName();
			EsEntity esEntity = new EsEntity();
			
			esEntity.setIndexName(indexName);
			esEntity.setType("insertOrUpdateOne");
			esEntity.setContent(JSON.toJSONString(entity));
			//插入mq
			MqMessage mqMessage = new MqMessage();
			mqMessage.setBody(esEntity);
			rabbitTemplate.convertAndSend(applicationName, mqMessage);
		}
		return isSuccess;
	}

	/**
	 * 根据查询对象查询list
	 * 
	 * @param queryParam 查询参数
	 * @param isEs 是否查询es
	 * @return
	 */
	@Override
	public List<T> queryList(QueryParam queryParam, boolean isEs) {
		List<T> list;
		if(isEs) { //查询es
			list = baseEsQueryDao.selectList(queryParam);
		}else{ //查询数据库
			QueryWrapper<T> queryWrapper = new QueryWrapper<>();
			Map<String, Parameter> params = queryParam.getParams();
			/** 模糊查询配置的字段 */
			if(StringUtils.isNotBlank(queryParam.getSearchText()) && !ListUtil.isEmpty(queryParam.getFuzzyFields())){
				String searchText = queryParam.getSearchText();
				List<String> fuzzyFields = queryParam.getFuzzyFields();
				if(queryParam.getFuzzyFields().size()==1){
					queryWrapper.like(CamelAndUnderLineConverter.humpToLine(queryParam.getFuzzyFields().get(0)), searchText);
				}else {
					queryWrapper.and(wrapper ->{
						for (int i = 0; i < fuzzyFields.size(); i++) {
							wrapper.like(CamelAndUnderLineConverter.humpToLine(queryParam.getFuzzyFields().get(i)),searchText);
							if(i!=fuzzyFields.size()-1){
								wrapper.or();
							}
						}
						return wrapper;
					});
				}
			}
			/** 基础字段查询*/
			setParam(queryWrapper,params);

			/** 复杂查询 */
			List<ComplexParam> complexParams = queryParam.getComplexParams();
			if(!ListUtil.isEmpty(complexParams)){
				complexParams.forEach(c->{
					if(ComplexParam.AND.equals(c.getLogic())){
						queryWrapper.and(w-> setWrapper(w,c));
					}else {
						queryWrapper.or(w-> setWrapper(w,c));
					}
				});
			}

			/** 设置整体排序方式 */
			Map<String, String> orderMap = queryParam.getOrderMap();
			for(String key : orderMap.keySet()){
				String value = orderMap.get(key);
				if(QueryParam.ASC.equals(value.toLowerCase())){
					queryWrapper.orderByAsc(CamelAndUnderLineConverter.humpToLine(key));
				}else if(QueryParam.DESC.equals(value.toLowerCase())){
					queryWrapper.orderByDesc(CamelAndUnderLineConverter.humpToLine(key));
				}
			}
			list = this.list(queryWrapper);
		}
		return list;
	}

	private static <T> QueryWrapper<T> setWrapper(QueryWrapper<T> w,ComplexParam c){
		/** 基础字段查询*/
		if(c.getParams()!=null && !c.getParams().isEmpty()){
			setParam(w,c.getParams());
		}
		List<ComplexParam> complexParams = c.getComplexParams();
		if(!ListUtil.isEmpty(complexParams)){
			complexParams.forEach(complex->{
				if(ComplexParam.AND.equals(complex.getLogic())){
					w.and(wrapper-> setWrapper(wrapper,complex));
				}else {
					w.or(wrapper-> setWrapper(wrapper,complex));
				}
			});
		}
		return w;
	}

	/**
	 * 查询分页对象
	 * 
	 * @param queryParam 查询参数
	 * @param isEs 是否查询es
	 * @return
	 */
	@Override
	public IPage<T> queryPage(QueryParam queryParam, boolean isEs) {
		IPage<T> page = new Page<>();
		page.setCurrent(queryParam.getPageIndex());
		page.setSize(queryParam.getPageSize());

		if(isEs) {
			page = baseEsQueryDao.selectPage(page, queryParam);
		}else{
			page = this.page(page, changeToQueryWrapper(queryParam));
		}
		return page;
	}


	/** 基础字段查询*/
	private static <T> void setParam(QueryWrapper<T> queryWrapper, Map<String, Parameter> params){
		for(Map.Entry<String, Parameter> entry: params.entrySet()){
			Parameter param = entry.getValue();
			String type = param.getType();
			switch(type) {
				case QueryParam.EQ:
					if(null == param.getValue()) {
						queryWrapper.isNull(CamelAndUnderLineConverter.humpToLine(entry.getKey()));
					} else {
						queryWrapper.eq(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.NE:
					if(null == param.getValue()) {
						queryWrapper.isNotNull(CamelAndUnderLineConverter.humpToLine(entry.getKey()));
					} else {
						queryWrapper.ne(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.IN:
					if(param.getValue() instanceof List) {
						queryWrapper.in(CamelAndUnderLineConverter.humpToLine(entry.getKey()), (List<?>)param.getValue());
					}else if(param.getValue() instanceof String) {
						String[] paramArr = param.getValue().toString().split(",");
						queryWrapper.in(CamelAndUnderLineConverter.humpToLine(entry.getKey()), paramArr);

					}
					break;
				case QueryParam.NOT_IN:
					if(param.getValue() instanceof List) {
						queryWrapper.notIn(CamelAndUnderLineConverter.humpToLine(entry.getKey()), (List<?>)param.getValue());
					}else if(param.getValue() instanceof String) {
						String[] paramArr = param.getValue().toString().split(",");
						queryWrapper.notIn(CamelAndUnderLineConverter.humpToLine(entry.getKey()), paramArr);

					}
					break;
				case QueryParam.LIKE:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.like(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.NOT_LIKE:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.notLike(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.LIKE_LEFT:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.likeLeft(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.LIKE_RIGHT:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.likeRight(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.BETWEEN:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						String[] paramArr = param.getValue().toString().split(",");
						queryWrapper.between(CamelAndUnderLineConverter.humpToLine(entry.getKey()), paramArr[0],paramArr[1]);
					}
					break;
				case QueryParam.LT:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.lt(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.LE:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.le(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.GT:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.gt(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
				case QueryParam.GE:
					if(param.getValue() != null && StringUtils.isNotBlank(param.getValue().toString())) {
						queryWrapper.ge(CamelAndUnderLineConverter.humpToLine(entry.getKey()), param.getValue());
					}
					break;
			}
		}
	}


	public static <T> QueryWrapper<T> changeToQueryWrapper(QueryParam queryParam) {
		QueryWrapper<T> queryWrapper = new QueryWrapper<>();
		Map<String, Parameter> params = queryParam.getParams();
		/** 模糊查询配置的字段 */
		if(StringUtils.isNotBlank(queryParam.getSearchText()) && !ListUtil.isEmpty(queryParam.getFuzzyFields())){
			String searchText = queryParam.getSearchText();
			List<String> fuzzyFields = queryParam.getFuzzyFields();
			if(queryParam.getFuzzyFields().size()==1){
				queryWrapper.like(CamelAndUnderLineConverter.humpToLine(queryParam.getFuzzyFields().get(0)), searchText);
			}else {
				queryWrapper.and(wrapper ->{
					for (int i = 0; i < fuzzyFields.size(); i++) {
						wrapper.like(CamelAndUnderLineConverter.humpToLine(queryParam.getFuzzyFields().get(i)),searchText);
						if(i!=fuzzyFields.size()-1){
							wrapper.or();
						}
					}
					return wrapper;
				});
			}
		}
		/** 基础字段查询*/
		setParam(queryWrapper,params);

		/** 复杂查询 */
		List<ComplexParam> complexParams = queryParam.getComplexParams();
		if(!ListUtil.isEmpty(complexParams)){
			complexParams.forEach(c->{
				if(ComplexParam.AND.equals(c.getLogic())){
					queryWrapper.and(w-> setWrapper(w,c));
				}else {
					queryWrapper.or(w-> setWrapper(w,c));
				}
			});
		}
		/** 设置整体排序方式 */
		LinkedHashMap<String,String> orderMap = queryParam.getOrderMap();
		for(String key : orderMap.keySet()){
			String value = orderMap.get(key);
			if(QueryParam.ASC.equals(value.toLowerCase())){
				queryWrapper.orderByAsc(CamelAndUnderLineConverter.humpToLine(key));
			}else if(QueryParam.DESC.equals(value.toLowerCase())){
				queryWrapper.orderByDesc(CamelAndUnderLineConverter.humpToLine(key));
			}
		}
		return queryWrapper;
	}
}
