package com.ejianc.framework.skeleton;

import java.sql.SQLException;
import java.util.Map;
import java.util.Objects;

import javax.sql.DataSource;

import org.springframework.aop.aspectj.AspectJExpressionPointcut;
import org.springframework.aop.support.DefaultPointcutAdvisor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.interceptor.DefaultTransactionAttribute;
import org.springframework.transaction.interceptor.NameMatchTransactionAttributeSource;
import org.springframework.transaction.interceptor.TransactionInterceptor;

import com.ejianc.framework.skeleton.refer.util.ContextUtil;
import com.ejianc.framework.skeleton.util.SeataUtils;
import com.google.common.collect.Maps;

import io.shardingjdbc.core.api.MasterSlaveDataSourceFactory;

@Configuration
@EnableConfigurationProperties(value={ShardingMasterSlaveConfig.class,TxAttrConfig.class})
public class ShardingDataSourceConfig {

	@Value("${eureka.client.serviceUrl.defaultZone:#{null}}")
	private String serviceUrl;
	
	@Autowired(required = false)
	private ShardingMasterSlaveConfig shardingMasterSlaveConfig;
	
	@Autowired(required = false)
	private TxAttrConfig txAttrConfig;
	
	@Bean("contextUtil")
	public ContextUtil contextUtil() {
		return new ContextUtil();
	}

	@Value("${database.dbtype:mysql}")
	private String dbType;

	@Bean("dataSource")
	public DataSource masterSlaveDataSource() {
		System.out.println("使用mysql数据库----");
		DataSource dataSource = null;
		if(shardingMasterSlaveConfig != null) {
			Map<String, DataSource> dataSourceMap = Maps.newHashMap();
			dataSourceMap.putAll(shardingMasterSlaveConfig.getDataSources());
			try {
				SeataUtils.getInstance().setServiceUrl(serviceUrl);
				SeataUtils.getInstance().setDataSource(shardingMasterSlaveConfig.getDataSources().get("ds_master"));
				if(Objects.equals("dmdb",dbType)){
					dataSource = shardingMasterSlaveConfig.getDataSources().get("ds_master");
				} else {
					dataSource = MasterSlaveDataSourceFactory.createDataSource(dataSourceMap, shardingMasterSlaveConfig.getMasterSlaveRule(), Maps.newHashMap());
				}
			} catch (SQLException e) {
				e.printStackTrace();
			}
		}
		return dataSource;
	}

	/**
	 * 事务管理器,在service层面上实现事务管理,达到平台无关性
	 * 
	 * @return
	 */
	@Bean
	public DataSourceTransactionManager getTransactionManager() {
		DataSourceTransactionManager transactionManager = new DataSourceTransactionManager();
		transactionManager.setDataSource(masterSlaveDataSource());
		return transactionManager;
	}

	/**
	 * 事务通知
	 * 
	 * @return
	 */
	@Bean
	public TransactionInterceptor getTxAdvice() {
		DefaultTransactionAttribute txAttrRequire = new DefaultTransactionAttribute();
		txAttrRequire.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRED);

		DefaultTransactionAttribute txAttrRequireReadonly = new DefaultTransactionAttribute();
		txAttrRequireReadonly.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRED);
		txAttrRequireReadonly.setReadOnly(true);

		NameMatchTransactionAttributeSource source = new NameMatchTransactionAttributeSource();
		
		if(txAttrConfig != null) {
			Map<String, String[]> txAttrMap = txAttrConfig.getTxAttrs();
			String[] txAttrRw = txAttrMap.get("txAttrRw");
			if(txAttrRw != null && txAttrRw.length > 0) {
				for(String rw:txAttrRw) {
					source.addTransactionalMethod(rw, txAttrRequire);
				}
			}
			String[] txAttrRead = txAttrMap.get("trAttrRead");
			if(txAttrRead != null && txAttrRead.length > 0) {
				for(String read:txAttrRead) {
					source.addTransactionalMethod(read, txAttrRequireReadonly);
				}
			}
		}
		source.addTransactionalMethod("*", txAttrRequire);
		return new TransactionInterceptor(getTransactionManager(), source);
	}
	
	
	@Bean
    public DefaultPointcutAdvisor defaultPointcutAdvisor(){
        DefaultPointcutAdvisor pointcutAdvisor = new DefaultPointcutAdvisor();
        pointcutAdvisor.setAdvice(getTxAdvice());
        AspectJExpressionPointcut pointcut = new AspectJExpressionPointcut();
        pointcut.setExpression("execution (* *..*Service.*(..))");
        pointcutAdvisor.setPointcut(pointcut);
        return pointcutAdvisor;
    }
}
