package com.ejianc.business.steelstructure.income.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ejianc.business.contractbase.pool.contractpool.api.IContractPoolApi;
import com.ejianc.business.contractbase.pool.contractpool.vo.ContractPoolVO;
import com.ejianc.business.contractbase.pool.enums.UpdateLevelEnum;
import com.ejianc.business.steelstructure.income.bean.ClaimEntity;
import com.ejianc.business.steelstructure.income.bean.ContractRegisterEntity;
import com.ejianc.business.steelstructure.income.bean.CostAdjustDetailEntity;
import com.ejianc.business.steelstructure.income.bean.CostAdjustEntity;
import com.ejianc.business.steelstructure.income.mapper.CostAdjustMapper;
import com.ejianc.business.steelstructure.income.service.IClaimService;
import com.ejianc.business.steelstructure.income.service.IContractRegisterService;
import com.ejianc.business.steelstructure.income.service.ICostAdjustDetailService;
import com.ejianc.business.steelstructure.income.service.ICostAdjustService;
import com.ejianc.business.steelstructure.income.utils.BillTypeCodeEnum;
import com.ejianc.business.steelstructure.income.utils.ValidateUtil;
import com.ejianc.business.steelstructure.income.vo.ClaimVO;
import com.ejianc.business.steelstructure.income.vo.CostAdjustDetailVO;
import com.ejianc.business.steelstructure.income.vo.CostAdjustVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IBillTypeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.collection.CollectionUtil;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 工程造价调整
 * 
 * @author generator
 * 
 */
@Service("costAdjustService")
public class CostAdjustServiceImpl extends BaseServiceImpl<CostAdjustMapper, CostAdjustEntity> implements ICostAdjustService{

    private static final String COST_ADJUST_APPLY = "COST_ADJUST_APPLY";//此处需要根据实际修改
    private final Logger logger = LoggerFactory.getLogger(this.getClass());


    @Autowired
    private IBillTypeApi billTypeApi;
    
    @Autowired
    private IBillCodeApi billCodeApi;
    
    @Autowired
    private IOrgApi iOrgApi;
    
    @Autowired
    private ICostAdjustDetailService detailService;
    
    @Autowired
    private IContractRegisterService registerService;

    @Autowired
    private IClaimService claimService;

    @Autowired
    private ValidateUtil validateUtil;

    @Autowired
    private IContractPoolApi contractPoolApi;


    @Override
    public CostAdjustVO saveOrUpdate(CostAdjustVO saveOrUpdateVO) {
        // 保存时校验合同version是否一致
        if (!Objects.equals(null, saveOrUpdateVO.getContractId())) {
            if(!validateUtil.validateUpStreamVersion(String.valueOf(saveOrUpdateVO.getContractId()),
                    BillTypeCodeEnum.施工合同.getCode(), saveOrUpdateVO.getContractVersion())){
                throw new BusinessException("该合同已被更新，请刷新后重做！");
            }
        }
        
        String operateType = null;
        CostAdjustEntity entity = BeanMapper.map(saveOrUpdateVO, CostAdjustEntity.class);
        
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(COST_ADJUST_APPLY, InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }

            operateType = "add";
        }else {
            operateType = "edit";
        }
        if(entity.getApprovalFlag() == null){
            entity.setApprovalFlag(0);
        }
        
        this.saveOrUpdate(entity);

        List<CostAdjustDetailEntity> detailList = entity.getDetailList();
        List<CostAdjustDetailEntity> newDetailList = new ArrayList<>();
        List<Long> deleteIds = new ArrayList<>();
        for (CostAdjustDetailEntity vo : detailList) {
            if (Objects.equals("del", vo.getRowState())) {
                deleteIds.add(vo.getId());
            }
            if (Objects.equals("add", vo.getRowState())) {
                vo.setId(null);
                vo.setAdjustId(entity.getId());
                newDetailList.add(vo);
            }
            if (Objects.equals("edit", vo.getRowState())) {
                newDetailList.add(vo);
            }
        }

        if (CollectionUtil.isNotEmpty(newDetailList)) {
            detailService.saveOrUpdateBatch(newDetailList, newDetailList.size(), false);
        }
        if (CollectionUtil.isNotEmpty(deleteIds)) {
            detailService.removeByIds(deleteIds, false);
        }


        claimService.updateCostAdjustId(entity.getId(), entity.getDetailList().stream().map(CostAdjustDetailEntity::getClaimId).collect(Collectors.toList()));

        return this.queryDetail(entity.getId());
    }

    @Override
    public CostAdjustVO queryDetail(Long id) {
        CostAdjustEntity entity = this.selectById(id);
        CostAdjustVO vo = BeanMapper.map(entity, CostAdjustVO.class);

        List<CostAdjustDetailVO> detailList = vo.getDetailList();
        if (ListUtil.isNotEmpty(detailList)) {
            List<Long> idList = detailList.stream().map(CostAdjustDetailVO::getClaimId).collect(Collectors.toList());

            LambdaQueryWrapper<ClaimEntity> wrapper = new LambdaQueryWrapper<>();
            wrapper.in(ClaimEntity::getId, idList);
            List<ClaimEntity> claimList = claimService.list(wrapper);
            Map<Long, ClaimEntity> claimMap = claimList.stream().collect(Collectors.toMap(ClaimEntity::getId, Function.identity()));
            for (CostAdjustDetailVO detailVO : detailList) {
                ClaimEntity claimEntity = claimMap.get(detailVO.getClaimId());
                detailVO.setReplyMny(claimEntity.getReplyMny());
                detailVO.setReplyTaxMny(claimEntity.getReplyTaxMny());
                detailVO.setReplyStatus(claimEntity.getReplyStatus());
                detailVO.setRowState("edit");
            }
            
            // 重新汇总二次经营批复金额
            BigDecimal replyMny = claimList.stream().filter(e -> e.getReplyMny() != null).map(ClaimEntity::getReplyMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            BigDecimal replyTaxMny = claimList.stream().filter(e -> e.getReplyTaxMny() != null).map(ClaimEntity::getReplyTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
            
            vo.setReplyMny(replyMny);
            vo.setReplyTaxMny(replyTaxMny);
        }
        
        return vo;
    }

    @Override
    public Boolean validateContract(Long contractId, Long id) {
        //同一个合同只能存在一个自由态或审批中的单据
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("contractId", new Parameter(QueryParam.EQ, contractId));
        queryParam.getParams().put("billState", new Parameter(QueryParam.NOT_IN, "1,3"));
        if (!Objects.equals(null, id)) {
            queryParam.getParams().put("id", new Parameter(QueryParam.NE, id));
        }
        List<CostAdjustEntity> list = super.queryList(queryParam, false);
        if (CollectionUtils.isNotEmpty(list)) {
            throw new BusinessException("当前合同存在非审批通过态的工程造价调整申请，不允许新增!");
        }
        return true;
    }

    @Override
    public CostAdjustVO queryMny(Long contractId) {
        List<Integer> billStateList = new ArrayList<>();
        billStateList.add(1);
        billStateList.add(3);
        
        // 查询补充协议总金额
        LambdaQueryWrapper<ContractRegisterEntity> contractWrapper = new LambdaQueryWrapper<>();
        contractWrapper.eq(ContractRegisterEntity::getMainContractId, contractId);
        contractWrapper.in(ContractRegisterEntity::getBillState, billStateList);
        List<ContractRegisterEntity> supplementList = registerService.list(contractWrapper);

        BigDecimal totalSupplementMny = supplementList.stream().filter(e -> e.getContractMny() != null).map(ContractRegisterEntity::getContractMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal totalSupplementTaxMny = supplementList.stream().filter(e -> e.getContractTaxMny() != null).map(ContractRegisterEntity::getContractTaxMny).reduce(BigDecimal.ZERO, BigDecimal::add);
        
        // 查询不含本期累计申请变动金额
        LambdaQueryWrapper<CostAdjustEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(CostAdjustEntity::getContractId, contractId);
        wrapper.in(CostAdjustEntity::getBillState, billStateList);
        wrapper.orderByDesc(BaseEntity::getCreateTime);
        List<CostAdjustEntity> list = this.list(wrapper);
        CostAdjustEntity entity = null;
        if (ListUtil.isNotEmpty(list)) {
            entity = list.get(0);
        }

        CostAdjustVO vo = new CostAdjustVO();
        vo.setTotalSupplementMny(totalSupplementMny);
        vo.setTotalSupplementTaxMny(totalSupplementTaxMny);
        
        if (Objects.equals(null, entity)) {
            vo.setLastApplyChangeMny(null);
            vo.setLastApplyChangeTaxMny(null);
        }else {
            vo.setLastApplyChangeMny(entity.getTotalApplyChangeMny());
            vo.setLastApplyChangeTaxMny(entity.getTotalApplyChangeTaxMny());
            vo.setApplyAdjustDate(entity.getApplyAdjustDate());
        }
        
        return vo;
    }

    @Override
    public List<ClaimVO> queryClaimList(Long contractId, String applyAdjustDate, String lastApplyAdjustDate) {
        List<Integer> billStateList = new ArrayList<>();
        billStateList.add(1);
        billStateList.add(3);

        LambdaQueryWrapper<ClaimEntity> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(ClaimEntity::getContractId, contractId);
        wrapper.in(ClaimEntity::getBillState, billStateList);
        wrapper.orderByDesc(ClaimEntity::getCreateTime);
        if (StringUtils.isBlank(lastApplyAdjustDate)) {
            wrapper.le(ClaimEntity::getCommitDate, applyAdjustDate);
        }else {
            wrapper.between(ClaimEntity::getCommitDate, lastApplyAdjustDate, applyAdjustDate);
        }
        
        List<ClaimEntity> list = claimService.list(wrapper);
        
        if (ListUtil.isEmpty(list)) {
            return new ArrayList<>();
        }else {
            return BeanMapper.mapList(list, ClaimVO.class);
        }
    }

    @Override
    public Boolean updateContractPool(ContractRegisterEntity entity) {
        ContractPoolVO poolVO = new ContractPoolVO();

        poolVO.setId(entity.getId());
        poolVO.setSourceId(entity.getId());
        poolVO.setTotalCostAdjustTaxMny(entity.getTotalCostAdjustTaxMny());
        poolVO.setTotalCostAdjustMny(entity.getTotalCostAdjustMny());
        poolVO.setTotalCostAdjustTax(entity.getTotalCostAdjustTax());
        poolVO.setUpdateLevel(UpdateLevelEnum.更新非空字段.getLevelCode());

        CommonResponse<ContractPoolVO> transDataResp = contractPoolApi.saveOrUpdateContract(poolVO);
        if(transDataResp.isSuccess()) {
            return true;
        } else {
            logger.error("合同id-{}推送合同池失败，{}", entity.getId(), transDataResp.getMsg());
            throw new BusinessException("更新合同池信息失败，请稍后重试！");
        }
    }
}
