package com.ejianc.business.store.util;

import com.ejianc.business.store.consts.InOutTypeEnum;
import com.ejianc.business.store.consts.StoreCommonConsts;
import com.ejianc.business.store.vo.FlowVO;
import com.ejianc.business.store.vo.InOutVO;
import com.ejianc.business.store.vo.SurplusUpdateVO;
import com.ejianc.business.store.vo.SurplusVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.support.idworker.util.IdWorker;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 仓库管理工具类
 * @date 2022/1/24
 */
public class StoreManageUtil {

    private static SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");


    /**
     * @param inOutTypeEnum 出入库类型
     * @param partyaFlag    是否甲供材标志 StoreCommonConsts.YES 是  StoreCommonConsts.NO否
     * @description: 获取到一个根据出入库类型初始化参数的流水VO
     * @return: com.ejianc.business.store.vo.FlowVO
     * @author songlx
     * @date: 2022/1/24
     */
    public static FlowVO getFlowVO(InOutTypeEnum inOutTypeEnum, Integer partyaFlag) {
        FlowVO flowVO = new FlowVO();
        //流水编号,出入库标识(1位,1入库2出库) + 出入库类型(两位,参照枚举) + 时间流水号(yyyyMMddHHmmssSSS)
        String billCode = inOutTypeEnum.getInOutFlag() + inOutTypeEnum.getInOutType().toString() + getTimeStr();
        flowVO.setBillCode(billCode);
        flowVO.setBillDate(new Date());
        flowVO.setInOutFlag(inOutTypeEnum.getInOutFlag());
        flowVO.setInOutType(inOutTypeEnum.getInOutType());
        flowVO.setInOutTypeName(inOutTypeEnum.getInOutTypeName());
        flowVO.setOutUseFlag(inOutTypeEnum.getOutUseFlag());
        flowVO.setPartyaFlag(partyaFlag != null ? partyaFlag : StoreCommonConsts.NO);
        flowVO.setReturnGoodsFlag(InOutTypeEnum.材料退货.getInOutType().equals(inOutTypeEnum.getInOutType()) ? StoreCommonConsts.YES : StoreCommonConsts.NO);
        flowVO.setReturnStoreFlag(InOutTypeEnum.领料退库.getInOutType().equals(inOutTypeEnum.getInOutType()) ? StoreCommonConsts.YES : StoreCommonConsts.NO);
        flowVO.setEffectiveDate(new Date());
        flowVO.setAccountFlag(StoreCommonConsts.NO);
        flowVO.setSettleFlag(StoreCommonConsts.NO);
        //生效标志:入库一般是审批后推送的直接传StoreCommonConsts.YES, 出库类型需要走校验什么默认否,由仓库方法内部出库成功后修改
        //退库也是审批通过后可以是生效的
        if (StoreCommonConsts.IN_OUT_TYPE_IN.equals(inOutTypeEnum.getOutUseFlag()) || InOutTypeEnum.领料退库.getInOutType().equals(inOutTypeEnum.getInOutType())) {
            flowVO.setEffectiveState(StoreCommonConsts.YES);
        } else {
            flowVO.setEffectiveState(StoreCommonConsts.NO);
        }
        //放入是否周转材标识
        flowVO.setTurnFlag(inOutTypeEnum.getTurnFlag());
        return flowVO;
    }


    /**
     * @param flowVOList 入库/出库流水VO
     * @param isRollback 是否回滚入库
     * @description: 根据流水VO更新库存
     * @return: com.ejianc.business.store.vo.SurplusUpdateVO
     * @author songlx
     * @date: 2022/1/25
     */
    public static SurplusUpdateVO getSurplusUpdateVO(Long storeId, List<FlowVO> flowVOList, boolean isRollback) {
        Set<Long> materialIdsSet = flowVOList.stream().map(e -> e.getMaterialId()).collect(Collectors.toSet());
        Set<Long> sourceIdsSet = flowVOList.stream().map(e -> e.getSourceId()).collect(Collectors.toSet());
        SurplusUpdateVO surplusUpdateVO = new SurplusUpdateVO();
        surplusUpdateVO.setMaterialIds(new ArrayList<>(materialIdsSet));
        surplusUpdateVO.setSourceIds(new ArrayList<>(sourceIdsSet));
        surplusUpdateVO.setFlowVOList(flowVOList);
        ArrayList<SurplusVO> surplusVOS = new ArrayList<>();
        for (FlowVO flowVO : flowVOList) {
            SurplusVO surplusVO = BeanMapper.map(flowVO, SurplusVO.class);
            surplusVO.setSurplusMny(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getMny()) : flowVO.getMny());
            surplusVO.setSurplusTaxMny(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getTaxMny()) : flowVO.getTaxMny());
            surplusVO.setSurplusNum(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getNum()) : flowVO.getNum());
            surplusVO.setInstoreMny(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getMny()) : flowVO.getMny());
            surplusVO.setInstoreTaxMny(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getTaxMny()) : flowVO.getTaxMny());
            surplusVO.setInstoreNum(isRollback ? ComputeUtil.convertToMinusNumber(flowVO.getNum()) : flowVO.getNum());
            //正向
            if (!isRollback) {
                surplusVO.setReturnGoodsNum(ComputeUtil.safeSub(BigDecimal.ZERO, flowVO.getNum()));
            } else {
                surplusVO.setReturnGoodsNum(ComputeUtil.convertToMinusNumber(flowVO.getNum()));
            }

            surplusVOS.add(surplusVO);
        }
        surplusUpdateVO.setStoreId(storeId);
        surplusUpdateVO.setSurplusVOList(surplusVOS);
        return surplusUpdateVO;
    }

    /**
     * @param inFlowVO   入库VO
     * @param outInOutVO 出库拟占用关系VO
     * @param useNum     出库占用该入库的数量
     * @description: 根据入库VO和出库VO得到出入关联VO
     * @return: com.ejianc.business.store.vo.InOutVO
     * @author songlx
     * @date: 2022/1/26
     */
    public static InOutVO getInOutVO(FlowVO inFlowVO, InOutVO outInOutVO, BigDecimal useNum) {
        InOutVO inOutVO = BeanMapper.map(inFlowVO, InOutVO.class);
        inOutVO.setId(null);
        inOutVO.setCreateUserCode(null);
        inOutVO.setCreateTime(null);
        inOutVO.setUpdateTime(null);
        inOutVO.setUpdateUserCode(null);
        inOutVO.setInStoreId(inFlowVO.getStoreId());
        inOutVO.setInStoreName(inFlowVO.getStoreName());
        inOutVO.setInProjectId(inFlowVO.getProjectId());
        inOutVO.setInProjectName(inFlowVO.getProjectName());
        inOutVO.setInFlowId(inFlowVO.getId());
        inOutVO.setInBillId(inFlowVO.getSourceId());
        inOutVO.setInBillDetailId(inFlowVO.getSourceDetailId());
        inOutVO.setInBillCode(inFlowVO.getSourceBillCode());
        inOutVO.setOutNum(BigDecimal.ZERO);
        inOutVO.setReturnStoreNum(BigDecimal.ZERO);
        inOutVO.setOutLockNum(useNum);
        inOutVO.setEffectiveState(StoreCommonConsts.NO);
        inOutVO.setOutUseFlag(StoreCommonConsts.UseOutFlag.USEING);

        inOutVO.setInOutType(outInOutVO.getInOutType());
        inOutVO.setInOutTypeName(outInOutVO.getInOutTypeName());
        inOutVO.setOutStoreId(outInOutVO.getOutStoreId());
        inOutVO.setOutStoreName(outInOutVO.getOutStoreName());
        inOutVO.setOutProjectId(outInOutVO.getOutProjectId());
        inOutVO.setOutProjectName(outInOutVO.getOutProjectName());
        inOutVO.setOutFlowId(outInOutVO.getOutFlowId());
        inOutVO.setOutBillDetailId(outInOutVO.getOutBillDetailId());
        inOutVO.setOutBillId(outInOutVO.getOutBillId());
        inOutVO.setOutBillCode(outInOutVO.getOutBillCode());
        inOutVO.setMaterialId(outInOutVO.getMaterialId());
        inOutVO.setOutDate(outInOutVO.getOutDate());

        inOutVO.setPickUnitId(outInOutVO.getPickUnitId());
        inOutVO.setPickUnitName(outInOutVO.getPickUnitName());
        inOutVO.setPickContractId(outInOutVO.getPickContractId());
        inOutVO.setPickContractName(outInOutVO.getPickContractName());
        return inOutVO;
    }

    /**
     * @param inFlowVOList
     * @description: 直入直出根据入库构建出库流水
     * @return: java.util.List<com.ejianc.business.store.vo.FlowVO>
     * @author songlx
     * @date: 2022/2/9
     */
    public static List<FlowVO> getOutFlowVOByInFlowVO(List<FlowVO> inFlowVOList) {
        List<FlowVO> flowVOS = new ArrayList<>();
        inFlowVOList.forEach(t -> {
            FlowVO out = BeanMapper.map(t, FlowVO.class);
            long outId = IdWorker.getId();
            out.setId(outId);
            out.setInOutFlag(StoreCommonConsts.IN_OUT_TYPE_OUT);
            t.setStraightOutFlowId(outId);
            t.setId(IdWorker.getId());
            flowVOS.add(out);
        });
        return flowVOS;
    }

    /**
     * @param inFlowVOList
     * @param outFlowVOList
     * @description: 直入直出根据出入库流水构建出入库关系占用VO
     * @return: java.util.List<com.ejianc.business.store.vo.InOutVO>
     * @author songlx
     * @date: 2022/2/9
     */
    public static List<InOutVO> getInOutVOsByInAndOutFlowVOList(List<FlowVO> inFlowVOList, List<FlowVO> outFlowVOList) {
        List<InOutVO> inOutVOS = new ArrayList<>();
        for (int j = 0; j < inFlowVOList.size(); j++) {
            FlowVO inFlowVO = inFlowVOList.get(j);
            FlowVO outFlowVO = outFlowVOList.get(j);

            InOutVO inOutVO = BeanMapper.map(inFlowVO, InOutVO.class);
            inFlowVO.setStraightOutFlowId(outFlowVO.getId());
            inOutVO.setId(null);
            inOutVO.setVersion(null);
            inOutVO.setOutDate(new Date());
            inOutVO.setInOutType(outFlowVO.getInOutType());
            inOutVO.setInOutTypeName(outFlowVO.getInOutTypeName());
            inOutVO.setInStoreId(inFlowVO.getStoreId());
            inOutVO.setInStoreName(inFlowVO.getStoreName());
            inOutVO.setInProjectId(inFlowVO.getProjectId());
            inOutVO.setInProjectName(inFlowVO.getProjectName());
            inOutVO.setInFlowId(inFlowVO.getId());
            inOutVO.setInBillId(inFlowVO.getSourceId());
            inOutVO.setInBillDetailId(inFlowVO.getSourceDetailId());
            inOutVO.setInBillCode(inFlowVO.getSourceBillCode());
            inOutVO.setOutStoreId(outFlowVO.getStoreId());
            inOutVO.setOutStoreName(outFlowVO.getStoreName());
            inOutVO.setOutProjectId(outFlowVO.getProjectId());
            inOutVO.setOutProjectName(outFlowVO.getProjectName());
            inOutVO.setOutFlowId(outFlowVO.getId());
            inOutVO.setOutBillId(outFlowVO.getSourceId());
            inOutVO.setOutBillDetailId(outFlowVO.getSourceDetailId());
            inOutVO.setOutBillCode(outFlowVO.getSourceBillCode());
            inOutVO.setOutNum(inFlowVO.getNum());
            inOutVO.setReturnStoreNum(BigDecimal.ZERO);
            inOutVO.setOutLockNum(BigDecimal.ZERO);
            inOutVO.setEffectiveDate(new Date());
            inOutVO.setEffectiveState(StoreCommonConsts.YES);
            inOutVO.setOutUseFlag(StoreCommonConsts.UseOutFlag.USE_FINISH);
            inOutVOS.add(inOutVO);
        }
        return inOutVOS;
    }


    static String getTimeStr() {
        return sdf.format(new Date());
    }

    public static void main(String[] args) {
        System.out.println(InOutTypeEnum.周转材收料入库.getTurnFlag());
        System.out.println(InOutTypeEnum.收料入库.getTurnFlag());
    }

    public static InOutVO getTurnInOutVO(FlowVO inFlowVO, InOutVO outInOutVO, BigDecimal useNum,
                                         BigDecimal thisLockNetMny,
                                         BigDecimal thisLockNetTaxMny,
                                         BigDecimal thisLockPurchaseMny,
                                         BigDecimal thisLockPurchaseTaxMny) {
        InOutVO inOutVO = BeanMapper.map(inFlowVO, InOutVO.class);

        //根据占用净值和数量计算净值单价
        inOutVO.setOutNetMny(thisLockNetMny);
        inOutVO.setOutNetTaxMny(thisLockNetTaxMny);
        BigDecimal netPrice = ComputeUtil.safeDiv(thisLockNetMny, useNum);
        BigDecimal netTaxPrice = ComputeUtil.safeDiv(thisLockNetTaxMny, useNum);
        inOutVO.setPrice(netPrice);
        inOutVO.setTaxPrice(netTaxPrice);

        inOutVO.setPurchaseMny(thisLockPurchaseMny);
        inOutVO.setPurchaseTaxMny(thisLockPurchaseTaxMny);

        inOutVO.setId(null);
        inOutVO.setCreateUserCode(null);
        inOutVO.setCreateTime(null);
        inOutVO.setUpdateTime(null);
        inOutVO.setUpdateUserCode(null);
        inOutVO.setInStoreId(inFlowVO.getStoreId());
        inOutVO.setInStoreName(inFlowVO.getStoreName());
        inOutVO.setInProjectId(inFlowVO.getProjectId());
        inOutVO.setInProjectName(inFlowVO.getProjectName());
        inOutVO.setInFlowId(inFlowVO.getId());
        inOutVO.setInBillId(inFlowVO.getSourceId());
        inOutVO.setInBillDetailId(inFlowVO.getSourceDetailId());
        inOutVO.setInBillCode(inFlowVO.getSourceBillCode());
        inOutVO.setOutNum(BigDecimal.ZERO);
        inOutVO.setReturnStoreNum(BigDecimal.ZERO);
        inOutVO.setOutLockNum(useNum);
        inOutVO.setEffectiveState(StoreCommonConsts.NO);
        inOutVO.setOutUseFlag(StoreCommonConsts.UseOutFlag.USEING);

        inOutVO.setInOutType(outInOutVO.getInOutType());
        inOutVO.setInOutTypeName(outInOutVO.getInOutTypeName());
        inOutVO.setOutStoreId(outInOutVO.getOutStoreId());
        inOutVO.setOutStoreName(outInOutVO.getOutStoreName());
        inOutVO.setOutProjectId(outInOutVO.getOutProjectId());
        inOutVO.setOutProjectName(outInOutVO.getOutProjectName());
        inOutVO.setOutFlowId(outInOutVO.getOutFlowId());
        inOutVO.setOutBillDetailId(outInOutVO.getOutBillDetailId());
        inOutVO.setOutBillId(outInOutVO.getOutBillId());
        inOutVO.setOutBillCode(outInOutVO.getOutBillCode());
        inOutVO.setMaterialId(outInOutVO.getMaterialId());
        inOutVO.setOutDate(outInOutVO.getOutDate());

        inOutVO.setPickUnitId(outInOutVO.getPickUnitId());
        inOutVO.setPickUnitName(outInOutVO.getPickUnitName());
        inOutVO.setPickContractId(outInOutVO.getPickContractId());
        inOutVO.setPickContractName(outInOutVO.getPickContractName());
        return inOutVO;
    }
}
