package com.ejianc.business.store.service.handler;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.store.bean.FlowEntity;
import com.ejianc.business.store.bean.InOutEntity;
import com.ejianc.business.store.consts.InOutTypeEnum;
import com.ejianc.business.store.consts.StoreCommonConsts;
import com.ejianc.business.store.service.IFlowService;
import com.ejianc.business.store.service.IInOutService;
import com.ejianc.business.store.service.IStoreService;
import com.ejianc.business.store.service.ISurplusService;
import com.ejianc.business.store.util.StoreManageUtil;
import com.ejianc.business.store.vo.FlowVO;
import com.ejianc.business.store.vo.InOutVO;
import com.ejianc.business.store.vo.StoreManageVO;
import com.ejianc.business.store.vo.SurplusUpdateVO;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.support.idworker.util.IdWorker;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author songlx
 * @version 1.0
 * @description: 直入直出处理器
 * @date 2022/1/25
 */
@Service
public class StraightInOutStoreHandler implements IStoreManageHandler {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    IFlowService flowService;

    @Autowired
    IInOutService inOutService;

    @Autowired
    ISurplusService surplusService;

    @Autowired
    IStoreService storeService;


    /**
     * @param storeManageVO
     * @description: 仓库直入直出具体实现逻辑, 直入直出审批通过后调用
     * 正向入库
     * 1 直接保存入库流水,生成出库流水
     * 2 保存出入库关系
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handle(StoreManageVO storeManageVO) {
        //设置仓库属性
        storeManageVO = storeService.setStoreAttr(storeManageVO);
        Long storeId = storeManageVO.getStoreId();
        Long sourceId = storeManageVO.getSourceId();
        if (sourceId != null) {
            QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("source_id", sourceId);
            int count = flowService.count(queryWrapper);
            if (count > 0) {
                InOutTypeEnum inOutTypeEnum = storeManageVO.getInOutTypeEnum();
                String billTypeName = inOutTypeEnum != null ? inOutTypeEnum.getInOutTypeName() : "无";
                logger.error("---start---------重复校验ERROR---------------------------------");
                logger.error("仓库入库流水重复校验：已存在该单据入库流水！来源单据类型：{}，单据ID：{}", billTypeName, sourceId);
                logger.error("---end-----------重复校验ERROR---------------------------------");
                return CommonResponse.success("仓库入库流水重复校验：已存在该单据入库流水！来源单据ID：" + sourceId, storeManageVO);
            }
        }
        List<FlowVO> inFlowVOList = storeManageVO.getFlowVOList();
        // 直入直出入库初始化, 入库即出库,出库数量完毕
        inFlowVOList.forEach(flowVO -> {
            flowVO.setId(IdWorker.getId());
            flowVO.setOutNum(flowVO.getNum());
            flowVO.setOutLockNum(BigDecimal.ZERO);
            flowVO.setSurplusNum(BigDecimal.ZERO);
            flowVO.setSurplusMny(BigDecimal.ZERO);
            flowVO.setSurplusTaxMny(BigDecimal.ZERO);
            flowVO.setOutUseFlag(StoreCommonConsts.UseOutFlag.USE_FINISH);
            flowVO.setEffectiveDate(new Date());
            flowVO.setEffectiveState(StoreCommonConsts.YES);
            flowVO.setEstimatePrice(flowVO.getPrice());
            flowVO.setEstimateTaxPrice(flowVO.getTaxPrice());
        });
        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, inFlowVOList, false);
        List<FlowVO> outFlowVOList = storeManageVO.getStraightOutFlowVOList();
        outFlowVOList.forEach(t -> {
            t.setId(IdWorker.getId());
            t.setEffectiveState(StoreCommonConsts.YES);
        });
        // 根据出入库流水获取
        List<InOutVO> inOutVOList = StoreManageUtil.getInOutVOsByInAndOutFlowVOList(inFlowVOList, outFlowVOList);
        List<InOutEntity> inOutEntities = BeanMapper.mapList(inOutVOList, InOutEntity.class);
        inOutService.saveOrUpdateBatch(inOutEntities);
        inFlowVOList.addAll(outFlowVOList);
        List<FlowEntity> flowEntities = BeanMapper.mapList(inFlowVOList, FlowEntity.class);
        boolean b = flowService.saveOrUpdateBatch(flowEntities);
        //入库成功后更新库存入库量,不更新余量
        if (b) {
            SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, false);
            //更新后将这些物资最新库存返回
            storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());

        }
        return CommonResponse.success(storeManageVO);
    }

    /**
     * @param storeManageVO
     * @description: 仓库直入直出回滚实现逻辑, 直入直出弃审调用
     * 1 删除入库流水和生成的出库流水
     * 2 删除出入库占用关系表
     * @return: com.ejianc.framework.core.response.CommonResponse<com.ejianc.business.store.vo.StoreManageVO>
     * @author songlx
     * @date: 2022/1/25
     */
    @Override
    public CommonResponse<StoreManageVO> handleRollback(StoreManageVO storeManageVO) {
        Long storeId = storeManageVO.getStoreId();
        // 直入直出入库的单据id,入库和出库都是系统一个来源sourceId
        Long sourceId = storeManageVO.getSourceId();
        //校验入库单是否被使用，结算或者对账
        CommonResponse<List<FlowEntity>> res = flowService.validateIsUse(sourceId, storeId);
        if(!res.isSuccess()){
            return CommonResponse.error(res.getMsg());
        }

        QueryWrapper<FlowEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("store_id", storeId);
        queryWrapper.eq("source_id", sourceId);
        List<FlowEntity> instoreList = flowService.list(queryWrapper);

        if(CollectionUtils.isEmpty(instoreList)){
            return CommonResponse.error("未查询到该入库单流水信息！");
        }

        List<FlowVO> flowVOList = BeanMapper.mapList(instoreList, FlowVO.class);

        //获取到流水里的入库单据,直入直出只根据入库回退库存就可以
        List<FlowVO> instoreFlowVOList = flowVOList.stream().filter(s -> s.getInOutFlag() == InOutTypeEnum.直入直出入库.getInOutFlag()).collect(Collectors.toList());

        SurplusUpdateVO surplusUpdateVO = StoreManageUtil.getSurplusUpdateVO(storeId, instoreFlowVOList, true);

        //删除入库和出库流水
        List<Long> delListIds = flowVOList.stream().map(FlowVO::getId).collect(Collectors.toList());
        flowService.removeByIds(delListIds);

        QueryWrapper<InOutEntity> inOutEntityQueryWrapper = new QueryWrapper<>();
        inOutEntityQueryWrapper.eq("in_bill_id", sourceId);
        boolean b = inOutService.remove(inOutEntityQueryWrapper);

        //入库成功后更新库存入库,不更新余量
        if (b) {
            SurplusUpdateVO surplusUpdateVORet = surplusService.updateStoreSurplus(surplusUpdateVO, true, false);
            //更新后将这些物资最新库存返回
            storeManageVO.setSurplusVOList(surplusUpdateVORet.getSurplusVOList());

        }
        return CommonResponse.success(storeManageVO);
    }
}
