package com.ejianc.business.sub.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.material.vo.ParamsCheckDsVO;
import com.ejianc.business.material.vo.ParamsCheckVO;
import com.ejianc.business.sub.bean.ContractEntity;
import com.ejianc.business.sub.bean.OddjobEntity;
import com.ejianc.business.sub.mapper.OddjobMapper;
import com.ejianc.business.sub.service.IContractService;
import com.ejianc.business.sub.service.IOddjobService;
import com.ejianc.business.sub.utils.BigDecimalUtils;
import com.ejianc.business.sub.vo.*;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.auth.session.UserContext;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 * 分包零星用工表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("OddjobServiceImpl")
public class OddjobServiceImpl extends BaseServiceImpl<OddjobMapper, OddjobEntity> implements IOddjobService {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final String SUB_ODDJOB_BILL_CODE = "SUB_ODDJOB";

    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private JedisPool jedisPool;
    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private SessionManager sessionManager;

    private static String PARAM_ODDJOB_ALL_COUNT = "P-0X474v30";

    @Override
    public OddjobVO insertOrUpdate(OddjobVO oddjobVO) {
        if(checkFinishContract(oddjobVO)){
            throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
        }
        ContractEntity contractEntity = contractService.selectById(oddjobVO.getContractId());
        //保存时校验合同version是否一致
        if(oddjobVO.getContractVersion()!=null&&oddjobVO.getContractVersion()!=0){
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try{
                locked = RedisTool.tryLock(jedis, String.valueOf(oddjobVO.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------"+locked);
                if(locked){
                    Integer version = contractEntity.getVersion()==null?0:contractEntity.getVersion();
                    Integer conVersion = oddjobVO.getContractVersion();
                    if(!version.equals(conVersion)){
                        throw new BusinessException("分包合同已被更新，请刷新后重做！");
                    }
                }else{
                    throw new BusinessException("出现并发操作,请稍后重试！");
                }
            }catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(oddjobVO.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }
        }

        if(StringUtils.isEmpty(oddjobVO.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_ODDJOB_BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                oddjobVO.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        if (null==oddjobVO.getIsUse()){
            oddjobVO.setIsUse(Boolean.FALSE);
        }
        OddjobEntity oddjobEntity = BeanMapper.map(oddjobVO, OddjobEntity.class);
        super.saveOrUpdate(oddjobEntity, false);
        return BeanMapper.map(oddjobEntity, OddjobVO.class);
    }

    @Override
    public OddjobVO queryDetail(Long id) {
        OddjobEntity oddjobEntity = super.selectById(id);
        OddjobVO oddjobVO =  BeanMapper.map(oddjobEntity, OddjobVO.class);
        return oddjobVO;
    }

    @Override
    public CommonResponse<IPage<OddjobVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        UserContext userContext = sessionManager.getUserContext();
        String authOrgIds = userContext.getAuthOrgIds();
        if (org.apache.commons.lang.StringUtils.isNotEmpty(authOrgIds)) {
            CommonResponse<List<OrgVO>> authResponse =
                   orgApi.findChildrenByParentIds(Arrays.stream(authOrgIds.split(",")).map(Long::parseLong).
                            collect(Collectors.toList()));
            param.getParams().put("orgId", new Parameter(QueryParam.IN, authResponse.getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        } else {
            param.getParams().put("orgId", new Parameter(QueryParam.IN, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        }
        IPage<OddjobEntity> page = queryPage(param, false);
        List<OddjobVO> oddjobVOList = BeanMapper.mapList(page.getRecords(), OddjobVO.class);
        IPage<OddjobVO> oddjobVOIPage = new Page<>();
        oddjobVOIPage.setCurrent(page.getCurrent());
        oddjobVOIPage.setRecords(oddjobVOList);
        oddjobVOIPage.setSize(page.getSize());
        oddjobVOIPage.setTotal(page.getTotal());
        oddjobVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", oddjobVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<OddjobVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(vos.stream().map(OddjobVO::getId).collect(Collectors.toList()), false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public OddjobVO queryDetailAdd(Long contractId) {
        ContractEntity contractEntity = contractService.selectById(contractId);
        OddjobVO oddjobVO = BeanMapper.map(contractEntity, OddjobVO.class);
        oddjobVO.setBillCode(null);
        oddjobVO.setEmployeeId(null);
        oddjobVO.setEmployeeName(null);
        oddjobVO.setContractId(contractEntity.getId());
        oddjobVO.setBillState(null);
        oddjobVO.setCreateUserCode(null);
        oddjobVO.setCreateTime(null);
        oddjobVO.setUpdateUserCode(null);
        oddjobVO.setUpdateTime(null);
        oddjobVO.setId(null);
        return oddjobVO;
    }

    @Override
    public OddjobRecordVO queryDetailRecord(Long id) {
        ContractEntity contractEntity = contractService.selectById(id);
        OddjobRecordVO oddjobRecordVO = new OddjobRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        //BigDecimal sumOddjobTaxMny = contractEntity.getSumOddjobTaxMny() == null ? BigDecimal.ZERO : contractEntity.getSumOddjobTaxMny();
        oddjobRecordVO.setContractId(id);
        oddjobRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        oddjobRecordVO.setContractStatus(Integer.valueOf(contractEntity.getContractStatus()));
        oddjobRecordVO.setContractTaxMny(contractTaxMny);



        LambdaQueryWrapper<OddjobEntity> lambda = Wrappers.<OddjobEntity>lambdaQuery();
        lambda.eq(OddjobEntity::getContractId,id);
        lambda.eq(OddjobEntity::getTenantId,InvocationInfoProxy.getTenantid());
        lambda.in(OddjobEntity::getOrgId, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        List<OddjobEntity> entitiesAllState = super.list(lambda);
        lambda.in(OddjobEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<OddjobEntity> entities = super.list(lambda);
        List<OddjobRecordDetailVO> oddjobRecordDetailVOList = BeanMapper.mapList(entities, OddjobRecordDetailVO.class);
        BigDecimal oddjobTaxMny = BigDecimal.ZERO;
        BigDecimal settleTaxMny = BigDecimal.ZERO;
        if(CollectionUtils.isNotEmpty(oddjobRecordDetailVOList)) {
            for(OddjobRecordDetailVO ojr:oddjobRecordDetailVOList){
                BigDecimal jobTaxMny = ojr.getJobTaxMny() == null ? BigDecimal.ZERO : ojr.getJobTaxMny();
                BigDecimal scale = ComputeUtil.bigDecimalPercent(jobTaxMny, contractTaxMny, 2);
                ojr.setScale(scale);

                oddjobTaxMny = oddjobTaxMny.add(jobTaxMny);
                BigDecimal oddjobSettleTaxMny = ojr.getOddjobSettleTaxMny() == null ? BigDecimal.ZERO : ojr.getOddjobSettleTaxMny();
                settleTaxMny = settleTaxMny.add(oddjobSettleTaxMny);
            }
//            oddjobRecordDetailVOList.forEach(entity ->{
//                BigDecimal jobTaxMny = entity.getJobTaxMny() == null ? BigDecimal.ZERO : entity.getJobTaxMny();
//                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (jobTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
//                entity.setScale(scale);
//            });
        }
        oddjobRecordVO.setDetailList( BeanMapper.mapList(entitiesAllState, OddjobRecordDetailVO.class));

        oddjobRecordVO.setSumOddjobTaxMny(oddjobTaxMny);
        oddjobRecordVO.setSumOddjobSettleTaxMny(settleTaxMny);
        BigDecimal sumScale = BigDecimalUtils.safeDiv(oddjobTaxMny,contractTaxMny).multiply(new BigDecimal(100));
        oddjobRecordVO.setSumScale(sumScale);
        return oddjobRecordVO;
    }

    @Override
    public void updateSettleByPks(List<Long> pks, Boolean flag) {
        LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
        updateWrapper.set(OddjobEntity::getSettleFlag, flag);
        updateWrapper.in(OddjobEntity::getId, pks);
        this.update(updateWrapper);
    }
    @Override
    public void updateSettleMnyByPks(List<SettleOddjobVO> list, Boolean flag) {
        if(null!=list&&list.size()>0){
            for(SettleOddjobVO vo :list){
                LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
                updateWrapper.set(OddjobEntity::getSettleFlag, flag);
                updateWrapper.set(OddjobEntity::getOddjobSettleTaxMny, vo.getSettleMny());
                updateWrapper.eq(OddjobEntity::getId, vo.getOddjobId());
                this.update(updateWrapper);
            }
        }
    }
    @Override
    public void updateSettleMny(List<FinishOddjobVO> list, Boolean flag) {
        if(null!=list&&list.size()>0){
            for(FinishOddjobVO vo :list){
                LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
                updateWrapper.set(OddjobEntity::getSettleFlag, flag);
                updateWrapper.set(OddjobEntity::getOddjobSettleTaxMny, vo.getSettleMny());
                updateWrapper.eq(OddjobEntity::getId, vo.getOddjobId());
                this.update(updateWrapper);
            }
        }
    }

    @Override
    public void excelExport(QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.setPageIndex(1);
        param.setPageSize(-1);
        IPage<OddjobEntity> pageData = queryPage(param, false);

        Map<String, Object> beans = new HashMap<String, Object>();
        if (null != pageData.getRecords() && CollectionUtils.isNotEmpty(pageData.getRecords())) {
            List<OddjobVO> list = BeanMapper.mapList(pageData.getRecords(), OddjobVO.class);
            list.forEach(vo -> {
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            });
            beans.put("records", list);
            ExcelExport.getInstance().export("oddjob-export.xlsx", beans, response);
        }
    }

    @Override
    public OddjobVO querySumOddjobMny(Long contractId) {
        QueryWrapper<OddjobEntity> wrapper = new QueryWrapper<>();
        wrapper.select(" IFNULL(sum(job_tax_Mny),0) AS sumOddjobMny");
        wrapper.eq("contract_id",contractId);
        wrapper.eq("dr",0);
        wrapper.in("bill_state",BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        OddjobEntity one = super.getOne(wrapper);
        OddjobVO oddjobVO = new OddjobVO();
        if(one != null){
            oddjobVO = BeanMapper.map(one, OddjobVO.class);
        }
        return oddjobVO;
    }

    /**
     * 单据管控参数查询，根据合同该合同下所有的零星用工金额且不受单据状态限制
     * @param contractId
     * @param id
     * @return
     */
    @Override
    public OddjobVO queryCheckParamSumOddjobMny(Long contractId, Long id) {
        QueryWrapper<OddjobEntity> wrapper = new QueryWrapper<>();
        wrapper.select(" IFNULL(sum(job_tax_Mny),0) AS sumOddjobMny");
        if(id!=null){
            wrapper.ne("id",id);
        }
        wrapper.eq("contract_id",contractId);
        wrapper.eq("dr",0);
        OddjobEntity one = this.getOne(wrapper);
        OddjobVO oddjobVO = new OddjobVO();
        if(one != null){
            oddjobVO = BeanMapper.map(one, OddjobVO.class);
        }
        return oddjobVO;
    }

    @Override
    public ParamsCheckVO checkAllParams(OddjobVO vo) {
        Long curOrgId = Optional.ofNullable(vo.getOrgId()).orElse(InvocationInfoProxy.getOrgId());
        // 存放预警结果
        Map<String, List<ParamsCheckDsVO>> paramsCheckVOMap = new HashMap<>();
        paramsCheckVOMap.put("alert",new ArrayList<>());
        paramsCheckVOMap.put("warn",new ArrayList<>());
        Long contractId = vo.getContractId();
//        List<OddjobContentVo> oddjobContentList = vo.getOddjobContentList();
        String[] paramsArray = {"none", "warn", "alert"};
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        paramsCheckVO.setWarnType(paramsArray[0]);
        /*if (CollectionUtils.isEmpty(oddjobContentList)) {
            return paramsCheckVO;
        }*/
        //CommonResponse<BillParamVO> countParamByCode = paramConfigApi.getBillParamByCode(PARAM_ODDJOB_ALL_COUNT);
        CommonResponse<List<BillParamVO>> countParamByCode = paramConfigApi.getBillParamByCodeAndOrgId(PARAM_ODDJOB_ALL_COUNT, curOrgId);
        if (!countParamByCode.isSuccess() || null == countParamByCode.getData()) {
            logger.info(countParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }
        //根据合同id查询分包合同
        ContractEntity contractEntity = contractService.selectById(contractId);
        //只有自采合同受单据管控校验，集采不受校验 20221105
        if("1".equals(contractEntity.getPurchaseType())){
            List<BillParamVO> data = countParamByCode.getData();
            for (BillParamVO datum : data) {
                if (0 != datum.getControlType()) {
                    // 超出比例
                    BigDecimal roleValue = datum.getRoleValue();

                    BigDecimal comMny = contractEntity.getContractTaxMny().multiply(roleValue.divide(BigDecimal.valueOf(100)));
                    List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();

                //本次累计用工金额
                BigDecimal jobTaxMny = vo.getJobTaxMny();
                // 根据合同id获取累计零工金额数据
                OddjobVO oddjobVO = this.queryCheckParamSumOddjobMny(contractId,vo.getId());
                //累计零工金额
                BigDecimal sumOddjobMny = oddjobVO.getSumOddjobMny();
                //含本期累计零工金额
                BigDecimal curSumOddjobMny = ComputeUtil.safeAdd(jobTaxMny,sumOddjobMny);

                    if(curSumOddjobMny.compareTo(comMny)>0){
                        ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                        paramsCheckDsVO.setOrgName(datum.getOrgName());
                        paramsCheckDsVO.setWarnItem("零工超合同");
                        paramsCheckDsVO.setWarnName("累计零工金额大于合同金额");
                        StringBuffer stringBuffer = new StringBuffer();
                        stringBuffer.append("本次零工金额：").append(jobTaxMny.setScale(2, BigDecimal.ROUND_HALF_UP)).append("元，含本次累计零工金额：").append(curSumOddjobMny.setScale(2, BigDecimal.ROUND_HALF_UP)).append("元，合同金额*").append(roleValue).append("%:").append(comMny.setScale(2, BigDecimal.ROUND_HALF_UP)).append("元。超出金额：").append(ComputeUtil.safeSub(curSumOddjobMny,comMny).setScale(2, BigDecimal.ROUND_HALF_UP)).append("元");
                        paramsCheckDsVO.setContent(stringBuffer.toString());
                        checkDsVOS.add(paramsCheckDsVO);
                        updateParamsCheckVOMap(paramsArray, paramsCheckVOMap, datum, paramsCheckDsVO);
                    }
                }
            }
        }
        //处理刚性、柔性预警
        ParamsCheckVO pc = new ParamsCheckVO();
        if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("alert"))) {
            pc.setWarnType("alert");
            pc.setDataSource(paramsCheckVOMap.get("alert"));
        } else if (CollectionUtils.isNotEmpty(paramsCheckVOMap.get("warn"))) {
            pc.setWarnType("warn");
            pc.setDataSource(paramsCheckVOMap.get("warn"));
        } else {
            pc.setWarnType("none");
            pc.setDataSource(null);
        }
        return pc;
    }

    private Boolean checkFinishContract(OddjobVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ContractEntity::getId, finishVO.getContractId());
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }

    /**
     * 更新参数控制结果
     *
     * @param paramsArray      参数数组
     * @param paramsCheckVOMap 预警结果map
     * @param billParamVO      控制参数
     * @param paramsCheckDsVO  预警信息
     */
    private static void updateParamsCheckVOMap(String[] paramsArray, Map<String, List<ParamsCheckDsVO>> paramsCheckVOMap, BillParamVO billParamVO, ParamsCheckDsVO paramsCheckDsVO) {
        if ("alert".equals(paramsArray[billParamVO.getControlType()])) {
            List<ParamsCheckDsVO> alert = paramsCheckVOMap.get("alert");
            alert.add(paramsCheckDsVO);
        }
        if ("warn".equals(paramsArray[billParamVO.getControlType()])) {
            List<ParamsCheckDsVO> warn = paramsCheckVOMap.get("warn");
            warn.add(paramsCheckDsVO);
        }
    }
}
