package com.ejianc.business.sub.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.cost.api.ICostDetailApi;
import com.ejianc.business.cost.vo.CostDetailVO;
import com.ejianc.business.finance.api.IPayContractApi;
import com.ejianc.business.finance.util.MathUtil;
import com.ejianc.business.finance.vo.ParamsCheckDsVO;
import com.ejianc.business.finance.vo.ParamsCheckVO;
import com.ejianc.business.finance.vo.SumPayMnyVO;
import com.ejianc.business.sub.bean.*;
import com.ejianc.business.sub.mapper.SettleMapper;
import com.ejianc.business.sub.service.*;
import com.ejianc.business.sub.utils.TreeNodeBUtil;
import com.ejianc.business.sub.vo.*;
import com.ejianc.business.sub.vo.comparator.SettleDetailComparatorVo;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.api.IParamConfigApi;
import com.ejianc.foundation.support.vo.BillParamVO;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static java.math.BigDecimal.ROUND_HALF_DOWN;

/**
 * <p>
 * 分包过程结算表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("SettleServiceImpl")
public class SettleServiceImpl extends BaseServiceImpl<SettleMapper, SettleEntity> implements ISettleService {

    private static final String SUB_SETTLE_BILL_CODE = "SUB_SETTLE";
    private static final String SUB_SETTLE = "SUB_SETTLE";
    private static final String SUB_SETTLE_DETAIL = "SUB_SETTLE_DETAIL";
    private static final String SUB_SETTLE_ODDJOB = "SUB_SETTLE_ODDJOB";
    private static final String SUB_SETTLE_OTHER = "SUB_SETTLE_OTHER";
    private static final String CHECK_PARAM_CODE = "P-2452lK17";

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IParamConfigApi paramConfigApi;

    @Autowired
    private ISettleDetailService settleDetailService;

    @Autowired
    private ISettleOddjobService settleOddjobService;

    @Autowired
    private ISettleOtherService settleOtherService;

    @Autowired
    private IPayContractApi payContractApi;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private ICostDetailApi costDetailApi;

    @Autowired
    private SessionManager sessionManager;


    @Override
    public SettleVO insertOrUpdate(SettleVO settleVO) {
        if(checkFinishContract(settleVO)){
            throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
        }
        //保存时校验合同version是否一致
        if(settleVO.getContractVersion()!=null&&settleVO.getContractVersion()!=0){
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try{
                locked = RedisTool.tryLock(jedis, String.valueOf(settleVO.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------"+locked);
                if(locked){
                    ContractEntity contractEntity = contractService.selectById(settleVO.getContractId());
                    Integer version = contractEntity.getVersion()==null?0:contractEntity.getVersion();
                    Integer conVersion = settleVO.getContractVersion();
                    if(!version.equals(conVersion)){
                        throw new BusinessException("该合同已被更新，请刷新后重做！");
                    }
                }else {
                     throw new BusinessException("出现并发操作,请稍后重试！");
                }
            }catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(settleVO.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }

        }
        Long tenantId = InvocationInfoProxy.getTenantid();
        if(checkSameContract(settleVO)){
            throw new BusinessException("同一合同只能存在一份自由态或审批中的分包过程结算单");
        }
        if(StringUtils.isEmpty(settleVO.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_SETTLE_BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                settleVO.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        if(checkSameBillCode(settleVO)){
            throw new BusinessException("存在相同编码，不允许保存!");
        }

        settleVO.setRelationFlag("0");
        settleVO.setProportionFlag("0");
        SettleEntity settleEntity = BeanMapper.map(settleVO, SettleEntity.class);
        //保存前清空主键和父主键，重新生成
        List<SettleDetailEntity> beforeDetails = settleEntity.getSettleDetailList();
        if(CollectionUtils.isNotEmpty(beforeDetails) && null == settleEntity.getId()){
            for(SettleDetailEntity cdEntity : beforeDetails){
                cdEntity.setId(null);
                cdEntity.setParentId(null);
            }
        }
        LambdaQueryWrapper<SettleEntity> lambdachange2 = Wrappers.<SettleEntity>lambdaQuery();
        lambdachange2.eq(SettleEntity::getTenantId, tenantId);
        lambdachange2.eq(SettleEntity::getContractId,settleVO.getContractId());
        lambdachange2.eq(SettleEntity::getOrgId, settleVO.getOrgId());
        if(null!=settleVO.getId()&&settleVO.getId()>0){
            lambdachange2.ne(SettleEntity::getId,settleVO.getId());
        }
        lambdachange2.in(SettleEntity::getBillState,BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambdachange2.gt(SettleEntity ::getSettleDate,settleVO.getSettleDate());
        int num2 = super.count(lambdachange2);
        if(num2>0){
            throw new BusinessException("结算日期必须大于等于该合同对应最新结算日期!");
        }
        BigDecimal settlementTaxMny = settleVO.getSettleTaxMny()==null?BigDecimal.ZERO:settleVO.getSettleTaxMny();
        BigDecimal offsetMny = settleVO.getOffsetMny()==null?BigDecimal.ZERO:settleVO.getOffsetMny();//本次冲抵金额
        BigDecimal syMny = settleVO.getSymny()==null?BigDecimal.ZERO:settleVO.getSymny();//剩余冲抵金额
        if(offsetMny.compareTo(syMny)>0){
            throw new BusinessException("本次冲抵金额不允许大于剩余可冲抵金额!");
        }
        if(settlementTaxMny.compareTo(BigDecimal.ZERO) > 0 && offsetMny.compareTo(settlementTaxMny)>0){
            throw new BusinessException("本次冲抵金额不允许大于本期结算金额");
        }
        super.saveOrUpdate(settleEntity, false);

        List<SettleDetailEntity> settleDetailEntities = settleEntity.getSettleDetailList();
        if(CollectionUtils.isNotEmpty(settleDetailEntities)){
            Map<String,Long> idMap=new HashMap<>();
            for(SettleDetailEntity cdEntity : settleDetailEntities){
                idMap.put(cdEntity.getTid(),cdEntity.getId());
            }
            for(SettleDetailEntity cdEntity : settleDetailEntities){
                if(StringUtils.isNotEmpty(cdEntity.getTpid())){
                    cdEntity.setParentId(idMap.get(cdEntity.getTpid()));
                }
            }
            settleDetailService.saveOrUpdateBatch(settleDetailEntities, settleDetailEntities.size(), false);
        }

        return queryDetail(settleEntity.getId(), true);
    }

    private Boolean checkSameBillCode(SettleVO settleVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SettleEntity::getBillCode, settleVO.getBillCode());
        lambda.eq(SettleEntity::getTenantId, tenantId);
        if(null != settleVO.getId() && settleVO.getId() > 0){
            lambda.ne(SettleEntity::getId,settleVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    private Boolean checkSameContract(SettleVO settleVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<SettleEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(SettleEntity::getContractId, settleVO.getContractId());
        lambda.eq(SettleEntity::getOrgId, settleVO.getOrgId());
        lambda.notIn(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.eq(SettleEntity::getTenantId, tenantId);
        if(null != settleVO.getId() && settleVO.getId() > 0){
            lambda.ne(SettleEntity::getId,settleVO.getId());
        }
        return super.list(lambda).size() > 0;
    }

    @Override
    public SettleVO queryDetail(Long id, Boolean detailHasChildren) {
        SettleEntity settleEntity = super.selectById(id);

//        Map<String, LinkedHashMap<String, String>> subOrderMap = new HashMap<>();
//        LinkedHashMap<String, String> map = new LinkedHashMap<>();
//        map.put("treeIndex", QueryParam.ASC);
//        subOrderMap.put("settleDetailList", map);
//        SettleEntity settleEntity = super.selectById(id, subOrderMap);
        
        SettleVO settleVO =  BeanMapper.map(settleEntity, SettleVO.class);
        List<SettleOddjobVO> settleOddjobList = settleVO.getSettleOddjobList();
        if (settleOddjobList != null && settleOddjobList.size() > 0){
            for (SettleOddjobVO settleOddjobVO : settleOddjobList){
                settleOddjobVO.setRowState("edit");
            }
            settleVO.setSettleOddjobList(settleOddjobList);
        }
        BigDecimal c = settleVO.getContractTaxMny()==null?BigDecimal.ZERO:settleVO.getContractTaxMny();
        BigDecimal cc = settleVO.getSumSettleTaxMny()==null?BigDecimal.ZERO:settleVO.getSumSettleTaxMny();
        if(c.compareTo(BigDecimal.ZERO)!=0&&cc.compareTo(BigDecimal.ZERO)!=0){
            settleVO.setSumSettleScale(new BigDecimal(100).multiply(cc.divide(c,8,ROUND_HALF_DOWN)));
        }
        BigDecimal settleTaxMny = settleVO.getSettleTaxMny()==null?BigDecimal.ZERO:settleVO.getSettleTaxMny();
        if(c.compareTo(BigDecimal.ZERO)!=0&&settleTaxMny.compareTo(BigDecimal.ZERO)!=0){
            settleVO.setSettleScale(new BigDecimal(100).multiply(settleTaxMny.divide(c,8,ROUND_HALF_DOWN)));
        }
        /** 剩余冲抵金额 = 累计预付款 - 累计已冲抵 */
        BigDecimal sumPrePayMny =settleEntity.getSumPrepayMny()==null?BigDecimal.ZERO:settleEntity.getSumPrepayMny();
        BigDecimal sumOffsetMny =settleEntity.getSumOffsetMny()==null?BigDecimal.ZERO:settleEntity.getSumOffsetMny();
        settleVO.setSymny(sumPrePayMny.subtract(sumOffsetMny));
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("settleId", new Parameter(QueryParam.EQ, id));
//        queryParam.getOrderMap().put("treeIndex", "asc");
        List<SettleDetailEntity> detailEntityList = settleDetailService.queryList(queryParam, false);
        if(CollectionUtils.isNotEmpty(detailEntityList)) {
            for(SettleDetailEntity cdEntity : detailEntityList){
                cdEntity.setTid(cdEntity.getId().toString());
                cdEntity.setTpid(cdEntity.getParentId()!= null&&cdEntity.getParentId()>0?cdEntity.getParentId().toString():"");
                cdEntity.setRowState("edit");
            }
            List<SettleDetailVO> resultMapList = BeanMapper.mapList(detailEntityList, SettleDetailVO.class);
            // 排序
            Collections.sort(resultMapList, new SettleDetailComparatorVo());
            if(detailHasChildren){
                settleVO.setSettleDetailList(TreeNodeBUtil.buildTree(resultMapList));
            }else{
                settleVO.setSettleDetailList(resultMapList);
            }
        }
        return settleVO;
    }

    @Override
    public CommonResponse<IPage<SettleVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        IPage<SettleEntity> page = queryPage(param, false);
        List<SettleVO> settleVOList = BeanMapper.mapList(page.getRecords(), SettleVO.class);
        IPage<SettleVO> settleVOIPage = new Page<>();
        settleVOIPage.setCurrent(page.getCurrent());
        settleVOIPage.setRecords(settleVOList);
        settleVOIPage.setSize(page.getSize());
        settleVOIPage.setTotal(page.getTotal());
        settleVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", settleVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<SettleVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(vos.stream().map(SettleVO::getId).collect(Collectors.toList()), false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public SettleRecordVO queryDetailRecord(Long id) {
        ContractEntity contractEntity = contractService.selectById(id);
        SettleRecordVO settleRecordVO = new SettleRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        settleRecordVO.setContractId(id);
        settleRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        settleRecordVO.setContractTaxMny(contractTaxMny);

        List<Long> commonOrgIds = new ArrayList<>();
        Long orgId = InvocationInfoProxy.getOrgId();
        CommonResponse<List<OrgVO>> orgListResp = orgApi.findChildrenByParentId(orgId);
        if(!orgListResp.isSuccess()) {
            throw new BusinessException("表查询失败, 查询组织信息失败！");
        }
        commonOrgIds.addAll(orgListResp.getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        LambdaQueryWrapper<SettleEntity> lambda = Wrappers.<SettleEntity>lambdaQuery();
        lambda.eq(SettleEntity::getContractId,id);
        lambda.in(SettleEntity::getOrgId,commonOrgIds);
        lambda.in(SettleEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.orderByDesc(SettleEntity::getCreateTime);
        List<SettleEntity> entities = super.list(lambda);
        if(CollectionUtils.isNotEmpty(entities)) {
            List<SettleRecordDetailVO> settleRecordDetailVOList = BeanMapper.mapList(entities, SettleRecordDetailVO.class);
            final BigDecimal[] settlementMoney = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
            settleRecordDetailVOList.forEach(entity ->{
                settlementMoney[0] = settlementMoney[0].add(entity.getSettleTaxMny()==null? BigDecimal.ZERO:entity.getSettleTaxMny());
                BigDecimal dSumSettleTaxMny = entity.getSumSettleTaxMny() == null ? BigDecimal.ZERO : entity.getSumSettleTaxMny();
                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (dSumSettleTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
                entity.setSumScale(scale);
            });
            //累计结算金额
            settleRecordVO.setSumSettleTaxMny(settlementMoney[0]);
            BigDecimal sumSettlementTaxMny = settleRecordVO.getSumSettleTaxMny()==null?BigDecimal.ZERO:settleRecordVO.getSumSettleTaxMny();//累计结算金额
//            if(sumSettlementTaxMny.compareTo(BigDecimal.ZERO) > 0 && contractTaxMny.compareTo(BigDecimal.ZERO) > 0){
//                settleRecordVO.setSumScale(new BigDecimal(100).multiply(sumSettlementTaxMny.divide(contractTaxMny,8,ROUND_HALF_DOWN)));
//            }else{
//                settleRecordVO.setSumScale(BigDecimal.ZERO);
//            }
            settleRecordVO.setSumScale(new BigDecimal(100).multiply(sumSettlementTaxMny.divide(contractTaxMny,8,ROUND_HALF_DOWN)));
            settleRecordVO.setDetailList(settleRecordDetailVOList);
        }
        return settleRecordVO;
    }

    @Override
    public SettleVO queryDetailAdd(Long contractId) {
        ContractEntity contractEntity = contractService.selectById(contractId);
        SettleVO settleVO = BeanMapper.map(contractEntity, SettleVO.class);
        settleVO.setBillCode(null);
        settleVO.setEmployeeId(null);
        settleVO.setEmployeeName(null);
        settleVO.setContractId(contractEntity.getId());
        settleVO.setBillState(null);
        settleVO.setCreateUserCode(null);
        settleVO.setCreateTime(null);
        settleVO.setUpdateUserCode(null);
        settleVO.setUpdateTime(null);
        settleVO.setId(null);
        settleVO.setVersion(null);
        settleVO.setTaxMny(null);

        CommonResponse<SumPayMnyVO> sumPayMnyVO = payContractApi.getSumPayMny(contractId,contractEntity.getOrgId());
        if(!sumPayMnyVO.isSuccess()) {
            throw new BusinessException("获取付款信息失败，请刷新后再试！");
        }
        BigDecimal sumPayMny = sumPayMnyVO.getData().getSumPayMny();
        BigDecimal sumPrePayMny = sumPayMnyVO.getData().getSumPrePayMny();
        settleVO.setSumPayMny(sumPayMny==null?BigDecimal.ZERO:sumPayMny);
        settleVO.setSumPrepayMny(sumPrePayMny==null?BigDecimal.ZERO:sumPrePayMny);
        LambdaQueryWrapper<SettleEntity> lambdachange = Wrappers.<SettleEntity>lambdaQuery();
        lambdachange.eq(SettleEntity::getTenantId, InvocationInfoProxy.getTenantid());
        lambdachange.eq(SettleEntity::getContractId,contractId);
        lambdachange.eq(SettleEntity::getOrgId,contractEntity.getOrgId());
        lambdachange.in(SettleEntity::getBillState,BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        List<SettleEntity> list = super.list(lambdachange);
        if(CollectionUtils.isNotEmpty(list)){
            final BigDecimal[] settlementMoney = {new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP),new BigDecimal("0.00").setScale(2, BigDecimal.ROUND_HALF_UP)};
            list.forEach(e-> {
                settlementMoney[0] = settlementMoney[0].add(e.getSettleTaxMny()==null? BigDecimal.ZERO:e.getSettleTaxMny());
                settlementMoney[1] = settlementMoney[1].add(e.getOffsetMny()==null? BigDecimal.ZERO:e.getOffsetMny());
            });
            //累计结算金额
            settleVO.setSumSettleTaxMny(settlementMoney[0]);
            //累计冲抵金额
            settleVO.setSumOffsetMny(settlementMoney[1]);
        }
        BigDecimal sumOffsetMnys = settleVO.getSumOffsetMny()==null?BigDecimal.ZERO:settleVO.getSumOffsetMny();
        if(settleVO.getSumPrepayMny().compareTo(BigDecimal.ZERO)>=0&&sumOffsetMnys.compareTo(BigDecimal.ZERO)>=0){
            settleVO.setSymny(settleVO.getSumPrepayMny().subtract(sumOffsetMnys));
        }

        BigDecimal c = settleVO.getContractTaxMny()==null?BigDecimal.ZERO:settleVO.getContractTaxMny();
        BigDecimal cc = settleVO.getSumSettleTaxMny()==null?BigDecimal.ZERO:settleVO.getSumSettleTaxMny();
        if(c.compareTo(BigDecimal.ZERO)>0&&cc.compareTo(BigDecimal.ZERO)>0){
            settleVO.setSumSettleScale(new BigDecimal(100).multiply(cc.divide(c,8,ROUND_HALF_DOWN)));
        }
        return settleVO;
    }

    @Override
    public JSONObject querySubSettleVOList(QueryParam param, boolean isEs) {
        Page<SubSettleVO> pages = new Page<>((long)param.getPageIndex(), (long)param.getPageSize());
        QueryWrapper wrapper = changeToQueryWrapper(param);
        List<SubSettleVO>  list = baseMapper.querySubSettleVOList(pages, wrapper);
        JSONObject page = new JSONObject();
        page.put("records", list);
        page.put("total", pages.getTotal());
        page.put("current", pages.getCurrent());
        page.put("size", pages.getSize());
        page.put("pages", pages.getPages());
        return page;
    }

    @Override
    public  CommonResponse<SettleVO> pushCost(SettleVO settleVO) {
        SettleEntity settleEntity = baseMapper.selectById(settleVO.getId());
        if (CollectionUtils.isNotEmpty(settleVO.getSettleDetailList())){
            List<SettleDetailEntity> detailEntities = BeanMapper.mapList(settleVO.getSettleDetailList(), SettleDetailEntity.class);
            settleEntity.setSettleDetailList(detailEntities);
        }
        if (CollectionUtils.isNotEmpty(settleVO.getSettleOddjobList())){
            List<SettleOddjobEntity> oddjobEntities = BeanMapper.mapList(settleVO.getSettleOddjobList(), SettleOddjobEntity.class);
            settleEntity.setSettleOddjobList(oddjobEntities);
        }
        if (CollectionUtils.isNotEmpty(settleVO.getSettleOtherList())){
            List<SettleOtherEntity> otherEntities = BeanMapper.mapList(settleVO.getSettleOtherList(), SettleOtherEntity.class);
            settleEntity.setSettleOtherList(otherEntities);
        }
        super.saveOrUpdate(settleEntity,false);
        //推送数据
        if (settleEntity.getBillState().equals(BillStateEnum.COMMITED_STATE.getBillStateCode()) || settleEntity.getBillState().equals(BillStateEnum.PASSED_STATE.getBillStateCode())) {
            costPush(settleEntity);
        }
        return CommonResponse.success(queryDetail(settleVO.getId(), true));
    }

    @Override
    public void costPush(SettleEntity settleEntity) {
        Boolean isPush = true;
        LambdaQueryWrapper<SettleDetailEntity> detailWrapper = new LambdaQueryWrapper<>();
        detailWrapper.eq(SettleDetailEntity::getSettleId, settleEntity.getId());
        List<SettleDetailEntity> detailEntities = settleDetailService.list(detailWrapper);
        LambdaQueryWrapper<SettleOddjobEntity> oddjobWrapper = new LambdaQueryWrapper<>();
        oddjobWrapper.eq(SettleOddjobEntity::getSettleId, settleEntity.getId());
        List<SettleOddjobEntity> oddjobEntities = settleOddjobService.list(oddjobWrapper);
        LambdaQueryWrapper<SettleOtherEntity> othWrapper = new LambdaQueryWrapper<>();
        othWrapper.eq(SettleOtherEntity::getSettleId, settleEntity.getId());
        List<SettleOtherEntity> otherEntities = settleOtherService.list(othWrapper);
        if(CollectionUtils.isEmpty(detailEntities) && CollectionUtils.isEmpty(oddjobEntities) && CollectionUtils.isEmpty(otherEntities)){
            isPush = false;
        }else if(CollectionUtils.isNotEmpty(detailEntities) && isPush){
            for(SettleDetailEntity detailEntity : detailEntities){
                if(detailEntity.getLeafFlag() && null == detailEntity.getSubjectId() && null == detailEntity.getAccountingId()){
                    isPush = false;
                    break;
                }
            }
        }else if(CollectionUtils.isNotEmpty(oddjobEntities) && isPush){
            for(SettleOddjobEntity oddjobEntity : oddjobEntities){
                if(null == oddjobEntity.getSubjectId() && null == oddjobEntity.getAccountingId()){
                    isPush = false;
                    break;
                }
            }
        }else if(CollectionUtils.isNotEmpty(otherEntities) && isPush){
            for(SettleOtherEntity otherEntity : otherEntities){
                if(null == otherEntity.getSubjectId() && null == otherEntity.getAccountingId()){
                    isPush = false;
                    break;
                }
            }
        }

        settleEntity.setSettleDetailList(detailEntities);
        settleEntity.setSettleOddjobList(oddjobEntities);
        settleEntity.setSettleOtherList(otherEntities);

        //更新是否关联
        LambdaUpdateWrapper<SettleEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(SettleEntity::getId, settleEntity.getId());
        updateWrapper.set(SettleEntity::getRelationFlag, isPush ? "1":"0");//(1:是，0：否)
        super.update(updateWrapper);

        //判断之前的单据是否关联
        String oldRelationFlag = settleEntity.getRelationFlag();
        //之前已关联
        if (oldRelationFlag.equals("1")){
            if (isPush){
                saveCost(settleEntity);
            }
            if (!isPush){
                //删除成本中心之前的数据
                costDetailApi.deleteSubject(settleEntity.getId());
            }
        }
        //之前未关联
        if (oldRelationFlag.equals("0")){
            if (isPush){
                saveCost(settleEntity);
            }
        }

    }

    @Override
    public void pullCost(Long id) {
        //更新是否关联
        LambdaUpdateWrapper<SettleEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(SettleEntity::getId, id);
        updateWrapper.set(SettleEntity::getRelationFlag, "0");//(1:是，0：否)
        super.update(updateWrapper);

        costDetailApi.deleteSubject(id);
    }

    @Override
    public ParamsCheckVO checkParams(Integer purchaseType, BigDecimal sumSettleMentTaxMny, BigDecimal contractTaxMny) {
        // 三种控制方式：不控制，提醒，无法保存 (默认为提醒)
        String[] paramsArray = {"none", "warn", "alert"};
        ParamsCheckVO paramsCheckVO = new ParamsCheckVO();
        List<ParamsCheckDsVO> checkDsVOS = new ArrayList<>();
        CommonResponse<BillParamVO> billParamByCode = paramConfigApi.getBillParamByCode(CHECK_PARAM_CODE);
        if(billParamByCode.isSuccess() && null != billParamByCode.getData()){
            BillParamVO billParamVO = billParamByCode.getData();
            BigDecimal roleValue = billParamVO.getRoleValue();
            BigDecimal comMny = contractTaxMny.multiply(roleValue.divide(BigDecimal.valueOf(100)
                    , 2, BigDecimal.ROUND_HALF_UP));

            if (2 == purchaseType) {
                // 集采合同直接赋值为不控制
                paramsCheckVO.setWarnType(paramsArray[0]);
                return  paramsCheckVO;
            }else {
                paramsCheckVO.setWarnType(paramsArray[billParamVO.getControlType()]);
            }

            if (sumSettleMentTaxMny.compareTo(comMny) > 0) {
                ParamsCheckDsVO paramsCheckDsVO = new ParamsCheckDsVO();
                paramsCheckDsVO.setWarnItem("合同超结");
                paramsCheckDsVO.setWarnName("结算金额大于合同金额");
                StringBuffer stringBuffer = new StringBuffer();
                stringBuffer.append("本次结算金额：").append(sumSettleMentTaxMny.toString())
                        .append("元，合同金额*").append(roleValue).append("%:")
                        .append(comMny).append("元。超出金额：")
                        .append(sumSettleMentTaxMny.subtract(comMny)).append("元");
                paramsCheckDsVO.setContent(stringBuffer.toString());
                checkDsVOS.add(paramsCheckDsVO);
            }
            paramsCheckVO.setDataSource(checkDsVOS);
        }else {
            logger.info(billParamByCode.getMsg());
            throw new BusinessException("获取控制参数失败");
        }

        return paramsCheckVO;
    }

    @Override
    public List<SettleDetailResultVO> querySettleDetail(Long contractId) {
        return baseMapper.querySettleDetail(contractId);
    }

    private void saveCost(SettleEntity settleEntity) {
        //税率
        BigDecimal taxRate = settleEntity.getTaxRate().divide(new BigDecimal(100));
        BigDecimal number = taxRate.add(new BigDecimal(1));
        //明细
        List<CostDetailVO> costDetailVOList = new ArrayList<CostDetailVO>();
        List<SettleDetailEntity> detailEntities = settleEntity.getSettleDetailList();
        if (CollectionUtils.isNotEmpty(detailEntities)){
            for (SettleDetailEntity detailEntity : detailEntities){
                if(detailEntity.getSubjectId() != null && detailEntity.getAccountingId() != null){
                    CostDetailVO costDetailVO = new CostDetailVO();
                    costDetailVO.setSubjectId(detailEntity.getSubjectId());
                    costDetailVO.setSourceId(detailEntity.getSettleId());
                    costDetailVO.setSourceDetailId(detailEntity.getId());
                    costDetailVO.setHappenMny(detailEntity.getSettleMny());
                    costDetailVO.setHappenTaxMny(detailEntity.getSettleTaxMny());
                    costDetailVO.setHappenDate(settleEntity.getSettleDate());
                    costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                    costDetailVO.setSourceType(SUB_SETTLE);
                    costDetailVO.setSourceTabType(SUB_SETTLE_DETAIL);
                    costDetailVO.setProjectId(settleEntity.getProjectId());
                    // 核算对象
                    costDetailVO.setAccountingId(detailEntity.getAccountingId());
                    costDetailVO.setOrgSubjectId(detailEntity.getOrgSubjectId());
                    costDetailVO.setOrgAccountingId(detailEntity.getOrgAccountingId());
                    costDetailVOList.add(costDetailVO);
                }
            }
        }
        //费用
        List<SettleOddjobEntity> oddjobList = settleEntity.getSettleOddjobList();
        if (CollectionUtils.isNotEmpty(oddjobList)){
            for (SettleOddjobEntity oddjobEntity : oddjobList){
                if (oddjobEntity.getSubjectId() != null && oddjobEntity.getAccountingId() != null) {
                    CostDetailVO costDetailVO = new CostDetailVO();
                    costDetailVO.setSubjectId(oddjobEntity.getSubjectId());
                    costDetailVO.setSourceId(oddjobEntity.getSettleId());
                    costDetailVO.setSourceDetailId(oddjobEntity.getId());
                    costDetailVO.setHappenTaxMny(oddjobEntity.getSettleTaxMny());
                    costDetailVO.setHappenMny(oddjobEntity.getSettleMny());
                    costDetailVO.setHappenDate(settleEntity.getSettleDate());
                    costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                    costDetailVO.setSourceType(SUB_SETTLE);
                    costDetailVO.setSourceTabType(SUB_SETTLE_ODDJOB);
                    costDetailVO.setProjectId(settleEntity.getProjectId());
                    // 核算对象
                    costDetailVO.setAccountingId(oddjobEntity.getAccountingId());
                    costDetailVO.setOrgSubjectId(oddjobEntity.getOrgSubjectId());
                    costDetailVO.setOrgAccountingId(oddjobEntity.getOrgAccountingId());
                    costDetailVOList.add(costDetailVO);
                }
            }
        }

        //费用
        List<SettleOtherEntity> otherEntityList = settleEntity.getSettleOtherList();
        if (CollectionUtils.isNotEmpty(otherEntityList)){
            for (SettleOtherEntity otherEntity : otherEntityList){
                if (otherEntity.getSubjectId() != null && otherEntity.getAccountingId() != null) {
                    CostDetailVO costDetailVO = new CostDetailVO();
                    costDetailVO.setSubjectId(otherEntity.getSubjectId());
                    costDetailVO.setSourceId(otherEntity.getSettleId());
                    costDetailVO.setSourceDetailId(otherEntity.getId());
                    costDetailVO.setHappenTaxMny(otherEntity.getTaxMoney());
                    costDetailVO.setHappenMny(otherEntity.getMoney());
                    costDetailVO.setHappenDate(settleEntity.getSettleDate());
                    costDetailVO.setCreateUserName(sessionManager.getUserContext().getUserName());
                    costDetailVO.setSourceType(SUB_SETTLE);
                    costDetailVO.setSourceTabType(SUB_SETTLE_OTHER);
                    costDetailVO.setProjectId(settleEntity.getProjectId());
                    // 核算对象
                    costDetailVO.setAccountingId(otherEntity.getAccountingId());
                    costDetailVO.setOrgSubjectId(otherEntity.getOrgSubjectId());
                    costDetailVO.setOrgAccountingId(otherEntity.getOrgAccountingId());
                    costDetailVOList.add(costDetailVO);
                }
            }
        }

        //成本中心
        if (CollectionUtils.isNotEmpty(costDetailVOList)){
            CommonResponse<String> stringCommonResponse = costDetailApi.saveSubject(costDetailVOList);
            if (stringCommonResponse.isSuccess()){
            }else {
                throw new BusinessException(stringCommonResponse.getMsg());
            }
        }

    }
    private Boolean checkFinishContract(SettleVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ContractEntity::getId, finishVO.getContractId());
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }
}
