package com.ejianc.business.sub.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.sub.bean.ContractEntity;
import com.ejianc.business.sub.bean.OddjobEntity;
import com.ejianc.business.sub.mapper.OddjobMapper;
import com.ejianc.business.sub.service.IContractService;
import com.ejianc.business.sub.service.IOddjobService;
import com.ejianc.business.sub.vo.*;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.core.util.ExcelExport;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 分包零星用工表 服务实现类
 * </p>
 *
 * @author zhangwx
 * @since 2020-06-05
 */
@Service("OddjobServiceImpl")
public class OddjobServiceImpl extends BaseServiceImpl<OddjobMapper, OddjobEntity> implements IOddjobService {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private static final String SUB_ODDJOB_BILL_CODE = "SUB_ODDJOB";

    @Autowired
    private IContractService contractService;

    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private IOrgApi orgApi;
    @Autowired
    private JedisPool jedisPool;

    @Override
    public OddjobVO insertOrUpdate(OddjobVO oddjobVO) {
        if(checkFinishContract(oddjobVO)){
            throw new BusinessException("该合同在相同组织下已经完工，不允许新增！");
        }
        ContractEntity contractEntity = contractService.selectById(oddjobVO.getContractId());
        //保存时校验合同version是否一致
        if(oddjobVO.getContractVersion()!=null&&oddjobVO.getContractVersion()!=0){
            Jedis jedis = jedisPool.getResource();
            boolean locked = false;
            try{
                locked = RedisTool.tryLock(jedis, String.valueOf(oddjobVO.getContractId()), "saveOrUpdate", 1000);
                logger.info("判断单据单据锁结果------"+locked);
                if(locked){
                    Integer version = contractEntity.getVersion()==null?0:contractEntity.getVersion();
                    Integer conVersion = oddjobVO.getContractVersion();
                    if(version!=conVersion){
                        throw new BusinessException("分包合同已被更新，请刷新后重做！");
                    }
                }else{
                    throw new BusinessException("出现并发操作,请稍后重试！");
                }
            }catch (Exception e) {
                e.printStackTrace();
            } finally {
                if(locked) {
                    RedisTool.releaseLock(jedis, String.valueOf(oddjobVO.getContractId()), "saveOrUpdate");
                }
                jedis.close();
            }
        }

        if(StringUtils.isEmpty(oddjobVO.getBillCode())){
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SUB_ODDJOB_BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                oddjobVO.setBillCode(billCode.getData());
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        OddjobEntity oddjobEntity = BeanMapper.map(oddjobVO, OddjobEntity.class);
        super.saveOrUpdate(oddjobEntity, false);
        return queryDetail(oddjobEntity.getId());
    }

    @Override
    public OddjobVO queryDetail(Long id) {
        OddjobEntity oddjobEntity = super.selectById(id);
        OddjobVO oddjobVO =  BeanMapper.map(oddjobEntity, OddjobVO.class);
        return oddjobVO;
    }

    @Override
    public CommonResponse<IPage<OddjobVO>> queryListVOs(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        IPage<OddjobEntity> page = queryPage(param, false);
        List<OddjobVO> oddjobVOList = BeanMapper.mapList(page.getRecords(), OddjobVO.class);
        IPage<OddjobVO> oddjobVOIPage = new Page<>();
        oddjobVOIPage.setCurrent(page.getCurrent());
        oddjobVOIPage.setRecords(oddjobVOList);
        oddjobVOIPage.setSize(page.getSize());
        oddjobVOIPage.setTotal(page.getTotal());
        oddjobVOIPage.setPages(page.getPages());
        return CommonResponse.success("查询成功！", oddjobVOIPage);
    }

    @Override
    public CommonResponse<String> deleteByIds(List<OddjobVO> vos) {
        if(CollectionUtils.isNotEmpty(vos)) {
            this.removeByIds(vos.stream().map(OddjobVO::getId).collect(Collectors.toList()), false);
        }
        return CommonResponse.success("删除成功！");
    }

    @Override
    public OddjobVO queryDetailAdd(Long contractId) {
        ContractEntity contractEntity = contractService.selectById(contractId);
        OddjobVO oddjobVO = BeanMapper.map(contractEntity, OddjobVO.class);
        oddjobVO.setBillCode(null);
        oddjobVO.setEmployeeId(null);
        oddjobVO.setEmployeeName(null);
        oddjobVO.setContractId(contractEntity.getId());
        oddjobVO.setBillState(null);
        oddjobVO.setCreateUserCode(null);
        oddjobVO.setCreateTime(null);
        oddjobVO.setUpdateUserCode(null);
        oddjobVO.setUpdateTime(null);
        oddjobVO.setId(null);
        return oddjobVO;
    }

    @Override
    public OddjobRecordVO queryDetailRecord(Long id) {
        ContractEntity contractEntity = contractService.selectById(id);
        OddjobRecordVO oddjobRecordVO = new OddjobRecordVO();
        BigDecimal contractTaxMny = contractEntity.getContractTaxMny() == null ? BigDecimal.ZERO : contractEntity.getContractTaxMny();
        //BigDecimal sumOddjobTaxMny = contractEntity.getSumOddjobTaxMny() == null ? BigDecimal.ZERO : contractEntity.getSumOddjobTaxMny();
        oddjobRecordVO.setContractId(id);
        oddjobRecordVO.setFinishFlag(contractEntity.getFinishFlag());
        oddjobRecordVO.setContractTaxMny(contractTaxMny);



        LambdaQueryWrapper<OddjobEntity> lambda = Wrappers.<OddjobEntity>lambdaQuery();
        lambda.eq(OddjobEntity::getContractId,id);
        lambda.eq(OddjobEntity::getTenantId,InvocationInfoProxy.getTenantid());
        lambda.in(OddjobEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(),BillStateEnum.COMMITED_STATE.getBillStateCode());
        lambda.in(OddjobEntity::getOrgId, orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList()));
        List<OddjobEntity> entities = super.list(lambda);
        List<OddjobRecordDetailVO> oddjobRecordDetailVOList = BeanMapper.mapList(entities, OddjobRecordDetailVO.class);
        BigDecimal oddjobTaxMny = BigDecimal.ZERO;
        BigDecimal settleTaxMny = BigDecimal.ZERO;
        if(CollectionUtils.isNotEmpty(oddjobRecordDetailVOList)) {
            for(OddjobRecordDetailVO ojr:oddjobRecordDetailVOList){
                BigDecimal jobTaxMny = ojr.getJobTaxMny() == null ? BigDecimal.ZERO : ojr.getJobTaxMny();
                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (jobTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
                ojr.setScale(scale);

                oddjobTaxMny = oddjobTaxMny.add(jobTaxMny);
                BigDecimal oddjobSettleTaxMny = ojr.getOddjobSettleTaxMny() == null ? BigDecimal.ZERO : ojr.getOddjobSettleTaxMny();
                settleTaxMny = settleTaxMny.add(oddjobSettleTaxMny);
            }
//            oddjobRecordDetailVOList.forEach(entity ->{
//                BigDecimal jobTaxMny = entity.getJobTaxMny() == null ? BigDecimal.ZERO : entity.getJobTaxMny();
//                BigDecimal scale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (jobTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
//                entity.setScale(scale);
//            });
        }
        oddjobRecordVO.setDetailList(oddjobRecordDetailVOList);

        oddjobRecordVO.setSumOddjobTaxMny(oddjobTaxMny);
        oddjobRecordVO.setSumOddjobSettleTaxMny(settleTaxMny);
        BigDecimal sumScale = contractTaxMny == BigDecimal.ZERO ? BigDecimal.ZERO : (oddjobTaxMny.divide(contractTaxMny, BigDecimal.ROUND_HALF_UP)).multiply(new BigDecimal(100));
        oddjobRecordVO.setSumScale(sumScale);
        Boolean editFlag = contractService.editFlag(id);
        oddjobRecordVO.setEditFlag(editFlag);
        return oddjobRecordVO;
    }

    @Override
    public void updateSettleByPks(List<Long> pks, Boolean flag) {
        LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
        updateWrapper.set(OddjobEntity::getSettleFlag, flag);
        updateWrapper.in(OddjobEntity::getId, pks);
        this.update(updateWrapper);
    }
    @Override
    public void updateSettleMnyByPks(List<SettleOddjobVO> list, Boolean flag) {
        if(null!=list&&list.size()>0){
            for(SettleOddjobVO vo :list){
                LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
                updateWrapper.set(OddjobEntity::getSettleFlag, flag);
                updateWrapper.set(OddjobEntity::getOddjobSettleTaxMny, vo.getSettleMny());
                updateWrapper.eq(OddjobEntity::getId, vo.getOddjobId());
                this.update(updateWrapper);
            }
        }
    }
    @Override
    public void updateSettleMny(List<FinishOddjobVO> list, Boolean flag) {
        if(null!=list&&list.size()>0){
            for(FinishOddjobVO vo :list){
                LambdaUpdateWrapper<OddjobEntity> updateWrapper = Wrappers.lambdaUpdate();
                updateWrapper.set(OddjobEntity::getSettleFlag, flag);
                updateWrapper.set(OddjobEntity::getOddjobSettleTaxMny, vo.getSettleMny());
                updateWrapper.eq(OddjobEntity::getId, vo.getOddjobId());
                this.update(updateWrapper);
            }
        }
    }

    @Override
    public void excelExport(QueryParam param, HttpServletResponse response) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("orgName");
        fuzzyFields.add("billCode");
        fuzzyFields.add("contractName");
//        fuzzyFields.add("customerName");
        fuzzyFields.add("supplierName");
        fuzzyFields.add("employeeName");
        param.getParams().put("tenant_id",new Parameter("eq",InvocationInfoProxy.getTenantid()));
        param.getParams().put("org_id",new Parameter("in",orgApi.findChildrenByParentId(InvocationInfoProxy.getOrgId()).getData().stream().map(OrgVO::getId).collect(Collectors.toList())));
        param.setPageIndex(1);
        param.setPageSize(-1);
        IPage<OddjobEntity> pageData = queryPage(param, false);

        Map<String, Object> beans = new HashMap<String, Object>();
        if (null != pageData.getRecords() && CollectionUtils.isNotEmpty(pageData.getRecords())) {
            List<OddjobVO> list = BeanMapper.mapList(pageData.getRecords(), OddjobVO.class);
            list.forEach(vo -> {
                vo.setBillStateName(BillStateEnum.getEnumByStateCode(vo.getBillState()).getDescription());
            });
            beans.put("records", list);
            ExcelExport.getInstance().export("oddjob-export.xlsx", beans, response);
        }
    }
    private Boolean checkFinishContract(OddjobVO finishVO) {
        Long tenantId = InvocationInfoProxy.getTenantid();
        LambdaQueryWrapper<ContractEntity> lambda = new LambdaQueryWrapper<>();
        lambda.eq(ContractEntity::getId, finishVO.getContractId());
        lambda.eq(ContractEntity::getTenantId, tenantId);
        lambda.eq(ContractEntity::getFinishFlag, true);
        return contractService.list(lambda).size() > 0;
    }
}
