package com.ejianc.business.sub.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.pub.utils.HttpTookit;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.foundation.support.api.*;
import com.ejianc.foundation.support.vo.BankAccountVO;
import com.ejianc.foundation.support.vo.BillTypeVO;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.fields.service.ICommenQueryFieldsService;
import com.ejianc.framework.skeleton.refer.constants.MetaDataUrlconstants;
import com.ejianc.framework.skeleton.refer.util.ContextUtil;
import com.ejianc.framework.skeleton.refer.util.ExceptionUtil;
import com.ejianc.framework.skeleton.refer.util.ReferHttpClientUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @date 2021-12-01 10:42:16
 */
@Component
public class NCUtil {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICommenQueryFieldsService commenQueryFieldsService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private ICustomerApi customerApi;

    @Autowired
    private ISupplierApi supplierApi;

    @Autowired
    private ISupplierWebApi supplierWebApi;

    @Autowired
    private IDefdocApi defdocApi;

    @Autowired
    private IBillTypeApi billTypeApi;

    @Autowired
    private IPushMessageApi pushMessageApi;

    @Value("${nc.ncArUrl}")
    private String ncArUrl;

    @Value("${common.env.base-host}")
    private String BASE_HOST;//运行环境域名

    private static final String REFER_CACHE_KEY = "refer_cache_key:";

    private final static Map<Integer, String> invoiceMainMap;// 开票主体分类

    private final static Map<Integer, String> taxWayMap;// 计税方式

    static {
        invoiceMainMap = new HashMap<>();
        invoiceMainMap.put(0, "PMDA06001");// 以集团名义开票
        invoiceMainMap.put(1, "PMDA06002");// 以区域名义开票

        taxWayMap = new HashMap<>();
        taxWayMap.put(0, "01");// 一般纳税人
        taxWayMap.put(1, "02");// 小规模纳税人
        taxWayMap.put(2, "03");// 其它
    }

    /**
     * 开票主体分类自定义档案编码
     * @param key
     * @return
     */
    public String getInvoiceMainDefCode(Integer key){
        if(key == null){
            throw new BusinessException("项目-开票主体不能为空");
        }
        if(invoiceMainMap.containsKey(key)){
            return invoiceMainMap.get(key);
        }
        return null;
    }

    /**
     * 计税方式自定义档案编码
     * @param key
     * @return
     */
    public String getTaxWayDefCode(Integer key){
        if(key == null){
            throw new BusinessException("项目-计税方式不能为空");
        }
        if(taxWayMap.containsKey(key)){
            return taxWayMap.get(key);
        }
        return null;
    }

    /**
     * 税率参照
     * @param taxRate
     * @return
     */
    public String getTaxRateDefCode(BigDecimal taxRate){
        if(taxRate == null){
            throw new BusinessException("税率不能为空");
        }
        List<DefdocDetailVO> defList = this.getDefDocListById(515260415240835083L);
        Map<String, String> defMap = defList.stream().collect(Collectors.toMap(x->x.getName(), DefdocDetailVO::getCode));
        String tax = taxRate.stripTrailingZeros().toPlainString() + "%";
        if(defMap.containsKey(tax)){
            return defMap.get(tax);
        } else {
            throw new BusinessException("录入税率“" + tax + "”和财务税率不一致，请联系系统管理员处理！");
        }
    }

    /**
     * 根据分类id查询分类下所有自定义档案
     * @param id
     * @return
     */
    public List<DefdocDetailVO> getDefDocListById(Long id){
        CommonResponse<List<DefdocDetailVO>> resp = defdocApi.getDefDocByDefId(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询自定义档案
     * @param id
     * @return
     */
    public DefdocDetailVO getDefDocById(Long id){
        if(id == null) return new DefdocDetailVO();
        CommonResponse<DefdocDetailVO> resp = defdocApi.getDefDocById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询组织
     * @param id
     * @return
     */
    public OrgVO queryOrgById(Long id){
        CommonResponse<OrgVO> resp = orgApi.detailById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id集合查询项目列表
     * @param ids
     * @return
     */
    public List<ProjectRegisterVO> queryProjectByIds(List<Long> ids){
        CommonResponse<List<ProjectRegisterVO>> resp = projectApi.queryProjectByIds(ids);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询供应商
     * @param id
     * @return
     */
    public SupplierVO querySupplierById(Long id){
        CommonResponse<SupplierVO> resp = supplierApi.queryById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询客户
     * @param id
     * @return
     */
    public CustomerVO queryCustomerById(Long id){
        CommonResponse<CustomerVO> resp = customerApi.detailById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询默认银行账户
     * @param id
     * @param type  supplierId or customerId or insideOrgId
     * @return
     */
    public BankAccountVO queryDefaultBankAccountById(Long id, String type){
        CommonResponse<BankAccountVO> resp = supplierWebApi.queryDefaultBankAccountById(id, type);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 如果道桥分公司本下，且以区域分公司开票，则传项目所属组织和道桥分公司，如果以集团名义开票，则传道桥分公司
     * 其他组织，还传项目所属组织
     * @param orgId 项目所属组织
     * @param invoiceMain 开票主体分类 0-以集团名义开票，1-以区域分公司名义开票
     * @return List<Long> 返回几个执行几次，财务组织使用对应的返回值
     */
    public List<Long> validateDaoQiaoOrg(Long orgId, Integer invoiceMain){
        Long daoQiaoId = 1406795679766782028L;// 青岛市政空间开发集团道桥分公司
        OrgVO orgVO = this.queryOrgById(orgId);
        // 道桥分公司本下
        if(orgVO.getInnerCode().contains(String.valueOf(daoQiaoId))) {
            if(new Integer(0).equals(invoiceMain) || daoQiaoId.equals(orgId)){
                return Arrays.asList(daoQiaoId);
            } else if(new Integer(1).equals(invoiceMain)){
                return Arrays.asList(orgId, daoQiaoId);
            }
        }
        return Arrays.asList(orgId);
    }

    /**
     * 校验项目所属组织和开票主体是否一致
     * @param orgId
     * @param mainOrgId
     * @return
     */
    public List<Long> validateInvoiceMainOrg(Long orgId, Long mainOrgId){
        if(!orgId.equals(mainOrgId)){
            return Arrays.asList(orgId, mainOrgId);
        }
        return Arrays.asList(orgId);
    }

    /**
     *  调用NC传凭证接口（应收、应付、收款、付款）
     * @param param
     * @return
     */
    public String postByJson(String param){
        Map<String, String> headers = new HashMap<>();
        logger.info("-传NC凭证入参:" + param);
        String back = null;
        try {
            back = HttpTookit.postByJson(ncArUrl, param, headers);
        } catch (Exception e) {
            try {
                back = HttpTookit.postByJson(ncArUrl, param, headers);
            } catch (Exception e1) {
                throw new BusinessException("推送NC接口异常!");
            }
        }
        logger.info("-传NC凭证回参:" + back);
        if(!back.contains("code")){
            throw new BusinessException("返回值解析异常!");
        }
        JSONObject json = JSONObject.parseObject(back);
        if(!"200".equals(json.getString("code"))){
            throw new BusinessException(json.getString("msg"));
        }else{
            return json.getString("result");
        }
    }

    /**
     *  调用NC联查凭证接口（应收、应付、收款、付款）
     * @param param
     * @return
     */
    public JSONObject postByJson2(String param){
        Map<String, String> headers = new HashMap<>();
        logger.info("--联查NC凭证入参:" + param);
        String back = null;
        try {
            back = HttpTookit.postByJson(ncArUrl, param, headers);
        } catch (Exception e) {
            try {
                back = HttpTookit.postByJson(ncArUrl, param, headers);
            } catch (Exception e1) {
                throw new BusinessException("推送NC接口异常!");
            }
        }
        logger.info("--联查NC凭证回参:" + back);
        JSONObject json = JSONObject.parseObject(back);
        if(json.containsKey("code") && !"200".equals(json.getString("code"))){
            throw new BusinessException(json.getString("msg"));
        }
        return json;
    }

    /**
     * 查询单据明细
     * @param businessKey 单据主键
     * @param billTypeCode 单据类型主键
     * @return
     */
    public JSONObject queryBillDetail(String businessKey, String billTypeCode){
        JSONObject json = commenQueryFieldsService.queryBillDetail(businessKey, billTypeCode);
        if(json.getInteger("code") != 0){
            throw new BusinessException(json.getString("msg"));
        }
        return json.getJSONObject("data");
    }

    /**
     * 生成外部跳转地址
     * @param url   除域名外的地址
     * @return
     */
    public String getWebSite(String url) {
        Long userid = InvocationInfoProxy.getUserid();
        String targeturl = BASE_HOST + url;// 目标路径
        logger.info("--->> targeturl: " + targeturl);
        try {
            targeturl = URLEncoder.encode(targeturl,"utf-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        String webSite = BASE_HOST + "portal/sso/index?userid=" + userid + "&targeturl=" + targeturl;
        logger.info("--->> 生成外部跳转地址: " + webSite);
        return webSite;
    }

    public void sendMsg(List<String> messageType, List<String> user, String msgType, String subject, String content, String mobileUrl) {
        logger.info("发送消息开始！===========");
        PushMsgParameter parameter = new PushMsgParameter();
        List<String> channel = new ArrayList<>();
        if (messageType.contains("sys")) {
            // 系统消息
            channel.add(PushMsgParameter.CHANNEL_TYPE_SYS);
        }
        if (messageType.contains("smsg")) {
            // 短信消息
            // channel.add(PushMsgParameter.CHANNEL_TYPE_NOTE);
        }
        if (messageType.contains("weChat")) {
            // 系统消息
            channel.add(PushMsgParameter.CHANNEL_TYPE_WEXINEE);
        }
        Collections.replaceAll(user, "1247777316689256450", "303581417601122400");// 系统管理员特殊处理
        String[] receivers = user.toArray(new String[user.size()]);
        parameter.setReceivers(receivers);// 收信人
        parameter.setChannel(channel.toArray(new String[channel.size()]));// 消息类型
        parameter.setMsgType(msgType);
        parameter.setSubject(subject);// 标题
        parameter.setContent(content);// 内容
        parameter.setTenantId(InvocationInfoProxy.getTenantid().toString());
        parameter.setSendUserId(InvocationInfoProxy.getUserid());
        if (messageType.contains("weChat")) {
            JSONObject weixineeParams = new JSONObject();
//            weixineeParams.put("agentid", cgddAgentid);
//            weixineeParams.put("secret", cgddSecret);
//            weixineeParams.put("content", subject);
            weixineeParams.put("msgtype", "textcard");
            weixineeParams.put("title", "消息提醒");
            weixineeParams.put("description", subject);
            weixineeParams.put("url", mobileUrl);
            weixineeParams.put("btntxt", "点击查看");
            parameter.setWeixineeParams(weixineeParams);
        }
        try {
            logger.info("发送信息{}", JSONObject.toJSONString(parameter));
            CommonResponse<String> result = pushMessageApi.pushMessage(parameter);
            if (result.isSuccess()) {
                logger.info("消息发送成功---------------->" + result.getMsg());
            } else {
                logger.info("消息发送失败---------------->" + result.getMsg());
            }
        } catch (Exception e) {
            logger.info("调用消息中心RPC服务异常--------------" + e);
        }
    }

    public String getCreateUserId(String userCode) {
        try {
            MetaDataUrlconstants urlconstants = ContextUtil.getBean(MetaDataUrlconstants.class);
            if (StringUtils.isNotBlank(userCode) && StringUtils.isNotBlank(urlconstants.getBaseHost())) {
                String cacheKey = REFER_CACHE_KEY + "idm-employee:" + userCode;
                RedisTemplate<String, Object> redisTemplate = ContextUtil.getBean("redisTemplate", RedisTemplate.class);
                Object cacheValue = redisTemplate.opsForValue().get(cacheKey);
                JSONObject jsonObject = null;
                if (cacheValue != null) {
                    jsonObject = JSON.parseObject(cacheValue.toString());
                } else {
                    String referData = getReferRestUrl("idm-employee");
                    JSONObject json = JSON.parseObject(referData);
                    String referUrl = json.get("projectName").toString();
                    Map<String, Object> reflist = new HashMap<>();
                    reflist = (Map<String, Object>) json.get("refMapList");
                    reflist.put("userCode", userCode);
                    String url = urlconstants.getBaseHost() + referUrl + "/commonrefer/getAuditInfo";
                    String jsonbackstr = null;
                    try {
                        jsonbackstr = ReferHttpClientUtils.getAndHeader(url, reflist);
                    } catch (Exception e) {
                    }
                    if (StringUtils.isNotBlank(jsonbackstr)) {
                        jsonObject = JSON.parseObject(jsonbackstr);
                        redisTemplate.opsForValue().set(cacheKey, jsonbackstr, 5, TimeUnit.DAYS);
                    }
                }
                if (jsonObject != null) {
                    return jsonObject.getString("id");
                }
            }
        } catch (Exception e) {
        }
        return userCode;
    }

    /**
     * 通过元数据服务获取实体信息
     * @param referCode
     * @return
     */
    private static String getReferRestUrl(String referCode) {
        String data = null;
        MetaDataUrlconstants urlconstants = ContextUtil.getBean(MetaDataUrlconstants.class);
        String backData = null;
        try {
            backData = ReferHttpClientUtils.getAndHeader(urlconstants.getMetaDataBaseUrl() + referCode, null);
        } catch (Exception e) {
        }
        JSONObject jsonobject = JSON.parseObject(backData);
        if (jsonobject.getString("code").equals("0")) {
            if (jsonobject.getString("data") != null) {
                data = JSON.parseObject(jsonobject.getString("data")).toJSONString();
            }
        } else {
            ExceptionUtil.wrappBusinessException("参照编码" + referCode + "的实体没有发布元数据!");
        }
        return data;
    }

    /**
     * 根据code查询单据类型
     * @param code
     * @return
     */
    public BillTypeVO getBillTypeByCode(String code){
        CommonResponse<BillTypeVO> resp = billTypeApi.getByCode(code);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据code查询单据类型名称
     * @param code
     * @return
     */
    public String getBillTypeName(String code){
        BillTypeVO vo = this.getBillTypeByCode(code);
        return StringUtils.isNotEmpty(vo.getBillName()) ? vo.getBillName() : "";
    }

    /**
     * 是否生成凭证
     * @param param
     * @return
     */
    public static void voucherParam(QueryParam param){
        if(param.getParams().containsKey("voucherState")){
            String voucherState = String.valueOf(param.getParams().get("voucherState").getValue());
            if(voucherState.contains("1")){
                param.getParams().put("source_id", new Parameter(QueryParam.NE, null));
            } else {
                param.getParams().put("source_id", new Parameter(QueryParam.EQ, null));
            }
            param.getParams().remove("voucherState");
        }
    }
}
