package com.ejianc.business.supbid.util;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

public class CommonUtils {

    /** 默认异常信息 */
    private static final String defaultErrMsg = "推送数据失败！";


    /** 招标类型：材料-MATERIAL,分包-SUB，设备采购-PURCHASE,设备租赁-RENT,周转材租赁-RMAT,其他-OTHER */
    public static final String TYPE_MATERIAL = "MATERIAL";
    /** 招标类型：材料-MATERIAL,分包-SUB，设备采购-PURCHASE,设备租赁-RENT,周转材租赁-RMAT,其他-OTHER */
    public static final String TYPE_SUB = "SUB";
    /** 招标类型：材料-MATERIAL,分包-SUB，设备采购-PURCHASE,设备租赁-RENT,周转材租赁-RMAT,其他-OTHER */
    public static final String TYPE_PURCHASE = "PURCHASE";

    private CommonUtils() {
    }

    /**
     * 校验返回的commonResponse是否是正常的，正常返回true，否则抛出异常
     *
     * @param commonResponse 接口调用结果
     * @param <T>            泛型
     * @return 校验结果  true，正常；false，异常
     */
    public static <T> boolean checkCommonResponse(CommonResponse<T> commonResponse) {
        return checkCommonResponse(commonResponse, true, defaultErrMsg, null);
    }

    /**
     * 校验返回的commonResponse是否是正常的，正常无结果，否则抛出异常
     *
     * @param <T>            泛型
     * @param commonResponse 接口调用结果
     * @param logger         日志
     */
    public static <T> void checkCommonResponse(CommonResponse<T> commonResponse, Logger logger) {
        checkCommonResponse(commonResponse, true, defaultErrMsg, logger);
    }

    /**
     * 校验返回的commonResponse是否是正常的
     *
     * @param commonResponse 返回结果
     * @param isThrow        是否抛出异常。true抛出异常；false，不抛出异常，返回false
     * @param errMsg         抛出异常时异常信息
     * @param logger         日志
     * @param <T>            泛型
     * @return 校验结果  true，正常；false，异常
     */
    public static <T> boolean checkCommonResponse(CommonResponse<T> commonResponse, boolean isThrow, String errMsg,
                                                  Logger logger) {
        if (!commonResponse.isSuccess()) {
            if (logger != null) {
                logger.info("调用接口失败，调用结果：{}，失败原因：{}", commonResponse.isSuccess(), commonResponse.getMsg());
            }
            if (isThrow) {
                throw new BusinessException(errMsg);
            }
            return false;
        }
        try {
            T data = commonResponse.getData();
            String dataString;
            if (data instanceof String) {
                dataString = String.valueOf(data);
            }
            else {
                dataString = JSONObject.toJSONString(data);
            }
            JSONObject jsonObject = JSONObject.parseObject(dataString);
            // 如果返回的结果包含code
            if (StringUtils.isNotEmpty(dataString) && jsonObject.containsKey("code")) {
                CommonResponse check = JSONObject.toJavaObject(jsonObject, CommonResponse.class);
                if (isThrow && !check.isSuccess()) {
                    if (logger != null) {
                        logger.error("调用接口失败，返回结果：{}", check.getMsg());
                    }
                    throw new BusinessException(errMsg);
                }
                return check.isSuccess();
            }
        }
        catch (Exception e) {
            if (logger != null) {
                logger.error("类型转换失败！返回结果不是json或CommonResponse");
            }
            // 异常情况按照true处理
            return true;
        }
        // 默认true
        return true;
    }
}
