package com.ejianc.foundation.share.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.ejianc.foundation.billcode.BillCodeException;
import com.ejianc.foundation.share.bean.MaterialCategoryEntity;
import com.ejianc.foundation.share.bean.MaterialEntity;
import com.ejianc.foundation.share.service.IMaterialCategoryService;
import com.ejianc.foundation.share.service.IMaterialService;
import com.ejianc.foundation.share.vo.*;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/materialApi/")
public class MaterialApi {
    // 日志
    private final Logger logger = LoggerFactory.getLogger(getClass());
    @Autowired
    private IMaterialService materialService;
    @Autowired
    private IMaterialCategoryService materialCategoryService;

    @PostMapping("queryExitFlag")
    public CommonResponse<MaterialVO> queryExitFlag(@RequestBody MaterialVO materialVO) {
        logger.info("调用queryExitFlag {}", JSONObject.toJSONString(materialVO));
        MaterialVO vo = materialService.queryExitFlag(materialVO.getName(), materialVO.getSpec(), materialVO.getUnitName(), materialVO.getCategoryId());
        return CommonResponse.success("查询成功！", vo);
    }

    @GetMapping("queryCategoryByCode")
    public CommonResponse<MaterialCategoryVO> queryCategoryByCode(@RequestParam(value = "code") String code) {
        List<MaterialCategoryEntity> list = materialCategoryService.queryByCode(String.valueOf(code));
        MaterialCategoryVO vo = null;
        if (ListUtil.isNotEmpty(list)) {
            vo = BeanMapper.map(list.get(0), MaterialCategoryVO.class);
        }
        if (vo == null) {
            return CommonResponse.error("查询失败,没有查询到该编码所属分类！");
        }
        return CommonResponse.success("查询成功！", vo);
    }

    @GetMapping("queryCategoryListByPid")
    public CommonResponse<List<MaterialCategoryVO>> queryCategoryListByPid(@RequestParam(value = "id") Long id) {
        List<MaterialCategoryVO> list = materialCategoryService.queryListByPid(id);
        return CommonResponse.success("查询成功！", list);
    }

    @GetMapping("queryCategoryListByChildren")
    public CommonResponse<List<MaterialCategoryVO>> queryCategoryListByChildren(@RequestParam(value = "ids") List<Long> ids) {
        List<MaterialCategoryVO> list = materialCategoryService.queryCategoryListByChildren(ids);
        return CommonResponse.success("查询成功！", list);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2020/10/19 根据sourceId查询物资
     * @Description queryMaterialBySourceId
     * @Param [sourceId]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.foundation.share.vo.MaterialVO>
     */
    @GetMapping("queryMaterialBySourceId")
    public CommonResponse<MaterialVO> queryMaterialBySourceId(String sourceId) {
        MaterialEntity entity = materialService.queryBySourceId(sourceId);
        if (entity != null) {
            return CommonResponse.success(BeanMapper.map(entity, MaterialVO.class));
        }
        return CommonResponse.error("查询失败,未查到相关物资。");
    }

    /**
     * @Author liyj
     * @Date 2020/10/29 根据ids查询物资列表（包括物资来源id，物资分类来源id）
     * @Param [ids]
     * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.foundation.share.vo.MaterialVO>
     */
    @GetMapping(value = "queryMaterialByIds")
    CommonResponse<List<MaterialVO>> queryMaterialByIds(@RequestParam(value = "ids") List<Long> ids) {
        List<MaterialEntity> entitys = materialService.queryMaterialByIds(ids);
        if (entitys != null && entitys.size() > 0) {
            return CommonResponse.success(BeanMapper.mapList(entitys, MaterialVO.class));
        }
        return CommonResponse.error("查询失败,未查到相关物资。");
    }

    ;

    /**
     * 根据主键查询物料分类详情
     *
     * @param id
     * @return
     * @Author zhangwx
     */
    @GetMapping("queryCategoryById")
    public CommonResponse<MaterialCategoryVO> queryCategoryById(@RequestParam(value = "id") Long id) {
        return CommonResponse.success("查询成功！", materialCategoryService.queryDetail(id));
    }

    /***
     * @description: 根据物资分类名称查询物资分类
     *
     */
    @PostMapping("/api/materialApi/queryCategorysByNames")
    public CommonResponse<List<MaterialCategoryVO>> queryCategorysByNames(@RequestBody List<String> nameList) {
        List<MaterialCategoryVO> entitys = materialCategoryService.queryCategorysByNames(nameList);
        return CommonResponse.success(entitys);
    }


    /**
     * @description: 根据innercode查询物资
     *
     * @param params innerCodes 父类id集合  categoryName 分类名称模糊查询
     * @return: com.ejianc.framework.core.response.CommonResponse<java.util.List < com.ejianc.foundation.share.vo.MaterialCategoryVO>>
     * @author songlx
     * @date: 2021/9/10
     */
    @PostMapping("queryListByInnerCode")
    public CommonResponse<List<MaterialCategoryVO>> queryListByInnerCode(@RequestBody Map<String, Object> params) {
        return CommonResponse.success("查询成功！", materialCategoryService.queryListByInnerCodes(params));
    }

    /***
     * @description: 根据物资名称查询物资
     *
     * @author zhangwx
     * @date: 2021/9/16
     */
    @GetMapping("queryMaterialListByNames")
    public CommonResponse<List<MaterialVO>> queryMaterialListByNames(@RequestParam(value = "nameList") List<String> nameList) {
        List<MaterialEntity> entitys = materialService.queryMaterialListByNames(nameList);
        if (entitys != null && entitys.size() > 0) {
            return CommonResponse.success(BeanMapper.mapList(entitys, MaterialVO.class));
        }
        return CommonResponse.error("查询失败,未查到相关物资。");
    }

    @PostMapping("queryMaterialsByNames")
    @ResponseBody
    public CommonResponse<List<MaterialVO>> queryMaterialsByNames(@RequestBody List<String> nameList) {
        List<MaterialEntity> entitys = materialService.queryMaterialListByNames(nameList);
        if (entitys != null && entitys.size() > 0) {
            return CommonResponse.success(BeanMapper.mapList(entitys, MaterialVO.class));
        }
        return CommonResponse.error("查询失败,未查到相关物资。");
    }

    /**
     * 根据物资名称、规格型号、计量单位 查询
     *
     * @param materialVO(name,spac,unitName,categoryId)
     * @return
     */
    @RequestMapping(value = "/queryMaterialList", method = RequestMethod.POST)
    public CommonResponse<List<MaterialVO>> queryMaterialList(@RequestBody MaterialVO materialVO) {
        logger.info("调用queryMaterialList {}", JSONObject.toJSONString(materialVO));
        List<MaterialVO> list = materialService.queryMaterialList(materialVO.getName(), materialVO.getSpec(), materialVO.getUnitName());
        return CommonResponse.success("查询成功！", list);
    }


    /**
     * 校验档案是否存在
     *
     * @param materialFuzzyMatchVO 材料设备模糊匹配VO
     *
     * @return {@link CommonResponse}<{@link MaterialFuzzyMatchVO}>
     */
    @PostMapping(value = "/checkArchive")
    public CommonResponse<MaterialFuzzyMatchVO> checkArchive(@RequestBody MaterialFuzzyMatchVO materialFuzzyMatchVO) {
        return CommonResponse.success("校验档案是否存在成功！", materialService.checkArchive(materialFuzzyMatchVO));
    }


    /**
     * 模糊匹配数量
     *
     * @param materialFuzzyMatchVO 材料设备模糊匹配VO
     *
     * @return {@link CommonResponse}<{@link MaterialFuzzyMatchVO}>
     */
    @PostMapping(value = "/fuzzyMatchNumber")
    public CommonResponse<MaterialFuzzyMatchVO> fuzzyMatchNumber(@RequestBody MaterialFuzzyMatchVO materialFuzzyMatchVO) {
        return CommonResponse.success("模糊匹配数量成功！", materialService.fuzzyMatchNumber(materialFuzzyMatchVO));
    }


    /**
     * 模糊匹配分页查询
     *
     * @param materialFuzzyMatchVO 材料设备模糊匹配VO
     *
     * @return {@link CommonResponse}<{@link IPage}<{@link MaterialPlusVO}>>
     */
    @PostMapping(value = "/fuzzyMatchQueryPage")
    public CommonResponse<IPage<MaterialPlusVO>> fuzzyMatchQueryPage(@RequestBody MaterialFuzzyMatchVO materialFuzzyMatchVO) {
        return CommonResponse.success("模糊匹配分页查询成功！", materialService.fuzzyMatchQueryPage(materialFuzzyMatchVO.getCurrent(), materialFuzzyMatchVO.getSize(), materialFuzzyMatchVO.getName(), materialFuzzyMatchVO.getType(), materialFuzzyMatchVO.getCategoryName(), materialFuzzyMatchVO.getSpec(), materialFuzzyMatchVO.getUnitName(), materialFuzzyMatchVO.getFuzzyName(), materialFuzzyMatchVO.getFuzzySpec(), materialFuzzyMatchVO.getFuzzyUnitName()));
    }


    /**
     * 批量插入档案
     *
     * @param materialVOMap 材料设备VO
     *
     * @return {@link CommonResponse}<{@link Map}<{@link String}, {@link MaterialVO}>>
     *
     * @throws BillCodeException 单据编码异常
     */
    @PostMapping(value = "/batchInsertArchive")
    public CommonResponse<Map<String, MaterialVO>> batchInsertArchive(@RequestBody Map<String, MaterialVO> materialVOMap) throws BillCodeException {
        return materialService.batchInsertArchive(materialVOMap, null, null, null);
    }

    @PostMapping(value = "/batchInsertArchiveBySourceBill")
    public CommonResponse<Map<String, MaterialVO>> batchInsertArchiveBySourceBill(@RequestBody MaterialInsertArchiveVO materialInsertArchiveVO) throws BillCodeException {
        return materialService.batchInsertArchive(materialInsertArchiveVO.getMaterialVOMap(), materialInsertArchiveVO.getSourceBillId(), materialInsertArchiveVO.getSourceBillCode(), materialInsertArchiveVO.getSourceBillType());
    }


    @PostMapping("queryMaterialsByCodes")
    @ResponseBody
    public CommonResponse<List<MaterialVO>> queryMaterialsByCodes(@RequestBody List<String> codeList) {
        List<MaterialEntity> entitys = materialService.queryMaterialListByCodes(codeList);
        if (entitys != null && entitys.size() > 0) {
            return CommonResponse.success(BeanMapper.mapList(entitys, MaterialVO.class));
        }
        return CommonResponse.error("查询失败,未查到相关物资。");
    }


    @PostMapping("queryByCode")
    public CommonResponse<MaterialVO> queryByCode(@RequestParam(value = "code") String code) {
        MaterialVO vo = materialService.queryByCode(code);
        return CommonResponse.success("查询成功！", vo);
    }
}
