package com.ejianc.foundation.support.controller.api;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.support.idworker.util.IdWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.foundation.support.bean.BillTypeEntity;
import com.ejianc.foundation.support.bean.ModuleEntity;
import com.ejianc.foundation.support.service.IBillTypeService;
import com.ejianc.foundation.support.service.IModuleService;
import com.ejianc.foundation.support.vo.ModuleVO;
import com.ejianc.framework.cache.redis.CacheManager;
import com.ejianc.framework.cache.serializer.Serializer;
import com.ejianc.framework.cache.serializer.impl.DefaultJDKSerializer;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.template.BaseVO;

@RestController
@RequestMapping("/api/module/")
public class ModuleApi {
	private Logger logger = LoggerFactory.getLogger(this.getClass());

	// 所有模块缓存
	private final static String MODEL_CACHE_KEY = "model_cache_key:";

	@Autowired
	private CacheManager cacheManager;
	@Autowired
	private IModuleService moduleService;
	@Autowired
	private IBillTypeService billTypeService;

	public static final String DEFAULT_CHARSET = "UTF-8";

	@Value("${oms.tenantid}")
	private Long omsTenantId;

	private Serializer serializer = new DefaultJDKSerializer();

	@RequestMapping(value = "getListByPid", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<List<ModuleVO>> queryModuleListByPid(@RequestParam(value="moduleId", required=true) Long moduleId) {
		try {
			List<ModuleVO> moduleList = moduleService.queryAllChildrenByPid(moduleId);
			return CommonResponse.success(moduleList);
		} catch (Exception e) {
			logger.error("系统异常："+e.getMessage());
			e.printStackTrace();
		}
		return CommonResponse.error("查询失败");
	}

	/**
	 * @Author mrsir_wxp
	 * @Date 2020/9/15 根据模块id查询模块信息
	 * @Description getModuleById
	 * @Param [moduleId]
	 * @Return com.ejianc.framework.core.response.CommonResponse<com.ejianc.foundation.support.vo.ModuleVO>
	 */
	@RequestMapping(value = "getModuleById", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<ModuleVO> getModuleById(@RequestParam(value="moduleId", required=true) Long moduleId) {
		try {
			ModuleEntity moduleEntity = moduleService.selectById(moduleId);
			if(moduleEntity != null) {
				ModuleVO moduleVO = BeanMapper.map(moduleEntity, ModuleVO.class);
				return CommonResponse.success(moduleVO);
			}
		} catch (Exception e) {
			logger.error("系统异常："+e.getMessage());
			return CommonResponse.error("查询失败");
		}
		return CommonResponse.success();
	}

	@PostMapping(value = "refreshModuleCache")
	public CommonResponse<String> refreshModuleCache() {
		try {
			QueryParam queryParam = new QueryParam();
			queryParam.getParams().put("tenant_id", new Parameter(QueryParam.EQ, omsTenantId));
			queryParam.getParams().put("dr", new Parameter(QueryParam.EQ, BaseVO.DR_UNDELETE));
			queryParam.getOrderMap().put("sequence", "asc");
			List<ModuleEntity> moduleList = moduleService.queryList(queryParam, false);

			if(!moduleList.isEmpty()) {
				cacheManager.piplineExecute(pipeline -> {
					String key = MODEL_CACHE_KEY + omsTenantId;
					pipeline.multi();
					for(ModuleEntity m : moduleList) {
						byte[] keyBytes = key.getBytes(Charset.forName(DEFAULT_CHARSET));
						byte[] fieldBytes = m.getId().toString().getBytes(Charset.forName(DEFAULT_CHARSET));
						byte[] contentBytes = serializer.marshalToByte(JSON.toJSONString(m));
						pipeline.hset(keyBytes, fieldBytes, contentBytes);
					}
					pipeline.exec();
				});
			}
			return CommonResponse.success("模块缓存刷新成功！");
		} catch (Exception e) {
			logger.error("系统异常：", e);
		}
		return CommonResponse.error("模块缓存刷新失败");
	}
	
	@RequestMapping(value = "queryModuleTree", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<List<ModuleVO>> queryModuleTree() {
		QueryWrapper<ModuleEntity> queryWrapper = new QueryWrapper<>();
		queryWrapper.eq("tenant_id", 999999l);
		List<ModuleEntity> moduleList = moduleService.list(queryWrapper);
		
		List<BillTypeEntity> billTypeEntities = billTypeService.queryAllBillTypeList();
		if(billTypeEntities != null && billTypeEntities.size() > 0) {
			for(BillTypeEntity billTypeEntity:billTypeEntities) {
				ModuleEntity moduleEntity = new ModuleEntity();
				moduleEntity.setId(billTypeEntity.getId());
				moduleEntity.setModuleCode(billTypeEntity.getBillCode());
				moduleEntity.setModuleName(billTypeEntity.getBillName());
				moduleEntity.setParentId(billTypeEntity.getModuleId());
				moduleList.add(moduleEntity);
			}
		}
		
		List<ModuleVO> moduleVoList = BeanMapper.mapList(moduleList, ModuleVO.class);
		return CommonResponse.success(moduleVoList);
	}
	
	@RequestMapping(value = "queryAllTopModuleList", method=RequestMethod.GET)
	@ResponseBody
	public CommonResponse<List<ModuleVO>> queryAllTopModuleList() {
		List<ModuleVO> moduleList = moduleService.queryAllTopModuleList();
		return CommonResponse.success(moduleList);
	}
	/**
	 * 根据名称查询模块
	 * */
	@GetMapping("queryModuleListByName")
	@ResponseBody
	public CommonResponse<List<ModuleVO>> queryModuleListByName(@RequestParam(value="name", required=true) String name) {
		QueryParam param = new QueryParam();
		param.setSearchText(name);
		param.getFuzzyFields().add("moduleName");
		param.getParams().put("tenantId",new Parameter(QueryParam.EQ,999999L));
		List<ModuleEntity> moduleList = moduleService.queryList(param,false);
		if(ListUtil.isEmpty(moduleList)){
			return CommonResponse.success(new ArrayList<>());
		}
		return CommonResponse.success(BeanMapper.mapList(moduleList,ModuleVO.class));
	}

	@GetMapping(value = "checkAndCreateCFSModule")
	@ResponseBody
	CommonResponse<Long> checkAndCreateCFSModule(){
		QueryParam param = new QueryParam();
		param.getParams().put("moduleCode",new Parameter(QueryParam.EQ,"cfsModuleCode"));
		List<ModuleEntity> moduleList = moduleService.queryList(param,false);
		if(ListUtil.isNotEmpty(moduleList)){
			return CommonResponse.success(moduleList.get(0).getId());
		}
		param = new QueryParam();
		param.getParams().put("parentId",new Parameter(QueryParam.EQ,null));
		moduleList = moduleService.queryList(param,false);
		ModuleEntity root =moduleList.get(0);
		Long moduleId = IdWorker.getId();
		ModuleEntity moduleEntity = new ModuleEntity();
		moduleEntity.setId(moduleId);
		moduleEntity.setParentId(root.getId());
		moduleEntity.setModuleCode("cfsModuleCode");
		moduleEntity.setInnerCode(root.getInnerCode()+"|"+moduleId);
		moduleEntity.setModuleName("自定义应用");
		moduleEntity.setSequence(100);
		moduleEntity.setDescription("自定义应用");
		moduleEntity.setTenantId(omsTenantId);
		moduleService.saveOrUpdate(moduleEntity,false);
		return CommonResponse.success(moduleId);
	}
}
