package com.ejianc.foundation.billcode.engine;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONArray;
import com.ejianc.foundation.billcode.BillCodeException;
import com.ejianc.foundation.billcode.elemproc.result.BillCodeInfo;
import com.ejianc.foundation.billcode.engine.persistence.IBillCodeEngineService;
import com.ejianc.foundation.billcode.engine.persistence.vo.BillCodeReternVO;
import com.ejianc.foundation.billcode.engine.persistence.vo.BillCodeSNVO;
import com.ejianc.foundation.billcode.engine.persistence.vo.PreCodeVO;
import com.ejianc.framework.cache.redis.CacheManager;

public class BillCodeEngine {

    private static final Logger logger = LoggerFactory.getLogger(BillCodeEngine.class);

    private final String BILL_CODE_SN_REFER_CACHE_KEY_PREFIX = "ejcbillcode";
    private final String RETURN_BILL_CODE_CACHE_KEY_PREFIX = "returnejcbillcode";
    private final String CACHE_KEY_SEPERATOR = ":";

    private BillCodeInfo info;
    private IBillCodeEngineService billCodeEngineService;
    private String lastsn;

//    private JedisCacheTool jedisCacheTool;

    private CacheManager cacheManager;

    private BillcodeSNReferProducer billcodeSNReferProducer;

    public BillCodeEngine(BillCodeInfo billCodeInfo, IBillCodeEngineService persisServ, CacheManager cacheManager,
                          BillcodeSNReferProducer billcodeSNReferProducer) {
        this.info = billCodeInfo;
        this.billCodeEngineService = persisServ;
        this.billcodeSNReferProducer = billcodeSNReferProducer;
        this.cacheManager = cacheManager;
    }

    public String getLastsn() {
        return lastsn;
    }

    public void setLastsn(String lastsn) {
        this.lastsn = lastsn;
    }

    public CacheManager getCacheManager() {
        return cacheManager;
    }

    public void setCacheManager(CacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    public BillcodeSNReferProducer getBillcodeSNReferProducer() {
        return billcodeSNReferProducer;
    }

    public void setBillcodeSNReferProducer(BillcodeSNReferProducer billcodeSNReferProducer) {
        this.billcodeSNReferProducer = billcodeSNReferProducer;
    }

    public void insertInitSN(Long tenantId) throws BillCodeException {
        BillCodeSNVO initvo = new BillCodeSNVO();
        initvo.setLastsn("0");
        initvo.setMarkStr(this.info.getBillCodeSNRefer());
        initvo.setMarkStrDesc(this.info.getBillCodeSNReferDesc());
        initvo.setRuleId(this.info.getPk_bilcodebase());
        initvo.setTenantId(tenantId);
        try {
            billCodeEngineService.insertSNVO(initvo);
        } catch (Exception arg2) {
            logger.error("新增初始化SN异常，", arg2);
            throw new BillCodeException(arg2.getMessage());
        }
    }

    public IBillCodeEngineService getBillCodeEngineService() {
        return billCodeEngineService;
    }

    public void setBillCodeEngineService(IBillCodeEngineService billCodeEngineService) {
        this.billCodeEngineService = billCodeEngineService;
    }

    public BillCodeInfo getInfo() {
        return info;
    }

    public void setInfo(BillCodeInfo info) {
        this.info = info;
    }

    public String[] getBillCode(int num, Long tenantId) throws BillCodeException {
        BillCodeSNVO billcodesnVO = this.setLastSN(tenantId);
        List<String> billcodes = new ArrayList<>();
        if (this.info.isautofill()) {
            this.getReturnedBillCode(num, billcodes, tenantId);
        }

        num -= billcodes.size();
        if (num > 0) {
            this.getSequenceBillCode(num, billcodes);
        }

        billcodesnVO.setLastsn(this.lastsn);
        cacheManager.set(getBillcodeSnReferCacheKey(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId), billcodesnVO);
//        jedisCacheTool.put();
        // 写入异步队列
        billcodeSNReferProducer.sendQueue(billcodesnVO);
        return (String[]) billcodes.toArray(new String[0]);
    }

    private void getSequenceBillCode(int num, List<String> billcodes) throws BillCodeException {
        if (this.info.getSerialNumInfo().getSnGenerator().isRunOutSNumber(this.lastsn, num, this.info.getSerialNumInfo().getSnLength()).booleanValue()) {
            throw new BillCodeException("没有多余的单据号可供分配");
        } else {
            String codenosn = this.info.getBillCodeTemplte();
            for (int i = 0; i < num; ++i) {
                this.getNextSNumber();
                String billcode = codenosn.replaceFirst("##SN##", this.lastsn);
                billcodes.add(billcode);
            }
        }
    }

    private void getNextSNumber() {
        this.lastsn = this.info.getSerialNumInfo().getSnGenerator().getNextSNumber(this.lastsn,
                Boolean.valueOf(this.info.getSerialNumInfo().isAppendZero()),
                this.info.getSerialNumInfo().getSnLength());
    }

    private void getReturnedBillCode(int num, List<String> billcodes, Long tenantId)  throws BillCodeException {
        List<BillCodeReternVO> rtnvos = null;
        String key = getReturnBillcodeSnReferCacheKey(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId);
        String rtnvosJsonStr = cacheManager.get(key);
        if(StringUtils.isNotBlank(rtnvosJsonStr)) {
            rtnvos = JSONArray.parseArray(rtnvosJsonStr, BillCodeReternVO.class);
        }
        if (rtnvos == null) {
            rtnvos = this.billCodeEngineService.queryRtnVO(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId);
            cacheManager.set(getReturnBillcodeSnReferCacheKey(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId), JSONArray.toJSONString(rtnvos));
        }
        ArrayList<BillCodeReternVO> alReturn = new ArrayList<BillCodeReternVO>();
        String codenosn = this.info.getBillCodeTemplte();
        int sucNum = 0;
        if(CollectionUtils.isNotEmpty(alReturn)) {
            for(BillCodeReternVO bcr : alReturn) {
                String billcode = codenosn.replaceFirst("##SN##", bcr.getRtnsn());
                ++sucNum;
                alReturn.add(bcr);
                billcodes.add(billcode);
                if(sucNum >= num) {
                    break;
                }
            }
        }

        try {
            if (alReturn.size() > 0) {
                List<String> rtnsns=  alReturn.stream().map(BillCodeReternVO::getRtnsn).collect(Collectors.toList());
                rtnvos = rtnvos.stream().filter(r -> !rtnsns.contains(r.getRtnsn())).collect(Collectors.toList());
                cacheManager.set(getReturnBillcodeSnReferCacheKey(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId), JSONArray.toJSONString(rtnvos));
                List<BillCodeReternVO> delList = new ArrayList<>();
                delList.add(alReturn.get(0));
                this.billCodeEngineService.deletRtnCodeBatch(delList);
            }
        } catch (Exception arg9) {
            logger.error("异常: ", arg9);
            throw new BillCodeException(arg9.getMessage());
        }
    }

    private String getReturnBillcodeSnReferCacheKey(String pk_bilcodebase, String billCodeSNRefer, Long tenantId) {
        return RETURN_BILL_CODE_CACHE_KEY_PREFIX + CACHE_KEY_SEPERATOR + pk_bilcodebase + billCodeSNRefer + tenantId;
    }

    private BillCodeSNVO setLastSN(Long tenantId) throws BillCodeException {
        BillCodeSNVO billcodesnVO = (BillCodeSNVO) cacheManager.get(getBillcodeSnReferCacheKey(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId));
        if (billcodesnVO == null) {
            billcodesnVO = this.billCodeEngineService.querySNVO(this.info.getPk_bilcodebase(), this.info.getBillCodeSNRefer(), tenantId);
        }

        if (billcodesnVO == null) {
            billcodesnVO = new BillCodeSNVO();
            billcodesnVO.setRuleId(this.info.getPk_bilcodebase());
            billcodesnVO.setMarkStr(this.info.getBillCodeSNRefer());
            billcodesnVO.setLastsn("0");
            billcodesnVO.setMarkStrDesc(this.info.getBillCodeSNReferDesc());
            billcodesnVO.setTenantId(tenantId);
            // 通过异步队列处理
            // this.service.insertSNVO(billcodesnVO);
            // TODO IUAP BUG
            this.lastsn = "0";
        } else {
            Boolean isEffectiveSN = this.info.getSerialNumInfo().getSnGenerator().isEffectiveSNumber(billcodesnVO.getLastsn());
            if (!isEffectiveSN.booleanValue()) {
                throw new BillCodeException("编码生成失败，可能是因为修改了编码规则的流水号生成器，未重置流水号，请检查。");
            }
            this.lastsn = this.info.getSerialNumInfo().getSnGenerator().getFormatedSNumber(billcodesnVO.getLastsn(),Boolean.valueOf(this.info.getSerialNumInfo().isAppendZero()),this.info.getSerialNumInfo().getSnLength());
        }
        return billcodesnVO;
    }

    private String getBillcodeSnReferCacheKey(String pk_bilcodebase, String billCodeSNRefer, Long tenantId) {
        return BILL_CODE_SN_REFER_CACHE_KEY_PREFIX + CACHE_KEY_SEPERATOR + pk_bilcodebase + billCodeSNRefer + tenantId;
    }

    public void insertPrecode(String billCode, Long tenantId) {
        if (this.info.isautofill()) {
            String sn = this.getSNByBillCode(billCode);
            if (sn == null || sn.isEmpty()) {
                return;
            }

            PreCodeVO precode = new PreCodeVO();
            precode.setPkRulebase(this.info.getPk_bilcodebase());
            precode.setMarkStr(this.info.getBillCodeSNRefer());
            precode.setMarkStrDesc(this.info.getBillCodeSNReferDesc());
            precode.setLastsn(sn);
            precode.setTenantId(tenantId);
            precode.setBillcode(billCode);
            this.billCodeEngineService.insertPreCode(precode);
        }
    }

    private String getSNByBillCode(String billCode) {
        if (billCode != null && billCode.length() != 0) {
            if (billCode.trim().length() != this.info.getBillcodeLength()) {
                return null;
            } else {
                int i = this.info.getBillCodeTemplte().indexOf("##SN##");
                if (i == -1) {
                    return null;
                } else {
                    int j = i + this.info.getSerialNumInfo().getSnLength();
                    String sn = billCode.substring(i, j);
                    return !this.info.getSerialNumInfo().getSnGenerator().isEffectiveSNumber(sn).booleanValue() ? null : sn;
                }
            }
        } else {
            return null;
        }
    }
}
