package com.ejianc.foundation.supplier.controller;

import com.ejianc.foundation.supplier.bean.*;
import com.ejianc.foundation.supplier.service.*;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.kit.collection.ListUtil;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.support.idworker.util.IdWorker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 *
 * 该元素其他综合接口
 *
 * */
@Controller
@RequestMapping("supplierCollect")
public class SupplierCollect {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ISupplerIncidentService incidentService;
    @Autowired
    private ISupplierYearService yearService;
    @Autowired
    private IBillCodeApi billCodeApi;
    @Autowired
    private ISupplierGradeService gradeService;
    @Autowired
    private ISupplierService supplierService;
    @Autowired
    private ISupplierPunishService punishService;

    /**
     *
     * 生成供应商年度评价单据
     *
     */
    @RequestMapping("createYearEvaluation")
    @ResponseBody
    public CommonResponse<String> createYearEvaluation(){
        List<SupplerIncidentEntity> list = incidentService.getYearIncident(LocalDate.now().getYear());
        if(ListUtil.isEmpty(list)){
            logger.info("没有可执行数据，未生成本年度评价单据！");
            return CommonResponse.success("执行成功！","没有可执行数据，未生成本年度评价单据！");
        }
        List<SupplierYearEntity> yearEntities = new ArrayList<>();
        for (SupplerIncidentEntity supplerIncidentEntity : list) {
            SupplierYearEntity entity = new SupplierYearEntity();
            CommonResponse<String> billCode = billCodeApi.getCodeBatchByRuleCode(SupplierYearController.BILL_CODE, InvocationInfoProxy.getTenantid());
            if(billCode.isSuccess()) {
                entity.setCode(billCode.getData());
            }else{
                entity.setCode("NDPJ-"+ IdWorker.getId());
            }
            entity.setSupplyId(supplerIncidentEntity.getSupplyId());
            entity.setSupplyName(supplerIncidentEntity.getSupplyName());
            entity.setOrgId(supplerIncidentEntity.getOrgId());
            entity.setOrgName(supplerIncidentEntity.getOrgName());
            entity.setOrgCode(supplerIncidentEntity.getOrgCode());
            entity.setEndDate(new Date());
            entity.setYearDeduct(0);
            entity.setBillState(0);
            entity.setStatus(0);
            entity.setProcessBranch(supplerIncidentEntity.getBuckleBranch());
            entity.setThiYearNum(supplerIncidentEntity.getBillState());//使用billState 作为与该供应商的合作次数
            entity.setTotalNum(supplerIncidentEntity.getBillState());//使用billState 作为与该供应商的合作次数
            entity.setTotalAmount(supplerIncidentEntity.getContractAmount());
            yearEntities.add(entity);
        }
        yearService.saveOrUpdateBatch(yearEntities);
        return CommonResponse.success("执行成功！","生成本年度评价单据成功！");
    }

    /**
     * 根据供应商等级设置 以及 供应商年度评价单据的分数 和 占比
     * 更改供应商等级
     * */
    @RequestMapping("createYearRating")
    @ResponseBody
    public CommonResponse<String> createYearRating(){
        List<SupplierYearEntity> list = yearService.queryTotalAvg();
        if(ListUtil.isEmpty(list)){
            logger.info("没有可执行数据，未更改任何供应商等级！");
            return CommonResponse.success("执行成功！","没有可执行数据，未更改任何供应商等级！");
        }
        QueryParam queryParam = new QueryParam();
        queryParam.getOrderMap().put("id",QueryParam.ASC);
        List<SupplierGradeEntity> gradeEntities = gradeService.queryList(queryParam,false);
        if(ListUtil.isEmpty(gradeEntities)){
            logger.info("执行失败！没有找到任何供应商等级设置！");
            return CommonResponse.success("执行失败！","没有找到任何供应商等级设置！");
        }
        Map<String, BigDecimal> gradeMap = gradeEntities.stream().collect(Collectors.toMap(SupplierGradeEntity::getName, SupplierGradeEntity::getReachScore));
        Map<String, BigDecimal> gradeTakeMap = gradeEntities.stream().collect(Collectors.toMap(SupplierGradeEntity::getName, SupplierGradeEntity::getLevelRate));
        if(!gradeMap.containsKey("A")){
            logger.info("执行失败！没有找到A级供应商等级设置！");
            return CommonResponse.success("执行失败！","没有找到A级供应商等级设置！");
        }
        if(!gradeMap.containsKey("B")){
            logger.info("执行失败！没有找到B级供应商等级设置！");
            return CommonResponse.success("执行失败！","没有找到B级供应商等级设置！");
        }
        if(!gradeMap.containsKey("C")){
            logger.info("执行失败！没有找到C级供应商等级设置！");
            return CommonResponse.success("执行失败！","没有找到C级供应商等级设置！");
        }
        if(!gradeMap.containsKey("D")){
            logger.info("执行失败！没有找到D级供应商等级设置！");
            return CommonResponse.success("执行失败！","没有找到D级供应商等级设置！");
        }
        for (SupplierYearEntity yearEntity : list) {
            BigDecimal yearDeduct = BigDecimal.valueOf(100+yearEntity.getYearDeduct());
            SupplierEntity supplier = supplierService.selectById(yearEntity.getSupplyId());
            supplier.setYearScore(yearDeduct);
            if(yearDeduct.compareTo(gradeMap.get("A"))>=0){
                supplier.setGradeName("A");
            } else if (yearDeduct.compareTo(gradeMap.get("B"))>=0) {
                supplier.setGradeName("B");
            }else if (yearDeduct.compareTo(gradeMap.get("C"))>=0) {
                supplier.setGradeName("C");
            }else {
                supplier.setGradeName("D");
            }
            supplierService.saveOrUpdate(supplier,false);
        }
        queryParam = new QueryParam();
        queryParam.getParams().put("billState", Parameter.getInInstance(new String[]{"1","3"}));
        List<SupplierEntity> supplierTotal = supplierService.queryList(queryParam,false);
        int total = supplierTotal.size();
        queryParam.getParams().put("gradeName", Parameter.getEqInstance("A"));
        queryParam.getOrderMap().put("yearScore",QueryParam.ASC);
        List<SupplierEntity> supplierA = supplierService.queryList(queryParam,false);
        int a = supplierA.size();
        int mostNum = BigDecimal.valueOf(total).multiply(gradeTakeMap.get("A")).divide(BigDecimal.valueOf(100),0, RoundingMode.DOWN).intValue();
        if(a>mostNum){
            /** 超过比例 */
            int over = a - mostNum;
            for (int i = 0; i < over; i++) {
                SupplierEntity supplier = supplierA.get(i);
                logger.info("供应商总数：{}，根据A级供应商规则，其达标分数为：{}，其数量占比为：{} ，最多{}个，供应商【{}】等级被调整到为B",total, gradeMap.get("A"), gradeTakeMap.get("A"), mostNum, supplier.getName());
                supplier.setGradeName("B");
                supplierService.saveOrUpdate(supplier, false);
            }
        }

        queryParam.getParams().put("gradeName", Parameter.getEqInstance("B"));
        List<SupplierEntity> supplierB = supplierService.queryList(queryParam,false);
        int b = supplierB.size();
        mostNum= BigDecimal.valueOf(total).multiply(gradeTakeMap.get("B")).divide(BigDecimal.valueOf(100),0, RoundingMode.DOWN).intValue();
        if(b>mostNum){
            /** 超过比例 */
            int over = b - mostNum;
            for (int i = 0; i < over; i++) {
                SupplierEntity supplier = supplierB.get(i);
                logger.info("供应商总数：{}，根据B级供应商规则，其达标分数为：{}，其数量占比为：{} ，最多{}个，供应商【{}】等级被调整到为C",total, gradeMap.get("B"), gradeTakeMap.get("B"), mostNum, supplier.getName());
                supplier.setGradeName("C");
                supplierService.saveOrUpdate(supplier, false);
            }
        }
        queryParam.getParams().put("gradeName", Parameter.getEqInstance("C"));
        List<SupplierEntity> supplierC = supplierService.queryList(queryParam,false);
        int c = supplierC.size();
        mostNum= BigDecimal.valueOf(total).multiply(gradeTakeMap.get("C")).divide(BigDecimal.valueOf(100),0, RoundingMode.DOWN).intValue();
        if(c>mostNum){
            /** 超过比例 */
            int over = c - mostNum;
            for (int i = 0; i < over; i++) {
                SupplierEntity supplier = supplierC.get(i);
                logger.info("供应商总数：{}，根据C级供应商规则，其达标分数为：{}，其数量占比为：{} ，最多{}个，供应商【{}】等级被调整到为D",total, gradeMap.get("C"), gradeTakeMap.get("C"), mostNum, supplier.getName());
                supplier.setGradeName("D");
                supplierService.saveOrUpdate(supplier, false);
            }
        }
        return CommonResponse.success("执行成功！","本年度供应商评级数据已生成！");
    }


    /**
     * 每天 0点10分执行 移除黑名单
     * */
    @Scheduled(cron = "0 10 0 * * ?")
    private void rmBlack(){
        logger.info("开始定时执行移除黑名单操作>>>>>>>>>>>>>>>>>>>>>>>>>>");
        /** 先处理供应商档案 */
        QueryParam param = new QueryParam();
        param.getParams().put("groupBlack", new Parameter(QueryParam.EQ, true));
        param.getParams().put("groupLastTime", new Parameter(QueryParam.LT, DateFormatUtil.formatDate(DateFormatUtil.PATTERN_ISO_ON_DATE,new Date())));
        param.getParams().put("billState", new Parameter(QueryParam.IN, "1,3"));
        List<SupplierEntity> list = supplierService.queryList(param,false);
        logger.info("本次移除集团供应商档案黑名单：{}个", list.size());
        if(ListUtil.isNotEmpty(list)){
            for (SupplierEntity supplier : list) {
                supplier.setGroupBlack(false);
                supplier.setGroupLastTime(null);
                supplierService.saveOrUpdate(supplier, false);
            }
        }
        param = new QueryParam();
        param.getParams().put("pushState", new Parameter(QueryParam.EQ,"2"));
        List<SupplierPunishEntity> punishList = punishService.queryList(param, false);
        logger.info("本次移除 组织级 供应商档案黑名单：{}个", punishList.size());
        if(ListUtil.isNotEmpty(punishList)){
            for (SupplierPunishEntity punish : punishList) {
                punish.setPushState("3");
                punishService.saveOrUpdate(punish, false);
                SupplierEntity supplier = supplierService.selectById(punish.getSupplyId());
                supplier.setBlackOrgIds(supplier.getBlackOrgIds().replace(","+punish.getApplyOrgId().toString(),""));
                supplier.setBlackOrgNames(supplier.getBlackOrgNames().replace(","+punish.getApplyOrg(),""));
                supplierService.saveOrUpdate(supplier, false);
            }
        }
        logger.info("<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<移除黑名单操作更新完成");
    }

}
