package com.ejianc.business.supsignature.signature.service.impl;

import cn.hutool.core.util.IdUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.ejianc.business.supsignature.signature.service.ISaasSignService;
import com.ejianc.business.supsignature.signature.vo.TicketDetailVO;
import com.ejianc.foundation.orgcenter.api.IUserApi;
import com.ejianc.foundation.tenant.api.IEnterpriseApi;
import com.ejianc.foundation.tenant.vo.EnterpriseVO;
import com.ejianc.foundation.usercenter.vo.UserVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.qiyuesuo.sdk.v2.SaaSSdkClient;
import com.qiyuesuo.sdk.v2.bean.Company;
import com.qiyuesuo.sdk.v2.bean.User;
import com.qiyuesuo.sdk.v2.http.StreamFile;
import com.qiyuesuo.sdk.v2.json.JSONUtils;
import com.qiyuesuo.sdk.v2.request.SaaSCompanyDetailRequest;
import com.qiyuesuo.sdk.v2.request.SaasCompanyAuthPageUrlRequest;
import com.qiyuesuo.sdk.v2.request.SaasPrivilegeUrlRequest;
import com.qiyuesuo.sdk.v2.response.SaaSCompanyAuthPageResult;
import com.qiyuesuo.sdk.v2.response.SaaSPrivilegeUrlResult;
import com.qiyuesuo.sdk.v2.response.SdkResponse;
import com.qiyuesuo.sdk.v2.utils.CryptUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import org.springside.modules.nosql.redis.JedisTemplate;

import java.io.IOException;
import java.util.List;

@Service("saasSignService")
public class ISaasSignServiceImpl implements ISaasSignService {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private JedisTemplate jedisTemplate;
    @Autowired
    private IUserApi userApi;
    @Autowired
    private IEnterpriseApi enterpriseApi;

    @Value("${qiyuesuoSaas.ssoUrl}")
    private String ssoUrl;              // 单点请求域名  https://cloudapi.qiyuesuo.cn（测试环境），https://cloudapi.qiyuesuo.com（正式环境）
    @Value("${qiyuesuoSaas.serverUrl}")
    private String serverUrl;    // 调用地址     测试环境域名为https://openapi.qiyuesuo.cn，生产环境域名为https://openapi.qiyuesuo.com
    @Value("${qiyuesuoSaas.agentToken}")
    private String agentToken;   // 接入令牌，用于调用企业认证、企业授权等接口
    @Value("${qiyuesuoSaas.agentSecret}")
    private String agentSecret;  // 接入秘钥，用于调用企业认证、企业授权等接口
    @Value("${qiyuesuoSaas.secret}")
    private String secret;       // 用于接收回调消息（企业认证、企业授权、合同状态回调）的签名校验

    private SaaSSdkClient getSaasSdkClient() {
        return new SaaSSdkClient(agentToken, agentSecret, serverUrl);
    }


    /**
     * 数据解密
     *
     * @param encrypt 秘文
     * @return 明文
     */
    @Override
    public String decrypt(String encrypt, String sourceName) {
        String result = null;
        try {
            result = CryptUtils.aesDerypt(encrypt, secret);
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("{}--回调数据解密失败：{}", sourceName, e.getMessage());
        }
        return result;
    }

    /**
     * 查询公司信息
     *
     * @param companyName 是 公司名称
     * @return 公司信息
     */
    @Override
    public CommonResponse<SdkResponse<Company>> queryCompanyDetail(String companyName) {
        logger.info("契约锁------查询公司信息，入参：companyName--{}", companyName);
        // 初始化sdkClient
        SaaSSdkClient sdkClient = getSaasSdkClient();

        SaaSCompanyDetailRequest companyDetailRequest = new SaaSCompanyDetailRequest();
        if (StringUtils.isBlank(companyName)) {
            throw new BusinessException("查询公司信息失败，失败原因：请求参数（公司名称）为空！");
        }
        companyDetailRequest.setCompanyName(companyName);

        String response = sdkClient.service(companyDetailRequest);
        SdkResponse<Company> result = JSONUtils.toQysResponse(response, Company.class);
        if (result.getCode() != 0) {
            logger.info("查询公司（{}）信息失败，失败原因：{}", companyName, result.getMessage());
        }
        logger.info("查询公司（{}）信息成功，返回参数：{}", companyName, JSONObject.toJSONString(result, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        return CommonResponse.success(result);
    }

    /**
     * 企业认证--获取企业认证链接
     *
     * @param companyName   是 待认证公司名称
     * @param applicantInfo 是 认证提交人信息（申请者姓名name， 联系方式contact，联系方式类型contactType：MOBILE、EMAIL），企业认证通过后，认证提交 人会自动成为该企业的系统管理员(例：{"name":"aaa","contact": "15100000000","contactType": "MOBILE"})
     * @param registerNo    否 待认证公司注册号
     * @param legalPerson   否 待认证公司法人姓名
     * @param license       否 营业执照
     * @param callbackUrl   否 认证回调地址
     * @return 企业认证链接
     */
    @Override
    public CommonResponse<SdkResponse<SaaSCompanyAuthPageResult>> queryCompanyAuthPage(String companyName, String applicantInfo, String registerNo, String legalPerson, MultipartFile license, String callbackUrl) {
        logger.info("契约锁------企业认证--获取企业认证链接，入参：companyName--{}，applicantInfo--{}，callbackUrl--{}", companyName, applicantInfo, callbackUrl);
        // 初始化sdkClient
        SaaSSdkClient sdkClient = getSaasSdkClient();

        SaasCompanyAuthPageUrlRequest urlRequest = new SaasCompanyAuthPageUrlRequest();
        if (StringUtils.isBlank(companyName) || StringUtils.isBlank(applicantInfo)) {
            throw new BusinessException("企业认证--获取企业认证链接失败，失败原因：请求参数（待认证公司名称或认证提交人信息）为空！");
        }
        urlRequest.setCompanyName(companyName);
        urlRequest.setApplicantInfo(applicantInfo);
        if (StringUtils.isNotBlank(registerNo)) {
            urlRequest.setRegisterNo(registerNo);
        }
        if (StringUtils.isNotBlank(legalPerson)) {
            urlRequest.setLegalPerson(legalPerson);
        }
        if (license != null && !license.isEmpty()) {
            try {
                urlRequest.setLicense(new StreamFile(license.getInputStream()));
            } catch (IOException e) {
                e.printStackTrace();
                throw new BusinessException("企业认证--获取企业认证链接失败，失败原因：请求参数（营业执照）IO读取异常！");
            }
        }
        if (StringUtils.isNotBlank(callbackUrl)) {
            urlRequest.setCallbackUrl(callbackUrl);
        }

        String response = sdkClient.service(urlRequest);
        SdkResponse<SaaSCompanyAuthPageResult> responseObject = JSONUtils.toQysResponse(response, SaaSCompanyAuthPageResult.class);
        if (responseObject.getCode() != 0) {
            logger.info("企业认证--获取企业认证链接失败，失败原因：{}", responseObject.getMessage());
        }
        logger.info("企业认证--获取企业认证链接成功，返回参数：{}", JSONObject.toJSONString(responseObject, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        return CommonResponse.success(responseObject);
    }

    /**
     * 企业授权--获取授权链接
     *
     * @param companyId        是 企业Id
     * @param appId            否 平台Id，appId于【配置企业单点登录信息接口】回调时返回
     * @param user             是 操作人信息
     * @param createToken      否 是否生成token
     * @param successUrl       否 授权后跳转页面
     * @param callbackUrl      否 token生成后回调地址
     * @param privilegeModules 否 指定授权模块，指定的模块在授权页面默认勾选；可行值如下：SEAL（印章管理）、TEMPLATE（模板管理）、CONTRACT（合同管理）、COMPANY_EMPLOYEE（企业与成员）、（角色与权限）、BASE_INFO（基本信息）、FILE_STATISTICS（文件统计）、CATEGORY（业务分类）、FEE（费用中心）
     * @return 授权链接
     */
    @Override
    public CommonResponse<SdkResponse<SaaSPrivilegeUrlResult>> queryCompanyPrivilegeUrl(Long companyId, Long appId, User user, Boolean createToken, String successUrl, String callbackUrl, List<String> privilegeModules) {
        logger.info("契约锁------企业授权--获取授权链接，入参：companyId--{}，appId--{}，user--{}，createToken--{}，successUrl--{}，callbackUrl--{}，privilegeModules--{}", companyId, appId, JSONObject.toJSONString(user, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue), createToken, successUrl, callbackUrl, JSONObject.toJSONString(privilegeModules, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        // 初始化sdkClient
        SaaSSdkClient sdkClient = getSaasSdkClient();

        SaasPrivilegeUrlRequest urlRequest = new SaasPrivilegeUrlRequest();
        if (companyId == null || user == null) {
            throw new BusinessException("企业授权--获取授权链接失败，失败原因：请求参数（企业Id或操作人信息）为空！");
        }
        urlRequest.setCompanyId(companyId);
        urlRequest.setUser(user);
        if (appId != null) {
            urlRequest.setAppId(appId);
        }
        if (createToken != null) {
            urlRequest.setCreateToken(createToken);
        }
        if (StringUtils.isNotBlank(successUrl)) {
            urlRequest.setSuccessUrl(successUrl);
        }
        if (StringUtils.isNotBlank(callbackUrl)) {
            urlRequest.setCallbackUrl(callbackUrl);
        }
        if (!privilegeModules.isEmpty()) {
            urlRequest.setPrivilegeModules(privilegeModules);
        }

        String response = sdkClient.service(urlRequest);
        SdkResponse<SaaSPrivilegeUrlResult> responseObject = JSONUtils.toQysResponse(response, SaaSPrivilegeUrlResult.class);
        if (responseObject.getCode() != 0) {
            logger.info("企业授权--获取授权链接失败，失败原因：{}", responseObject.getMessage());
        }
        logger.info("企业授权--获取授权链接成功，返回参数：{}", JSONObject.toJSONString(responseObject, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        return CommonResponse.success(responseObject);
    }


    @Override
    public String ssoLogout() {

        // 拼接单点登录链接
        String ssoLogoutUrl = ssoUrl + "/saas/ssogateway";

        return HttpUtil.get(ssoLogoutUrl, 5);
    }

    /**
     * CAS验证
     * 该接口用于验证SaaS调用单点登录接口时传入的ticket参数是否正确（是否由SaaS服务商提供并颁发），并将对应的ticket兑换为正确的登录用户信息。
     * 注意：为防重放攻击，请保证每一个ticket在经过该接口验证完成后，都会失效。
     * 请求方式：POST
     *
     * @param ticket 单点登录请求凭证，由SaaS服务商调用单点登录接口时传入
     * @return String 登录用户信息
     */
    @Override
    public String casVerification(String ticket) {
        logger.info("契约锁------CAS验证--获取登录用户信息，入参：ticket--{}", ticket);

        String ticketDetail = jedisTemplate.get(ticket);
        if (StringUtils.isBlank(ticketDetail)) {
            logger.info("当前单点登录请求凭证--ticket：{}，获取登录用户信息失败！", ticket);
        }

        // 防止重放--获取后从redis删除ticket
        Boolean delFlag = jedisTemplate.del(ticket);
        if (!delFlag) {
            logger.info("当前单点登录请求凭证--ticket：{}，删除失败！", ticket);
        }

        return ticketDetail;
    }


    /**
     * 获取当前登录人信息
     *
     * @return UserVO
     */
    @Override
    public CommonResponse<UserVO> getUserByUserId() {
        // 获取当前用户信息
        CommonResponse<UserVO> user = userApi.findUserByUserId(InvocationInfoProxy.getUserid());
        if (!user.isSuccess()) {
            logger.info("查询用户（{}）信息失败，失败原因：{}", InvocationInfoProxy.getUserid(), user.getMsg());
            throw new BusinessException("查询用户信息失败，失败原因：" + user.getMsg());
        }
        // 校验用户姓名和手机号
        if (StringUtils.isBlank(user.getData().getUserName()) || StringUtils.isBlank(user.getData().getUserMobile())) {
            throw new BusinessException("当前用户（" + InvocationInfoProxy.getUserid() + "）姓名或手机号为空，请维护相关信息后再试！");
        }
        return user;
    }

    /**
     * 获取当前登录组织信息
     *
     * @return EnterpriseVO
     */
    @Override
    public CommonResponse<EnterpriseVO> getEnterpriseByTenantId() {
        CommonResponse<EnterpriseVO> resp = enterpriseApi.getEnterpriseByTenantId(InvocationInfoProxy.getTenantid());
        logger.info("获取当前登录组织信息，入参：{}，出参：{}", InvocationInfoProxy.getTenantid(), JSONObject.toJSONString(resp, SerializerFeature.PrettyFormat, SerializerFeature.WriteMapNullValue));
        if (!resp.isSuccess() || StringUtils.isBlank(resp.getData().getFullName())) {
            throw new BusinessException("查询查询公司信息失败，失败原因：未获取到当前用户公司名称!");
        }
        return resp;
    }


    /**
     * 生成单点登录链接
     * <p>
     * 单点登录
     * 请求域名：https://cloudapi.qiyuesuo.cn（测试环境）https://cloudapi.qiyuesuo.com（正式环境）
     * 请求地址：/saas/ssogateway
     * 请求方式：GET
     * <p>
     * ticket        是 单点登录请求凭证（UUID），凭证由SaaS服务商生成、维护，需保证凭证的仅支持单次验证特性，即完成一次验证即失效
     * pageType      否 单点登录指定的目标页面，默认为主页INDEX_PAGE（“主页”）、SEAL_PAGE（“公司公章页面”）、TEMPLATE_PAGE（“模板页面”） 、CATEGORY_PAGE（“公司业务分类页面”）、ROLE_PAGE（“公司角色权限页面”）、CONTRACT_LIST_PAGE（“合同列表页面”）、CONTRACT_STATISTICS_PAGE（“公司合同统计页面”）、COMPANY_INFO_PAGE（“公司基本信息页面”）、ORGANIZE_PAGE（“公司组织架构页面“）、COMPANY_FEE_PAGE（”公司费用中心“）、CONTRACT_DETAIL_PAGE（”合同签署页面“）。H5页面仅支持CONTRACT_LIST_PAGE、CONTRACT_DETAIL_PAGE、INDEX_PAGE以个人身份单点登录时，仅支持指定以上页面说明中，非公司开头的页面
     * companyId     否 单点登录指定的公司ID，指定后用户会以指定的公司身份完成登录，登录的用户必须为该公司的员工，若未指定，则以个人身份登录
     * contractId    否 合同ID，仅指定CONTRACT_DETAIL_PAGE（”合同签署页面“）时生效
     * hideSidebar   否 是否隐藏页面的侧边栏，默认不隐藏
     * saasClient    否 自定义参数，用于判断客户端类型，例如h5,web
     *
     * @return String url
     */
    @Override
    public String ssoLoginUrl(String companyId, String saasClient) {
        // 单点登录前生成凭证，并保存用户信息提供给契约锁验证
        String uuid = IdUtil.simpleUUID();

        // 获取当前用户信息
        CommonResponse<UserVO> user = this.getUserByUserId();

        TicketDetailVO ticketDetail = new TicketDetailVO();
        ticketDetail.setName(user.getData().getUserName());
        ticketDetail.setContact(user.getData().getUserMobile());

        // 将数据放入redis
        jedisTemplate.set(uuid, JSONObject.toJSONString(ticketDetail));// 生成凭证

        // 拼接单点登录链接
        return ssoUrl + "/saas/ssogateway?ticket=" + uuid + "&pageType=CONTRACT_LIST_PAGE" + "&companyId=" + companyId + "&hideSidebar=true" + "&saasClient=" + saasClient;
    }


}
