package com.ejianc.business.tax.util;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.market.api.IProjectApi;
import com.ejianc.business.market.vo.ProjectRegisterVO;
import com.ejianc.business.pub.utils.HttpTookit;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.share.vo.CustomerVO;
import com.ejianc.foundation.share.vo.SupplierVO;
import com.ejianc.foundation.support.api.ICustomerApi;
import com.ejianc.foundation.support.api.IDefdocApi;
import com.ejianc.foundation.support.api.ISupplierApi;
import com.ejianc.foundation.support.api.ISupplierWebApi;
import com.ejianc.foundation.support.vo.BankAccountVO;
import com.ejianc.foundation.support.vo.DefdocDetailVO;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.skeleton.fields.service.ICommenQueryFieldsService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yqls
 * @date 2021-12-01 10:42:16
 */
@Component
public class NCUtil {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private ICommenQueryFieldsService commenQueryFieldsService;

    @Autowired
    private IOrgApi orgApi;

    @Autowired
    private IProjectApi projectApi;

    @Autowired
    private ICustomerApi customerApi;

    @Autowired
    private ISupplierApi supplierApi;

    @Autowired
    private ISupplierWebApi supplierWebApi;

    @Autowired
    private IDefdocApi defdocApi;

    @Value("${nc.ncArUrl}")
    private String ncArUrl;

    private final static Map<Integer, String> invoiceMainMap;// 开票主体分类

    private final static Map<Integer, String> taxWayMap;// 计税方式

    static {
        invoiceMainMap = new HashMap<>();
        invoiceMainMap.put(0, "PMDA06001");// 以集团名义开票
        invoiceMainMap.put(1, "PMDA06002");// 以区域名义开票

        taxWayMap = new HashMap<>();
        taxWayMap.put(0, "01");// 一般纳税人
        taxWayMap.put(1, "02");// 小规模纳税人
        taxWayMap.put(2, "03");// 其它
    }

    /**
     * 开票主体分类自定义档案编码
     * @param key
     * @return
     */
    public String getInvoiceMainDefCode(Integer key){
        if(invoiceMainMap.containsKey(key)){
            return invoiceMainMap.get(key);
        }
        return null;
    }

    /**
     * 计税方式自定义档案编码
     * @param key
     * @return
     */
    public String getTaxWayDefCode(Integer key){
        if(taxWayMap.containsKey(key)){
            return taxWayMap.get(key);
        }
        return null;
    }

    /**
     * 税率参照
     * @param taxRate
     * @return
     */
    public String getTaxRateDefCode(BigDecimal taxRate){
        if(taxRate == null){
            throw new BusinessException("税率不能为空");
        }
        List<DefdocDetailVO> defList = this.getDefDocListById(515260415240835083L);
        Map<String, String> defMap = defList.stream().collect(Collectors.toMap(x->x.getName(), DefdocDetailVO::getCode));
        String tax = taxRate.stripTrailingZeros().toPlainString() + "%";
        if(defMap.containsKey(tax)){
            return defMap.get(tax);
        } else {
            throw new BusinessException("录入税率“" + tax + "”和财务税率不一致，请联系系统管理员处理！");
        }
    }

    /**
     * 根据分类id查询分类下所有自定义档案
     * @param id
     * @return
     */
    public List<DefdocDetailVO> getDefDocListById(Long id){
        CommonResponse<List<DefdocDetailVO>> resp = defdocApi.getDefDocByDefId(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询自定义档案
     * @param id
     * @return
     */
    public DefdocDetailVO getDefDocById(Long id){
        if(id == null) return new DefdocDetailVO();
        CommonResponse<DefdocDetailVO> resp = defdocApi.getDefDocById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询组织
     * @param id
     * @return
     */
    public OrgVO queryOrgById(Long id){
        CommonResponse<OrgVO> resp = orgApi.detailById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id集合查询项目列表
     * @param ids
     * @return
     */
    public List<ProjectRegisterVO> queryProjectByIds(List<Long> ids){
        CommonResponse<List<ProjectRegisterVO>> resp = projectApi.queryProjectByIds(ids);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询供应商
     * @param id
     * @return
     */
    public SupplierVO querySupplierById(Long id){
        CommonResponse<SupplierVO> resp = supplierApi.queryById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询客户
     * @param id
     * @return
     */
    public CustomerVO queryCustomerById(Long id){
        CommonResponse<CustomerVO> resp = customerApi.detailById(id);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 根据id查询默认银行账户
     * @param id
     * @param type  supplierId or customerId or insideOrgId
     * @return
     */
    public BankAccountVO queryDefaultBankAccountById(Long id, String type){
        CommonResponse<BankAccountVO> resp = supplierWebApi.queryDefaultBankAccountById(id, type);
        if(!resp.isSuccess()){
            throw new BusinessException(resp.getMsg());
        }
        return resp.getData();
    }

    /**
     * 如果道桥分公司本下，且以区域分公司开票，则传项目所属组织和道桥分公司，如果以集团名义开票，则传道桥分公司
     * 其他组织，还传项目所属组织
     * @param orgId 项目所属组织
     * @param invoiceMain 开票主体分类 0-以集团名义开票，1-以区域分公司名义开票
     * @return List<Long> 返回几个执行几次，财务组织使用对应的返回值
     */
    public List<Long> validateDaoQiaoOrg(Long orgId, Integer invoiceMain){
        Long daoQiaoId = 1406795679766782028L;// 青岛市政空间开发集团道桥分公司
        OrgVO orgVO = this.queryOrgById(orgId);
        // 道桥分公司本下
        if(orgVO.getInnerCode().contains(String.valueOf(daoQiaoId))) {
            if(new Integer(0).equals(invoiceMain)){
                return Arrays.asList(daoQiaoId);
            } else if(new Integer(1).equals(invoiceMain)){
                return Arrays.asList(orgId, daoQiaoId);
            }
        }
        return Arrays.asList(orgId);
    }

    /**
     *  调用NC传凭证接口（应收、应付、收款、付款）
     * @param param
     * @return
     */
    public String postByJson(String param){
        Map<String, String> headers = new HashMap<>();
        logger.info("-传NC凭证入参:" + param);
        String back = null;
        try {
            back = HttpTookit.postByJson(ncArUrl, param, headers);
        } catch (Exception e) {
            throw new BusinessException("推送NC接口异常!");
        }
        logger.info("-传NC凭证回参:" + back);
        JSONObject json = JSONObject.parseObject(back);
        if(!"200".equals(json.getString("code"))){
            throw new BusinessException(json.getString("msg"));
        }else{
            return json.getString("result");
        }
    }

    /**
     * 查询单据明细
     * @param businessKey 单据主键
     * @param billTypeCode 单据类型主键
     * @return
     */
    public JSONObject queryBillDetail(String businessKey, String billTypeCode){
        JSONObject json = commenQueryFieldsService.queryBillDetail(businessKey, billTypeCode);
        if(json.getInteger("code") != 0){
            throw new BusinessException(json.getString("msg"));
        }
        return json.getJSONObject("data");
    }

}
