package com.ejianc.business.weigh.wagon.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.weigh.wagon.bean.WagonProjectEntity;
import com.ejianc.business.weigh.wagon.enums.AutoWeighEnums;
import com.ejianc.business.weigh.wagon.service.IWagonProjectService;
import com.ejianc.business.weigh.wagon.vo.WagonMaterialVO;
import com.ejianc.foundation.orgcenter.api.IOrgApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;

import com.ejianc.business.weigh.wagon.mapper.WagonMaterialMapper;
import com.ejianc.business.weigh.wagon.bean.WagonMaterialEntity;
import com.ejianc.business.weigh.wagon.service.IWagonMaterialService;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 称重材料设置
 *
 * @author generator
 */
@Service("wagonMaterialService")
public class WagonMaterialServiceImpl extends BaseServiceImpl<WagonMaterialMapper, WagonMaterialEntity> implements IWagonMaterialService {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IOrgApi iOrgApi;

    @Autowired
    private IWagonProjectService wagonProjectService;

    /**
     * 批量查询材料称重设置
     *
     * @param voList 材料
     */
    @Override
    public Map<String, Object> insertBatch(List<WagonMaterialVO> voList) {
        if (CollectionUtils.isEmpty(voList)) throw new BusinessException("插入列表不能为空！");
        // 校验
        Long orgId = InvocationInfoProxy.getOrgId();
        // 唯一性校验（材料id+orgID唯一）
        List<Long> materialIdList = voList.stream().map(WagonMaterialVO::getMaterialId).collect(Collectors.toList());
        QueryParam checkParam = new QueryParam();
        checkParam.getParams().put("orgId", new Parameter(QueryParam.EQ, orgId));
        checkParam.getParams().put("materialId", new Parameter(QueryParam.IN, materialIdList));
        List<WagonMaterialEntity> checkList = super.queryList(checkParam, false);
        List<WagonMaterialVO> insertList = new ArrayList<>();
        List<String> errorMsgList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(checkList)) {
            List<Long> checkMaterialIdList = checkList.stream().map(WagonMaterialEntity::getMaterialId)
                    .collect(Collectors.toList());
            for (WagonMaterialVO vo : voList) {
                // 查询结果中不包含该材料id，则允许保存
                if (checkMaterialIdList.contains(vo.getMaterialId())) {
                    String errorMsg = "【" + vo.getMaterialName() + "】在当前组织中已存在，请勿重复添加！";
                    errorMsgList.add(errorMsg);
                }
                else {
                    insertList.add(vo);
                }
            }
        }
        else {
            insertList.addAll(voList);
        }
        // 处理数据
        CommonResponse<OrgVO> orgResponse = iOrgApi.getOneById(orgId);
        logger.info("查询组织结果：{}", JSONObject.toJSONString(orgResponse));
        if (!orgResponse.isSuccess()) {
            throw new BusinessException("查询组织信息失败！");
        }
        OrgVO orgVO = orgResponse.getData();
        for (WagonMaterialVO vo : insertList) {
            vo.setOrgCode(orgVO.getCode());
            vo.setOrgId(orgVO.getId());
            vo.setOrgName(orgVO.getName());
            vo.setInnerCode(orgVO.getInnerCode());
            vo.setParentOrgCode(orgVO.getpCode());
            vo.setParentOrgId(orgVO.getParentId());
            vo.setParentOrgName(orgVO.getpFName());
            vo.setAutomaticWeigh(AutoWeighEnums.关闭自动称重.getCode());
        }

        List<WagonMaterialVO> rtnVoList;
        if (CollectionUtils.isNotEmpty(insertList)) {
            List<WagonMaterialEntity> insertEntityList = BeanMapper.mapList(insertList, WagonMaterialEntity.class);
            super.saveOrUpdateBatch(insertEntityList, insertEntityList.size());
            rtnVoList = BeanMapper.mapList(insertEntityList, WagonMaterialVO.class);
        }
        else {
            rtnVoList = new ArrayList<>();
        }
        return getBatchResultMap(errorMsgList, rtnVoList);
    }

    /**
     * 修改自动称重开关
     *
     * @param saveOrUpdateVO 修改信息
     */
    @Override
    public void updateAutomaticWeigh(WagonMaterialVO saveOrUpdateVO) {
        if (saveOrUpdateVO.getId() == null) {
            throw new BusinessException("修改主键不能为空！");
        }
        WagonMaterialEntity saveEntity = super.selectById(saveOrUpdateVO.getId());
        if (saveEntity.getScaleFactor() == null) {
            throw new BusinessException("请先设置换算系数！");
        }
        saveEntity.setAutomaticWeigh(saveOrUpdateVO.getAutomaticWeigh());
        super.saveOrUpdate(saveEntity, false);
    }

    /**
     * 批量设置开启状态及换算系数
     *
     * @param voList 材料列表
     * @return 设置结果
     */
    @Override
    public Map<String, Object> updateAutomaticWeighBatch(List<WagonMaterialVO> voList) {
        if (CollectionUtils.isEmpty(voList)) throw new BusinessException("插入列表不能为空！");
        Long orgId = InvocationInfoProxy.getOrgId();
        // 唯一性校验（材料id+orgID唯一）
        List<Long> idList = voList.stream().map(WagonMaterialVO::getId).collect(Collectors.toList());
        // 批量设置，都取同一个值
        Integer automaticWeigh = voList.get(0).getAutomaticWeigh();
        BigDecimal scaleFactor = voList.get(0).getScaleFactor();
        QueryParam param = new QueryParam();
        param.getParams().put("id", new Parameter(QueryParam.IN, idList));
        List<WagonMaterialEntity> entityList = super.queryList(param, false);

        List<String> errorMsgList = new ArrayList<>(); // 异常信息存储
        List<WagonMaterialEntity> saveEntityList = new ArrayList<>(); // 需要保存的数据
        for (WagonMaterialEntity entity : entityList) {
            if (orgId.equals(entity.getOrgId())) {
                entity.setAutomaticWeigh(automaticWeigh);
                entity.setScaleFactor(scaleFactor);
                saveEntityList.add(entity);
            }
            else {
                String errorMsg = "材料【" + entity.getMaterialName() + "】不在当前组织下，无法进行设置！";
                errorMsgList.add(errorMsg);
            }
        }
        List<WagonMaterialVO> rtnVoList;
        if (CollectionUtils.isNotEmpty(saveEntityList)) {
            super.saveOrUpdateBatch(saveEntityList, saveEntityList.size());
            rtnVoList = BeanMapper.mapList(saveEntityList, WagonMaterialVO.class);
        }
        else {
            rtnVoList = new ArrayList<>();
        }
        return getBatchResultMap(errorMsgList, rtnVoList);
    }

    /**
     * 获取批量处理结果
     *
     * @param errorMsgList 异常信息列表
     * @param rtnVoList    处理结果列表
     * @return 处理结果map
     */
    private Map<String, Object> getBatchResultMap(List<String> errorMsgList, List<WagonMaterialVO> rtnVoList) {
        Map<String, Object> rtnMap = new HashMap<>();
        rtnMap.put("successSize", rtnVoList.size());
        rtnMap.put("failSize", errorMsgList.size());
        rtnMap.put("successList", rtnVoList);
        rtnMap.put("errorMsgList", errorMsgList);
        return rtnMap;
    }

    /**
     * 查询项目下材料自动称重设置，本上查询，查询最近的设置
     *
     * @param projectId  项目id
     * @param materialId 材料id
     * @return 查询结果，如果未设置，返回空对象
     */
    @Override
    public WagonMaterialVO queryMaterialAutomaticWeigh(Long projectId, Long materialId) {
//        Long orgId = InvocationInfoProxy.getOrgId();
        List<WagonMaterialVO> rtnList = this
                .queryMaterialAutomaticWeighBatch(projectId, Collections.singletonList(materialId));
        if (CollectionUtils.isNotEmpty(rtnList)) {
            return rtnList.get(0);
        }
        return new WagonMaterialVO();
    }

    /**
     * 查询项目下材料自动称重设置，本上查询，查询最近的设置
     *
     * @param projectId      项目id
     * @param materialIdList 材料id列表
     * @return 查询结果，如果未设置，返回空对象
     */
    @Override
    public List<WagonMaterialVO> queryMaterialAutomaticWeighBatch(Long projectId, List<Long> materialIdList) {
        logger.info("项目id：{}，材料id：{}；开始查询项目下材料自动称重设置>>>>>>>>>>>>>>>>>>>>>>", projectId,
                JSONObject.toJSONString(materialIdList));
        List<WagonMaterialVO> rtnList = new ArrayList<>();
        if (projectId == null || CollectionUtils.isEmpty(materialIdList)) {
            return rtnList;
        }
        materialIdList = materialIdList.stream().distinct().collect(Collectors.toList());
        // 1. 查询项目设置
        QueryParam projectParam = new QueryParam();
        projectParam.getParams().put("projectId", new Parameter(QueryParam.EQ, projectId));
        List<WagonProjectEntity> projectEntityList = wagonProjectService.queryList(projectParam, false);
        logger.info("查询称重项目设置结果：{}", JSONObject.toJSONString(projectEntityList));
        // 1.1 项目是否设置称重信息
        if (CollectionUtils.isEmpty(projectEntityList)) {
            logger.info("项目[{}]未设置自动称重信息", projectId);
            return rtnList;
        }
        // 1.2 是否存在启用的地磅
        boolean isProjectStartFlag = false;
        for (WagonProjectEntity entity : projectEntityList) {
            // 存在启用
            if (entity.getStartFlag() == 0) {
                isProjectStartFlag = true;
                break;
            }
        }
        // 不存在启用地磅，返回空
        if (!isProjectStartFlag) {
            logger.info("项目[{}]不存在启用的地磅", projectId);
            return rtnList;
        }

        // 获取项目部orgId
        Long orgId = projectEntityList.get(0).getOrgId();
        // 2. 查询称重材料设置
        // 2.1 查询组织信息
        CommonResponse<OrgVO> orgResponse = iOrgApi.getOneById(orgId);
        logger.info("查询组织结果：{}", JSONObject.toJSONString(orgResponse));
        if (!orgResponse.isSuccess()) {
            throw new BusinessException("查询组织信息失败！");
        }
        OrgVO orgVO = orgResponse.getData();
        String[] innerCodes = orgVO.getInnerCode().split("\\|");
        // 2.2 查询组织下材料（本上）
        QueryParam materialParam = new QueryParam();
        materialParam.getParams().put("materialId", new Parameter(QueryParam.IN, materialIdList));
        materialParam.getParams().put("orgId", new Parameter(QueryParam.IN, Arrays.asList(innerCodes)));
        List<WagonMaterialEntity> materialEntityList = super.queryList(materialParam, false);
        logger.info("查询称重材料设置结果：{}", JSONObject.toJSONString(materialEntityList));
        if (CollectionUtils.isEmpty(materialEntityList)) {
            logger.info("材料[{}]未设置自动称重信息", JSONObject.toJSONString(materialIdList));
            return rtnList;
        }
        // 2.2.1 按照材料分组查询
        Map<Long, List<WagonMaterialEntity>> groupByMaterialMap = materialEntityList.stream()
                .collect(Collectors.groupingBy(WagonMaterialEntity::getMaterialId));
        // 2.2.2 循环材料，查询材料符合条件的
        for (Long materialId : materialIdList) {
            List<WagonMaterialEntity> wagonEntityList = groupByMaterialMap.get(materialId);
            logger.info("材料[{}]称重信息：{}", materialId, JSONObject.toJSONString(wagonEntityList));
            if (CollectionUtils.isEmpty(wagonEntityList)) {
                logger.info("材料[{}]未设置自动称重信息", materialId);
                continue;
            }
            // 2.2.3 按照orgId分组
            Map<Long, List<WagonMaterialEntity>> wagonMaterialMap = wagonEntityList.stream()
                    .collect(Collectors.groupingBy(WagonMaterialEntity::getOrgId));
            // 2.2.4 使用组织内码，从下往上找最近设置的
            for (int i = innerCodes.length - 1; i >= 0; i--) {
                Long innerOrgId = Long.parseLong(innerCodes[i]);
                List<WagonMaterialEntity> orgMaterialList = wagonMaterialMap.get(innerOrgId);
                if (CollectionUtils.isEmpty(orgMaterialList)) {
                    logger.info("材料id[{}]组织id[{}]未设置自动称重信息", materialId, innerOrgId);
                    continue;
                }
                // 材料+组织唯一
                WagonMaterialEntity wagonMaterialEntity = orgMaterialList.get(0);
                logger.info("材料id[{}]组织id[{}]设置自动称重结果：{}，详情信息：{}", materialId, innerOrgId,
                        AutoWeighEnums.getEnumByCode(wagonMaterialEntity.getAutomaticWeigh()).getDescription(),
                        JSONObject.toJSONString(wagonMaterialEntity));
                // 开启自动称重，返回该数据
                if (AutoWeighEnums.开启自动称重.getCode().equals(wagonMaterialEntity.getAutomaticWeigh())) {
                    WagonMaterialVO rtnVo = BeanMapper.map(wagonMaterialEntity, WagonMaterialVO.class);
                    rtnList.add(rtnVo);
                    break;
                }
            }
        }
        logger.info("<<<<<<<<<<<<<<<<项目id：{}，材料id：{}；查询项目下材料自动称重设置结束，查询结果：{}", projectId,
                JSONObject.toJSONString(materialIdList), JSONObject.toJSONString(rtnList));
        return rtnList;
    }
}
