package com.ejianc.business.zdsmaterial.erp.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.zdsmaterial.accept.bean.AcceptEntity;
import com.ejianc.business.zdsmaterial.cloudstore.constants.CommonConstants;
import com.ejianc.business.zdsmaterial.erp.bean.DeliveryDetailEntity;
import com.ejianc.business.zdsmaterial.erp.bean.DeliveryEntity;
import com.ejianc.business.zdsmaterial.erp.bean.OrderDetailEntity;
import com.ejianc.business.zdsmaterial.erp.bean.OrderEntity;
import com.ejianc.business.zdsmaterial.erp.enums.DeliveryAcceptStatusEnums;
import com.ejianc.business.zdsmaterial.erp.enums.DeliveryCloseStatusEnums;
import com.ejianc.business.zdsmaterial.erp.enums.OrderDeliveryStatusEnums;
import com.ejianc.business.zdsmaterial.erp.mapper.DeliveryMapper;
import com.ejianc.business.zdsmaterial.erp.service.IDeliveryDetailService;
import com.ejianc.business.zdsmaterial.erp.service.IDeliveryService;
import com.ejianc.business.zdsmaterial.erp.service.IOrderService;
import com.ejianc.business.zdsmaterial.erp.vo.DeliveryDetailVO;
import com.ejianc.business.zdsmaterial.erp.vo.DeliveryVO;
import com.ejianc.foundation.message.api.IPushMessageApi;
import com.ejianc.foundation.message.vo.PushMsgParameter;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.kit.time.DateFormatUtil;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseEntity;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 发货单
 *
 * @author generator
 *
 */
@Service("deliveryService")
public class DeliveryServiceImpl extends BaseServiceImpl<DeliveryMapper, DeliveryEntity> implements IDeliveryService{

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${common.env.base-host}")
    private String BASE_HOST;

    @Autowired
    private IDeliveryDetailService deliveryDetailService;

    private final String OPERATE = "deliveryBill";

    @Autowired
    private ISystemDataPushService systemDataPushService;

    @Autowired
    private JedisPool jedisPool;

    @Autowired
    private IOrderService orderService;

    @Autowired
    private DeliveryMapper mapper;

    @Autowired
    private IPushMessageApi pushMessageApi;

    private final String UPDATE_SUPDELIVERY_SERVER_URL = "/ejc-zdssupbusiness-web/openapi/supDelivery/syncBill";

    @Override
    public DeliveryVO saveSupDelivery(DeliveryVO deliveryVO) {
        OrderEntity order = null;

        //更新订单信息
        order = orderService.selectById(deliveryVO.getOrderId());
        Map<Long, OrderDetailEntity> orderDetail = order.getDetailList().stream().collect(Collectors.toMap(item -> item.getId(), item -> item));

        deliveryVO.getDetailList().stream().filter(item -> orderDetail.containsKey(item.getSourceDetailId())).forEach(detail -> {
            orderDetail.get(detail.getSourceDetailId()).setDeliveredNum(ComputeUtil.safeAdd(orderDetail.get(detail.getSourceDetailId()).getDeliveredNum(), detail.getDeliveryNum()));
        });

        order.setDeliveredMny(ComputeUtil.safeAdd(deliveryVO.getDeliveryMny(), order.getDeliveredMny()));
        order.setDeliveredTaxMny(ComputeUtil.safeAdd(deliveryVO.getDeliveryTaxMny(), order.getDeliveredTaxMny()));
        order.setDeliveryStatus(order.getOrderTaxMny().compareTo(order.getDeliveredTaxMny()) == 0 ? OrderDeliveryStatusEnums.全部发货.getCode() : OrderDeliveryStatusEnums.部分发货.getCode());
        orderService.saveOrUpdate(order, false);

        deliveryVO.setAcceptStatus(DeliveryAcceptStatusEnums.待验收.getCode()); //默认待验收
        DeliveryEntity saveEntity = BeanMapper.map(deliveryVO, DeliveryEntity.class);

        saveEntity.setSourceId(saveEntity.getId().toString());
        clearInvalidData(saveEntity);
        saveEntity.getDetailList().stream().forEach(detail -> {
            detail.setSourceId(detail.getId().toString());
            clearInvalidData(detail);
        });

        super.saveOrUpdate(saveEntity, false);

        //向合同接货人发送消息
        // 标题：【订单发货】：【单据编号】，【时间】已发货，请尽快验收！
        // 内容：【单据编号】，【项目名称】，【合同名称】，【供应商名称】的订单已发货。
        logger.info("订单发货向合同验收人id-{}发送消息", deliveryVO.getSiteAcceptorId().toString());
        PushMsgParameter parameter = new PushMsgParameter();
        parameter.setSubject("【订单发货】：【"+deliveryVO.getBillCode()+"】，【"+ DateFormatUtil.formatDate("yyyy-MM-dd HH:mm:ss", new Date()) +"】已发货，请尽快验收！");
        StringBuilder content = new StringBuilder();
        content.append("【").append(deliveryVO.getBillCode()).append("】，【")
                .append(deliveryVO.getProjectName()).append("】，【")
                .append(deliveryVO.getContractName()).append("】，【").append(deliveryVO.getSupplierName()).append("】的订单已发货");
        parameter.setContent(content.toString());
        parameter.setPcUrl(BASE_HOST + CommonConstants.物资验收PC详情 + saveEntity.getId().toString() + (null != saveEntity.getProjectId() ? "&projectId=" + saveEntity.getProjectId().toString() : ""));
        parameter.setMobileUrl(BASE_HOST + CommonConstants.物资验收移动端详情 + saveEntity.getId().toString() + (null != saveEntity.getProjectId() ? "&projectId=" + saveEntity.getProjectId().toString() : ""));

        String[] recIds = new String[]{deliveryVO.getSiteAcceptorId().toString()}; //验收人Id
        parameter.setReceivers(recIds);
        sendMsg(parameter, deliveryVO.getId(), "订单发货向合同验收人发送信息");

        return BeanMapper.map(saveEntity, DeliveryVO.class);
    }

    public void sendMsg(PushMsgParameter parameter, Long billId, String oprMsg) {
        parameter.setSaveFlag(true);
        parameter.setTenantId(InvocationInfoProxy.getTenantid().toString());
        parameter.setMsgType("notice");
        parameter.setChannel(new String[]{PushMsgParameter.CHANNEL_TYPE_SYS, PushMsgParameter.CHANNEL_TYPE_EMAIL});

        CommonResponse<String> sendResp = pushMessageApi.pushMessage(parameter);
        if (!sendResp.isSuccess()) {
            logger.error("{}-发送消息失败，单据Id-{}，发送消息结果-{}", oprMsg, billId, JSONObject.toJSONString(sendResp, SerializerFeature.PrettyFormat,
                    SerializerFeature.WriteMapNullValue));
        } else {
            logger.info("{}-发送消息成功！", oprMsg);
        }
    }

    private void clearInvalidData(BaseEntity entity) {
        entity.setCreateTime(null);
        entity.setUpdateTime(null);
        entity.setTenantId(InvocationInfoProxy.getTenantid());
        entity.setUpdateUserCode(null);
        entity.setCreateUserCode(null);
    }


    @Override
    public void updateAcceptStatus(Long id,Integer acceptStatus) {
        UpdateWrapper<DeliveryEntity> updateWrapper = new UpdateWrapper<>();
        updateWrapper.eq("id",id);
        updateWrapper.set("accept_status",acceptStatus);
        Integer closeStatus = DeliveryCloseStatusEnums.未关闭.getCode();
        if (acceptStatus.equals(DeliveryAcceptStatusEnums.验收完成.getCode())||acceptStatus.equals(DeliveryAcceptStatusEnums.部分退回.getCode())){
            closeStatus = DeliveryCloseStatusEnums.已关闭.getCode();
        }
        updateWrapper.set("close_status", closeStatus);

        super.update(updateWrapper);
    }

    @Override
    public String updateSupDelivery(DeliveryEntity deliveryEntity) {
        String msg = null;
        Jedis jedis = null;
        boolean locked = false;

        String key = OPERATE + "::" + deliveryEntity.getId().toString();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            //加锁失败
            if(!locked) {
                releaseLock(jedis, false, key, OPERATE);
                return  "发货单状态同步知供方失败，获取锁失败！";
            }

            Map<String, String> params = new HashMap<>();
            params.put("billData", JSONObject.toJSONString(deliveryEntity));

            CommonResponse<String> writeBackResp = systemDataPushService.exchangeDataAndFilesWithEachLinkSystem(UPDATE_SUPDELIVERY_SERVER_URL,
                    params, deliveryEntity.getSupplierId().toString(), null);
            logger.error("发货单状态同步知供方失败请求结果，{}", JSONObject.toJSONString(writeBackResp));
            if(!writeBackResp.isSuccess()) {
                releaseLock(jedis, true, key, OPERATE);
                logger.error("发货单id-{}状态同步知供方发送请求失败，{}", deliveryEntity.getId(), writeBackResp.getMsg());
                return "发货单状态同步供方失败";
            }

            String operateRespStr = writeBackResp.getData();
            CommonResponse<String> operateResp = JSONObject.parseObject(operateRespStr, CommonResponse.class);
            if(!operateResp.isSuccess()) {
                logger.error("发货单id-{}推送供应链，平台处理失败，{}", deliveryEntity.getId(), operateResp.getMsg());
                releaseLock(jedis, true, key, OPERATE);
                return "发货单状态同步供方失败";
            }

        } catch (Exception e) {
            logger.error("发货单id-{}状态同步供方异常，", deliveryEntity, e);
            msg = "操作失败，发货单状态同步供方失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }

        return msg;
    }

    @Override
    public void returnDeliveryOrOrder(AcceptEntity entity, Boolean status) {
        deliveryDetailService.returnAlreadyNum(entity,status);
        if (status){
            updateAcceptStatus(entity.getDeliveryBillId(),DeliveryAcceptStatusEnums.验收中.getCode());
        }else {
            //删除判断是否有剩余验收量 有的话就是部分验收
            deliveryDetailService.checkAcceptStatus(entity.getDeliveryBillId());
        }
        DeliveryEntity deliveryEntity = super.selectById(entity.getDeliveryBillId());
        String s = updateSupDelivery(deliveryEntity);
        if (StringUtils.isNotBlank(s)){
            throw new BusinessException(s);
        }
    }

    @Override
    public List<DeliveryVO> getAllByOrderId(Long orderId) {
        List<DeliveryVO> resp = new ArrayList<>();

        QueryWrapper<DeliveryEntity> query = new QueryWrapper<>();
        query.eq("order_id", orderId);
        query.orderByDesc("create_time");
        List<DeliveryEntity> list = super.list(query);

        if(CollectionUtils.isNotEmpty(list)) {
            return BeanMapper.mapList(list, DeliveryVO.class);
        }

        return resp;
    }

    @Override
    public List<DeliveryVO> queryPageData(Page<DeliveryVO> page, QueryWrapper wrapper) {
        return baseMapper.queryPageData(page, wrapper);
    }

    @Override
    public DeliveryEntity getBySourceId(String sourceId) {
        QueryWrapper<DeliveryEntity> query = new QueryWrapper<>();
        query.eq("source_id", sourceId);

        DeliveryEntity e = super.getOne(query);
        if(null != e) {
            List<DeliveryDetailEntity> detailList = deliveryDetailService.getAllByDeliveryId(e.getId());
            e.setDetailList(detailList);
        }

        return e;
    }

    @Override
    public String billDel(DeliveryEntity delivery) {
        String msg = null;
        Jedis jedis = null;
        boolean locked = false;

        String key = OPERATE + "::" + delivery.getId().toString();

        try {
            jedis = jedisPool.getResource();
            //对单据进行加锁
            locked = RedisTool.tryLock(jedis, key, OPERATE, 600);
            //加锁失败
            if(!locked) {
                releaseLock(jedis, false, key, OPERATE);
                return  "单据作废失败，单据数据已被修改！";
            }

            //删除单据
            boolean delResult  = deleteByDeliveryId(delivery.getId());
            if(!delResult) {
                releaseLock(jedis, true, key, OPERATE);
                return "操作失败，单据删除失败！";
            }

        } catch (Exception e) {
            logger.error("发货单id-{}作废异常，", delivery.getId(), e);
            msg = "操作失败！";
        } finally {
            releaseLock(jedis, locked, key, OPERATE);
        }


        return msg;
    }

    private boolean deleteByDeliveryId(Long deliveryId) {
        mapper.deleteByDeliveryId(deliveryId);
        deliveryDetailService.deleteByDeliveryId(deliveryId);
        return true;
    }

    public void releaseLock(Jedis jedis, boolean locked, String key, String OPERATE) {
        try {
            if(locked) {
                RedisTool.releaseLock(jedis, key, OPERATE);
            }
        } finally {
            if(null != jedis) {
                jedis.close();
            }
        }
    }


}
