package com.ejianc.business.zdsmaterial.pricelib.processor;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.zdsmaterial.material.vo.MaterialCategoryVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceCalcSubConditionVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.PriceLibVO;
import com.ejianc.business.zdsmaterial.util.ComputeUtil;
import jodd.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

/**
 * 电力电缆单品牌计算公式处理器
 *
 * 公式：基础单价 *（ 1 +（原材料网价-原材料基础价）/ 系数1 * 系数2）
 *
 *
 * @Author: caojie
 * @CreateTime:2025-07-07 10:00
 * @Version: 1.0
 */
@Component
public class D1101SingleBrandProcessor implements BaseCalcProcessor {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public boolean calcBeforeCheck(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {
        MaterialCategoryVO categoryVO = priceLibVO.getCategoryVO();
        String categoryName = "";
        if(null != categoryVO) {
            categoryName = categoryVO.getName();
        }

        if(null == priceLibVO.getBaseMaterialNetPrice()) {
            logger.error("【{}】公式：D1101SingleBrandProcessor 计算终止，原材料网价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料网价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialUnitPrice()) {
            logger.error("【{}】公式：D1101SingleBrandProcessor 计算终止，原材料基础价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：原材料基础价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialPrice()) {
            logger.error("【{}】公式：D1101SingleBrandProcessor 计算终止，基础单价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：基础单价为空");
            return false;
        }
        if(StringUtil.isBlank(calcFormula.getCalcParams())) {
            logger.error("【{}】公式：D1101SingleBrandProcessor 计算终止， 辅助计算参数为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：辅助计算参数为空");
            return false;
        }

        return true;
    }

    @Override
    public BigDecimal calc(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {

        //公式：基础单价 *（ 1 +（原材料网价-原材料基础价）/ 系数1 * 系数2）

        if(!calcBeforeCheck(priceLibVO, calcFormula)) {
            return BigDecimal.ZERO;
        }
        priceLibVO.setCalcContent(calcFormula.getCalcDisplayContent());
        JSONObject params = JSONObject.parseObject(calcFormula.getCalcParams());

        BigDecimal result = BigDecimal.ZERO;

        //计算 原材料网价- 原材料基础价
        result = ComputeUtil.safeSub(priceLibVO.getBaseMaterialNetPrice(), priceLibVO.getBaseMaterialUnitPrice());
        //计算  中间结果 / 系数1
        result = ComputeUtil.safeDiv(result, new BigDecimal(params.getString("参数1")));
        //计算 中间结果 * 系数2 + 1
        result = ComputeUtil.safeAdd(ComputeUtil.safeMultiply(result, new BigDecimal(params.getString("参数2"))), BigDecimal.ONE);
        //计算 中间结果 * 基础单价
        result = ComputeUtil.safeMultiply(result, priceLibVO.getBaseMaterialPrice());
        priceLibVO.setAdjustedPrice(result);
        priceLibVO.setAdjustedFirstPrice(result);

        priceLibVO.setCalcMny(ComputeUtil.safeMultiply(result, priceLibVO.getSourceNum()));

        return result;
    }
}
