package com.ejianc.business.zdsmaterial.pricelib.processor;

import com.alibaba.fastjson.JSONObject;
import com.ejianc.business.zdsmaterial.material.vo.MaterialCategoryVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.MaterialPriceCalcSubConditionVO;
import com.ejianc.business.zdsmaterial.pricelib.vo.PriceLibVO;
import com.ejianc.business.zdsmaterial.util.ComputeUtil;
import jodd.util.StringUtil;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

/**
 * 碳钢管道单品牌计算公式处理器
 *
 * 公式：基准价格(单价)*(1+（浮动网价-基准原材网料价）/ 基准原材料网价)+米重*运费*0.001
 *
 *
 * @Author: caojie
 * @CreateTime:2025-07-07 10:00
 * @Version: 1.0
 */
@Component
public class C3001BrandAndPropertyProcessor implements BaseCalcProcessor {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public boolean calcBeforeCheck(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {
        MaterialCategoryVO categoryVO = priceLibVO.getCategoryVO();
        String categoryName = "";
        if(null != categoryVO) {
            categoryName = categoryVO.getName();
        }
        if(null == priceLibVO.getBaseMaterialNetPrice()) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止，原材料网价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("原材料网价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialUnitPrice()) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止，原材料基础价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("原材料基础价为空");
            return false;
        }
        if(null == priceLibVO.getBaseMaterialPrice()) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止，基础单价为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("基础单价为空");
            return false;
        }
        if(CollectionUtils.isEmpty(calcFormula.getDetailList())) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止， 计算规则元素清单为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算规则元素清单为空");
            return false;
        }
        if(StringUtil.isBlank(priceLibVO.getExtParam1())) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止， 米重为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[米重]为空");
            return false;
        }
        if(StringUtil.isBlank(priceLibVO.getCalcFields())) {
            logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止， 距离为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
            priceLibVO.setCalcContent("未计算：计算因素[距离]为空");
            return false;
        }
        JSONObject paramJson = JSONObject.parseObject(priceLibVO.getCalcFields());
        if(null == paramJson.get("distance")) {
            if(StringUtil.isBlank(priceLibVO.getCalcFields())) {
                logger.error("【{}】公式：C3001BrandAndPropertyProcessor 计算终止， 距离为空, {}", categoryName, JSONObject.toJSONString(priceLibVO));
                priceLibVO.setCalcContent("未计算：计算因素[距离]为空");
                return false;
            }
        }

        return true;
    }

    @Override
    public BigDecimal calc(PriceLibVO priceLibVO, MaterialPriceCalcSubConditionVO calcFormula) {

        //公式：基准价格(单价)*(1+（浮动网价-基准原材网料价）/ 基准原材料网价)+米重*运费*0.001

        if(!calcBeforeCheck(priceLibVO, calcFormula)) {
            return BigDecimal.ZERO;
        }
        priceLibVO.setCalcContent(calcFormula.getCalcDisplayContent());
        JSONObject params = JSONObject.parseObject(priceLibVO.getCalcFields());

        BigDecimal result = BigDecimal.ZERO;

        //计算 原材料网价- 原材料基础价
        result = ComputeUtil.safeSub(priceLibVO.getBaseMaterialNetPrice(), priceLibVO.getBaseMaterialUnitPrice());
        //计算 中间结果 / 基准原材料网价
        result = ComputeUtil.safeDiv(result, priceLibVO.getBaseMaterialUnitPrice());
        //计算 中间结果 + 1
        result = ComputeUtil.safeAdd(result, BigDecimal.ONE);
        //计算 中间结果 * 基准价格(单价)
        result = ComputeUtil.safeMultiply(result, priceLibVO.getBaseMaterialPrice());
        //计算 中间结果 + 米重*运费*0.001
        result = ComputeUtil.safeAdd(result, ComputeUtil.safeMultiply(ComputeUtil.safeMultiply(new BigDecimal(params.getString("distance")), new BigDecimal(priceLibVO.getExtParam1())), new BigDecimal("0.001")));
        priceLibVO.setAdjustedPrice(result);
        priceLibVO.setAdjustedFirstPrice(result);

        priceLibVO.setCalcMny(ComputeUtil.safeMultiply(result, priceLibVO.getSourceNum()));

        return result;
    }
}
