package com.ejianc.business.panhuo.shelf.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringPool;
import com.ejianc.business.material.bean.FlowmeterEntity;
import com.ejianc.business.material.service.IFlowmeterService;
import com.ejianc.business.material.utils.OrgUtil;
import com.ejianc.business.panhuo.shelf.bean.ShelfDetailEntity;
import com.ejianc.business.panhuo.shelf.bean.ShelfEntity;
import com.ejianc.business.panhuo.shelf.mapper.ShelfMapper;
import com.ejianc.business.panhuo.shelf.service.IShelfService;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.orgcenter.vo.OrgVO;
import com.ejianc.foundation.panhuo.shelf.vo.ShelfDetailVO;
import com.ejianc.foundation.panhuo.shelf.vo.ShelfVO;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.util.ComputeUtil;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.joining;

/**
 * 物资上架申请
 *
 * @author generator
 *
 */
@Service("shelfService")
public class ShelfServiceImpl extends BaseServiceImpl<ShelfMapper, ShelfEntity> implements IShelfService {
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String BILL_CODE = "ZJKJ_MATERIAL_SHELF";

    @Autowired
    private IBillCodeApi billCodeApi;

    @Autowired
    private IFlowmeterService flowmeterService;

    @Autowired
    private OrgUtil orgUtil;

    @Autowired
    private IAttachmentApi attachmentApi;

    @Override
    public ShelfVO saveOrUpdate(ShelfVO saveOrUpdateVO) {
        ShelfEntity data = saveOrUpdateVO.getId() == null ? null : selectById(saveOrUpdateVO.getId());
        // 校验上架数量
        this.validateStoreNum(saveOrUpdateVO);

        ShelfEntity entity = BeanMapper.map(saveOrUpdateVO, ShelfEntity.class);
        if(entity.getId() == null || entity.getId() == 0){
            BillCodeParam billCodeParam = BillCodeParam.build(BILL_CODE,InvocationInfoProxy.getTenantid(),saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if(billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            }else{
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        // 分子公司
        Map<Long, OrgVO> corpMap = orgUtil.findListByIds(new ArrayList<>(Arrays.asList(entity.getOrgId())));
        OrgVO corp = corpMap.containsKey(entity.getOrgId()) ? corpMap.get(entity.getOrgId()) : new OrgVO();
        entity.setCorpId(corp.getId());
        entity.setCorpCode(corp.getCode());
        entity.setCorpName(corp.getName());
        if(entity.getCreateUserId() == null){
            entity.setCreateUserId(InvocationInfoProxy.getUserid());
        }
        super.saveOrUpdate(entity, false);
        // 处理图片
        List<ShelfDetailEntity> detailList = entity.getDetailList();
        if(CollectionUtils.isNotEmpty(detailList)){
            List<String> fileIds = new ArrayList<>();
            for(ShelfDetailEntity detail : detailList){
                if(StringUtils.isNotEmpty(detail.getFileIds())){
                    List<String> strings = Arrays.asList(detail.getFileIds().split(","));
                    fileIds.addAll(strings);
                    attachmentApi.updateAttachRef(detail.getId(), strings.stream().map(s -> Long.parseLong(s.trim())).collect(Collectors.toList()));
                }
                if (null == detail.getStoreId()) {
                    detail.setOriginalTaxPrice(detail.getOriginalPrice());
                    detail.setOriginalMny(ComputeUtil.safeMultiply(detail.getNum(), detail.getOriginalPrice()));
                    detail.setOriginalTaxMny(ComputeUtil.safeMultiply(detail.getNum(), detail.getOriginalTaxPrice()));
                }
            }
            if(data != null) {
                List<ShelfDetailEntity> dataList = data.getDetailList();
                List<String> delIds = new ArrayList<>();
                for(ShelfDetailEntity detail : dataList){
                    if(StringUtils.isNotEmpty(detail.getFileIds())){
                        delIds.addAll(Arrays.asList(detail.getFileIds().split(",")));
                    }
                }
                delIds.removeAll(fileIds);
                if(CollectionUtils.isNotEmpty(delIds)){                     // 用于删除文件
                    attachmentApi.delete(delIds.stream().collect(joining(StringPool.COMMA)));
                }
            }
        }
        super.saveOrUpdate(entity, false);
        return this.queryDetail(entity.getId());
    }

    @Override
    public ShelfVO queryDetail(Long id) {
        ShelfEntity entity = super.selectById(id);
        ShelfVO vo = BeanMapper.map(entity, ShelfVO.class);
        if(BillStateEnum.COMMITED_STATE.equals(entity.getBillState()) || BillStateEnum.PASSED_STATE.equals(entity.getBillState()) ){
            return vo;
        }
        // 获取库存数量
        Map<String, BigDecimal> storeMap = getStoreMap(vo);
        List<ShelfDetailVO> detailList = vo.getDetailList();
        for(ShelfDetailVO detail : detailList){
            String key = detail.getStoreId() + "@" + detail.getMaterialId();
            if(storeMap.containsKey(key)){
                detail.setStoreNum(storeMap.get(key));
            }
//            // 库存保存即占用，剩余可用量不包含占用量，导致剩余可用量无法包含自身，校验不通过
//            detail.setStoreNum(ComputeUtil.safeAdd(detail.getStoreNum(), detail.getNum()));
        }
        return vo;
    }

    @Override
    public Boolean validateStoreNum(ShelfVO vo) {
        Map<String, BigDecimal> storeMap = getStoreMap(vo);
        for (ShelfDetailVO detailVO : vo.getDetailList()) {
            if (null != detailVO.getStoreId()) {
                String key = detailVO.getStoreId() + "@" + detailVO.getMaterialId();
                BigDecimal subtract = storeMap.get(key);
                if (ComputeUtil.isGreaterThan(detailVO.getNum(), subtract)) {
                    throw new BusinessException("物资编号:" + detailVO.getMaterialCode() + ",物资名称:" + detailVO.getMaterialName() + "库存不足，不能上架!");
                }
            }
        }
        return true;
    }

    private Map<String, BigDecimal> getStoreMap(ShelfVO vo) {
        Map<String, BigDecimal> result = new HashMap<>();
        List<ShelfDetailVO> list = vo.getDetailList();
        QueryWrapper<FlowmeterEntity> wrapper = new QueryWrapper<>();
        wrapper.and(q -> {
            for (ShelfDetailVO f : list) {
                q.or(iq -> iq.eq("store_id", f.getStoreId()).eq("material_id", f.getMaterialId()));
            }
            return q;
        });
        List<FlowmeterEntity> flowmeterEntities = flowmeterService.list(wrapper);
        Map<String, BigDecimal> numInMap = new HashMap<>();
        Map<String, BigDecimal> numOutMap = new HashMap<>();
        for (FlowmeterEntity flow : flowmeterEntities) {
            String key = flow.getStoreId() + "@" + flow.getMaterialId();
            if (flow.getOperationType().equals("入库")) {
                BigDecimal inNum = numInMap.containsKey(key) ? numInMap.get(key) : BigDecimal.ZERO;
                inNum = ComputeUtil.safeAdd(inNum, flow.getQuantity());
                numInMap.put(key, inNum);
            } else if (flow.getOperationType().equals("出库")) {
                BigDecimal outNum = numOutMap.containsKey(key) ? numOutMap.get(key) : BigDecimal.ZERO;
                outNum = ComputeUtil.safeAdd(outNum, flow.getQuantity());
                numOutMap.put(key, outNum);
            }
        }
        for (ShelfDetailVO detailVO : list) {
            String key = detailVO.getStoreId() + "@" + detailVO.getMaterialId();
            BigDecimal inNum = numInMap.containsKey(key) ? numInMap.get(key) : BigDecimal.ZERO;
            BigDecimal outNum = numOutMap.containsKey(key) ? numOutMap.get(key) : BigDecimal.ZERO;
            BigDecimal subtract = ComputeUtil.safeSub(inNum, outNum);
            result.put(key, subtract);
        }
        return result;
    }
}
