package com.ejianc.poc.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.ejianc.framework.auth.session.SessionManager;
import com.ejianc.framework.openidclient.openapi.helper.OpenAPIInvokeParameter;
import com.ejianc.framework.openidclient.openapi.service.IZtjOpenApiService;
import com.ejianc.poc.controller.OrgController;
import com.ejianc.poc.vo.BaseFieldVO;
import com.ejianc.poc.vo.OrgVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.net.URLEncoder;
import java.util.*;

/**
 * @Author mrsir_wxp
 * @Date 2021/4/14 中铁建API接口集合服务汇总
 * @Description ZtjApisService
 */

@Service
public class ZtjApisService {
    @Autowired
    private IZtjOpenApiService ztjOpenApiService;
    @Autowired
    private SessionManager sessionManager;

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取所有二级单位集合
     * @Description getAllSecondOrgList
     * @Return [].id:二级集团单位providerId  [].root:组织机构对象
     * [ {
     *   "id" : "hr46",
     *   "root" : {
     *     "type" : 1,
     *     "id" : 1,
     *     "name" : "中铁十一局",
     *     "code" : "0000100011",
     *     "order" : 1,
     *     "show" : true,
     *     "fullname" : "中铁十一局集团有限公司",
     *     "virtual" : false
     *   }
     * }, {
     *   "id" : "hrdemo1",
     *   "root" : {
     *     "type" : 1,
     *     "id" : 78643,
     *     "name" : "中铁十八局",
     *     "code" : "0000100018",
     *     "order" : "001",
     *     "show" : true,
     *     "fullname" : "中铁十八局",
     *     "virtual" : false
     *   }
     * } ]
     */
    public JSONArray getAllSecondOrgList(){
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "orglist";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取获取二级集团单位
     * @Description getSecondOrgInfo
     * @Return
     * {
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * }
     */
    public JSONObject getSecondOrgInfo(String providerId){
        if(StringUtils.isEmpty(providerId)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/root";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取组织机构PATH
     * @Description getSecondOrgInfo
     * @Return [] 组织机构集合，第0个为请求组织机构，最后一个为最顶级组织机构；
     * [ {
     *   "type" : 1,
     *   "id" : 100003,
     *   "name" : "机关",
     *   "code" : "000010001102",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团机关",
     *   "virtual" : true
     * }, {
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * } ]
     */
    public JSONArray getOrgPath(String providerId,String orgid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(orgid)){
            return new JSONArray();
        }
        if(OrgVO.rootOrgCode.equals(providerId)){
            JSONObject p = new JSONObject();
            p.put("code",OrgVO.rootOrgCode);
            p.put("id",OrgVO.rootOrgId);
            p.put("providerId",OrgVO.rootOrgCode);
            p.put("fullname","中国铁建");
            p.put("name","中国铁建");
            p.put("virtual",false);
            p.put("show",true);
            p.put("order",1);
            p.put("type",1);
            JSONArray array = new JSONArray();
            array.add(p);
            return array;
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/path/"+orgid;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取指定单位
     * @Description getSureOrg
     * @Return
     * {
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * }
     */
    public JSONObject getSureOrg(String providerId,String orgid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(orgid)){
            return new JSONObject();
        }
        if(OrgVO.rootOrgCode.equals(providerId)){
            JSONObject p = new JSONObject();
            p.put("code",OrgVO.rootOrgCode);
            p.put("id",OrgVO.rootOrgId);
            p.put("providerId",OrgVO.rootOrgCode);
            p.put("fullname","中国铁建");
            p.put("name","中国铁建");
            p.put("virtual",false);
            p.put("show",true);
            p.put("order",1);
            p.put("type",1);
            return p;
        }

        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/company/"+orgid;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            JSONObject p = JSONObject.parseObject(userDetailStr);
            p.put("providerId",providerId);
            return p;
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取直接上级单位
     * @Description getOrgParent
     * @Return
     *{
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * }
     */
    public JSONObject getOrgParent(String providerId,String orgid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(orgid)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/company/"+orgid+"/parent";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        JSONObject p = null;
        if(StringUtils.isNotEmpty(userDetailStr)){
            p = JSONObject.parseObject(userDetailStr);
            p.put("providerId",providerId);
        }else {
            p = new JSONObject();
            p.put("id",OrgVO.rootOrgId);
            p.put("code",OrgVO.rootOrgCode);
            p.put("providerId",OrgVO.rootOrgCode);
            p.put("name","中国铁建");
            p.put("fullname","中国铁建");
            p.put("virtual",false);
            p.put("show",true);
            p.put("order",1);
            p.put("type",1);
        }
        return p;
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取单位直接下级组织机构集合
     * @Description getOrgDirectChildren
     * @Return 返回单位直接下级组织机构集合
     * [
     * {
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * },
     * {
     *   "type" : 1,
     *   "id" : 2,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * }]
     */
    public JSONArray getOrgDirectChildren(String providerId,String orgid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(orgid)){
            return new JSONArray();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/company/"+orgid+"/children";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取指定部门
     * @Description getSureDepartment
     * @Return
     *{
     *   "type" : 1,
     *   "id" : 1,
     *   "name" : "中铁十一局",
     *   "code" : "0000100011",
     *   "order" : 1,
     *   "show" : true,
     *   "fullname" : "中铁十一局集团有限公司",
     *   "virtual" : false
     * }
     */
    public JSONObject getSureDepartment(String providerId,String deptid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(deptid)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/department/"+deptid ;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取部门上级
     * @Description getDepartmentParent
     * @Return
     *{
     *   "type" : 2,
     *   "id" : 108350,
     *   "name" : "职能部门",
     *   "code" : "00001000110001377002",
     *   "order" : 1,
     *   "virtual" : true
     * }
     */
    public JSONObject getDepartmentParent(String providerId,String deptid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(deptid)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/department/"+deptid +"/parent";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取部门下级组织机构
     * @Description getDepartmentDirectChildren
     * @Return 返回部门下级组织机构，可以是部门、岗位；
     * [ {
     *   "type" : 3,
     *   "id" : 138104,
     *   "name" : "业务员",
     *   "code" : "000010001100013770028800899001",
     *   "order" : 1
     * }, {
     *   "type" : 3,
     *   "id" : 138105,
     *   "name" : "主任",
     *   "code" : "000010001100013770028800899002",
     *   "order" : 3
     * } ]
     */
    public JSONArray getDepartmentDirectChildren(String providerId,String deptid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(deptid)){
            return new JSONArray();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/department/"+deptid+"/children";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取岗位
     * @Description getPosition
     * @Return
     *{
     *   "type" : 3,
     *   "id" : 138104,
     *   "name" : "业务员",
     *   "code" : "000010001100013770028800899001",
     *   "order" : 1
     * }
     */
    public JSONObject getPosition(String providerId,String positionId){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(positionId)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/position/"+positionId  ;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取岗位所在部门
     * @Description getPositionDepartment
     * @Return
     *{
     *   "type" : 2,
     *   "id" : 108368,
     *   "name" : "总工办公室",
     *   "code" : "0000100011000137700288008",
     *   "order" : 13,
     *   "virtual" : false
     * }
     */
    public JSONObject getPositionDepartment(String providerId,String positionId){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(positionId)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/position/"+positionId +"/parent";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取岗位人员列表
     * @Description getPositionUsers
     * @Return 返回岗位人员列表
     * [ {
     *   "id" : 712522,
     *   "mainPosition" : true,
     *   "name" : "王宪",
     *   "gender" : 2,
     *   "catagory" : 101301,
     *   "positionStatus" : 101401,
     *   "order" : 3
     * } ]
     */
    public JSONArray getPositionUsers(String providerId,String positionId){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(positionId)){
            return new JSONArray();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/position/"+positionId+"/users";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取用户
     * @Description getUser
     * @Return
     *{
     *   "id" : 555643,
     *   "name" : "杨培策",
     *   "gender" : 1,
     *   "catagory" : 101302,
     *   "positionStatus" : 101401,
     *   "order" : 5
     * }
     */
    public JSONObject getUser(String providerId,String userid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(userid)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/user/"+userid ;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取用户主岗位
     * @Description getUserMainPosition
     * @Return
     *[ {
     *   "type" : 3,
     *   "id" : 130716,
     *   "name" : "部员",
     *   "code" : "000010001100003000027700399002",
     *   "order" : 1,
     *   "mainPosition" : true
     * } ]
     */
    public JSONObject getUserMainPosition(String providerId,String userid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(userid)){
            return new JSONObject();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/user/"+userid +"/mainposition";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseObject(userDetailStr);
        }
        return new JSONObject();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取用户所有岗位集合
     * @Description getUserAllPosition
     * @Return
     *{
     *   "type" : 3,
     *   "id" : 130716,
     *   "name" : "部员",
     *   "code" : "000010001100003000027700399002",
     *   "order" : 1,
     *   "mainPosition" : true
     * }
     */
    public JSONArray getUserAllPosition(String providerId,String userid){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(userid)){
            return new JSONArray();
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/user/"+userid +"/positions";
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 搜索用户
     * @Description queryUsers
     * username  搜索的用户姓名
     * mainPositionOnly  true 表示仅搜索主岗位人员，false 全部人员。默认全部
     * @Return
     *[ {
     *   "user" : {
     *     "id" : 555643,
     *     "name" : "杨培策",
     *     "gender" : 1,
     *     "catagory" : 101302,
     *     "positionStatus" : 101401,
     *     "order" : 5
     *   },
     *   "positions" : [ {
     *     "id" : 130716,
     *     "mainPosition" : true
     *   } ]
     * } ]
     */
    public JSONArray queryUsers(String providerId,String orgid,String username,Boolean mainPositionOnly){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(orgid)){
            return new JSONArray();
        }
        if(mainPositionOnly==null){
            mainPositionOnly = false;
        }
        if(username==null){
            username = "";
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "org/"+providerId+"/searchuser/"+orgid +"?username="+ URLEncoder.encode(username)+"&mainPositionOnly="+mainPositionOnly;
        String userDetailStr = ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
        if(StringUtils.isNotEmpty(userDetailStr)){
            return JSONObject.parseArray(userDetailStr);
        }
        return new JSONArray();
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取所有组织机构
     * @Description queryAllTreeOrg
     * @Return
     */
    public String queryAllTreeOrg(String providerId,String companyid ){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(companyid)){
            return "";
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "tree/"+providerId+"/"+companyid;
        return ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/14 获取所有组织机构与用户
     * @Description queryAllTreeOrgAndUsers
     * @Return
     */
    public String queryAllTreeOrgAndUsers(String providerId,String companyid ){
        if(StringUtils.isEmpty(providerId) || StringUtils.isEmpty(companyid)){
            return "";
        }
        OpenAPIInvokeParameter apiInvokeParameter = new OpenAPIInvokeParameter();
        apiInvokeParameter.invoke_method = "GET";
        apiInvokeParameter.apiUrl = "tree/"+providerId+"/"+companyid+"/users";
        return ztjOpenApiService.getHrOpenApi(apiInvokeParameter);
    }

    /**
     * @Author mrsir_wxp
     * @Date 2021/4/15 根据应用权限id 查询组织根节点
     * @Description getTreeRootByAppId
     * @Param [appId]
     * @Return java.util.List<com.ejianc.poc.vo.BaseFieldVO>
     */
    public List<BaseFieldVO> getTreeRootByAppId(String appId,Integer expandLevel){
        List<BaseFieldVO> root = new ArrayList<>();
        JSONObject authJson = sessionManager.getUserContext().getUserAuthJson();
        if(authJson==null){
            return root;
        }
        String userAuthJson = authJson.getString(appId);
        if(StringUtils.isEmpty(userAuthJson)){
            return root;
        }
        String orgIds = userAuthJson.split("#")[0];
        String providerIds =  userAuthJson.split("#")[1];
        if(StringUtils.isEmpty(orgIds) || StringUtils.isEmpty(providerIds)){
            return root;
        }
        if(orgIds.contains(String.valueOf(OrgVO.rootOrgId))){
            //返回根组织
            root.add(OrgVO.getRootOrg());
            return root;
        }
        String[] orgArr = orgIds.split(",");
        String[] providerArr = providerIds.split(",");
        //先分组，根据providerId分组  providerId---orgIds
        Map<String,List<String>> p_o = new HashMap<>();
        for (int i = 0; i < providerArr.length; i++) {
            List<String> oList = p_o.computeIfAbsent(providerArr[i], k -> new ArrayList<>());
            if(!oList.contains(orgArr[i])){
                oList.add(orgArr[i]);
            }
        }
        for (String provider:p_o.keySet()){
            List<String> oList = p_o.get(provider);
            JSONArray data = new JSONArray();
            if(oList.size() == 1){//说明此provider  就一个根
                JSONObject sure = getSureOrg(provider,oList.get(0));
                data.add(sure);
            }else {
                List<JSONObject> objectList=new ArrayList<>();
                for (String orgId:oList){
                    objectList.add(getSureOrg(provider,orgId));
                }
                objectList.sort((o1,o2)->{
                    if(o1.getString("code").length()>o2.getString("code").length()){
                        return 1;
                    }else if(o1.getString("code").length()==o2.getString("code").length()){
                        return 0;
                    }
                    return -1;
                });
                List<JSONObject> toRemove = new ArrayList<>();
                for (int i = 0; i < objectList.size(); i++) {
                    JSONObject o1 = objectList.get(i);
                    for (int j = i+1; j < objectList.size(); j++) {
                        JSONObject o2 = objectList.get(j);
                        if(o2.getString("code").startsWith(o1.getString("code"))){
                            /** 说明o2包含o1 ，o2是o1的下级单位，o2不再比较 */
                            toRemove.add(o2);
                        }
                    }
                }
                /** 删除子节点 */
                objectList.removeAll(toRemove);
                data.addAll(objectList);
            }
            for (int i = 0; i < data.size(); i++) {
                JSONObject o = data.getJSONObject(i);
                JSONArray path = getOrgPath(provider,o.getString("id"));
                if(path!=null){
                    for (int j = 0; j < path.size(); j++) {
                        JSONObject p = path.getJSONObject(j);
                        if(p.getInteger("type") == 1 && j > 0){
                            o.put("parentIdOrgId",p.getString("id"));
                            break;
                        }
                    }
                }
            }
            OrgController.transToBaseFieldVO(data,root,provider,expandLevel,null);
        }

        return root;
    }

    public static void main(String[] args) {
        List<JSONObject> objectList = new ArrayList<JSONObject>();
        JSONObject o1 = new JSONObject();
        o1.put("code", "0000100011");
        o1.put("name", "中铁十一局");
        objectList.add(o1);
        o1 = new JSONObject();
        o1.put("code", "0000100012");
        o1.put("name", "中铁十二局");
        objectList.add(o1);
        JSONObject o2 = new JSONObject();
        o2.put("code", "000010001102");
        o2.put("name", "中铁十一局集团机关");
        objectList.add(o2);
        JSONObject o3 = new JSONObject();
        o3.put("code", "0000100018");
        o3.put("name", "中铁十八局");
        objectList.add(o3);
        o3 = new JSONObject();
        o3.put("code", "00001000180111");
        o3.put("name", "中铁十八局00001000180111");
        objectList.add(o3);
        JSONObject o4 = new JSONObject();
        o4.put("code", "000010001802");
        o4.put("name", "中铁十八局机关");
        objectList.add(o4);
        o1 = new JSONObject();
        o1.put("code", "0000100017");
        o1.put("name", "中铁十七局");
        objectList.add(o1);

        objectList.sort((o11, o21) -> {
            if (o11.getString("code").length() > o21.getString("code").length()) {
                return 1;
            }
            if (o11.getString("code").length() == o21.getString("code").length()) {
                return 0;
            }
            return -1;
        });
        List<JSONObject> toRemove = new ArrayList<>();
        for (int i = 0; i < objectList.size(); i++) {
            JSONObject o11 = objectList.get(i);
            for (int j = i+1; j < objectList.size(); j++) {
                JSONObject o22 = objectList.get(j);
                if(o22.getString("code").startsWith(o11.getString("code"))){
                    /** 说明o2包含o1 ，o2是o1的下级单位，o2不再比较 */
                    toRemove.add(o22);
                }
            }
        }
        /** 删除子节点 */
        objectList.removeAll(toRemove);

        System.out.println(objectList);
    }
}
