package com.ejianc.business.supbusiness.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.supbusiness.bean.*;
import com.ejianc.business.supbusiness.constant.InteractiveConstants;
import com.ejianc.business.supbusiness.enums.AutomaticWeighEnums;
import com.ejianc.business.supbusiness.enums.CloseFlagEnum;
import com.ejianc.business.supbusiness.enums.DeliverCheckStateEnum;
import com.ejianc.business.supbusiness.enums.OrderDeliverStateEnum;
import com.ejianc.business.supbusiness.mapper.DeliveryMapper;
import com.ejianc.business.supbusiness.service.IDeliveryDetailService;
import com.ejianc.business.supbusiness.service.IDeliveryService;
import com.ejianc.business.supbusiness.service.IOrderService;
import com.ejianc.business.supbusiness.service.IWeighkeyRecordService;
import com.ejianc.business.supbusiness.utils.CommonUtils;
import com.ejianc.business.supbusiness.vo.*;
import com.ejianc.foundation.support.api.IBillCodeApi;
import com.ejianc.foundation.support.vo.BillCodeParam;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestMethod;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

/**
 * 消耗材送货单
 *
 * @author generator
 */
@Service("deliveryService")
public class DeliveryServiceImpl extends BaseServiceImpl<DeliveryMapper, DeliveryEntity> implements IDeliveryService {
    private Logger logger = LoggerFactory.getLogger(getClass());

    private static final String BILL_CODE = "ZJKJ-SUP-DELIVERY";//此处需要根据实际修改

    @Autowired
    private IBillCodeApi billCodeApi;
    /**
     * 材料订单信息
     */
    @Autowired
    private IOrderService orderService;
    @Autowired
    private IDeliveryDetailService deliveryDetailService;
    @Autowired
    private ISystemDataPushService systemDataPushService;
    @Autowired
    private IWeighkeyRecordService weighkeyRecordService;

    /** 通知送货单 */
//    @Value("${proMaterial.pushMaterialDelivery}")
//    private String pushMaterialDelivery;
//    /** 通知订单送货状态 */
//    @Value("${proMaterial.pushMaterialOrderDeliver}")
//    private String pushMaterialOrderDeliver;

    /**
     * 查询送货单
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public IPage<DeliveryVO> queryDeliverList(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName"); // 合同名称
        fuzzyFields.add("projectName"); // 项目名称
        fuzzyFields.add("orgName"); // 采购方名称
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        if (!param.getParams().containsKey("closeFlag")) {
            param.getParams().put("closeFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));
        }
        // 默认创建时间倒序
        if (param.getOrderMap().isEmpty()) {
            param.getOrderMap().put("createTime", QueryParam.DESC);
        }
        IPage<DeliveryEntity> page = super.queryPage(param, false);
        IPage<DeliveryVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), DeliveryVO.class));
        return pageData;
    }

    /**
     * 保存、修改送货单
     *
     * @param saveOrUpdateVO 送货单数据
     * @return 保存数据
     */
    @Override
    public DeliveryVO saveDeliver(DeliveryVO saveOrUpdateVO) {
        // 校验是否存在未生效的送货单
        Long orderId = saveOrUpdateVO.getOrderId();
        OrderEntity orderEntity = orderService.selectById(orderId);
        if (CloseFlagEnum.CLOSE.getCode().equals(orderEntity.getOrderFlag())) {
            throw new BusinessException("当前订单已关闭！");
        }
        QueryParam checkParam = new QueryParam();
        checkParam.getParams().put("orderId", new Parameter(QueryParam.EQ, orderId));
        checkParam.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(BillStateEnum.UNCOMMITED_STATE.getBillStateCode(),
                        BillStateEnum.APPROVING_HAS_STATE.getBillStateCode(),
                        BillStateEnum.UNAPPROVED.getBillStateCode(),
                        BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode())));
        // 校验排除当前的送货当
        if (saveOrUpdateVO.getId() != null) {
            checkParam.getParams().put("id", new Parameter(QueryParam.NE, saveOrUpdateVO.getId()));
        }
        List<DeliveryEntity> checkList = super.queryList(checkParam);
        if (CollectionUtils.isNotEmpty(checkList)) {
            throw new BusinessException("当前订单存在未生效的送货单！");
        }

        // 校验子表是否全部都是已送货或者未送货
        if (CollectionUtils.isEmpty(saveOrUpdateVO.getDeliveryDetailList())) {
            throw new BusinessException("送货单列表不能为空！");
        }
        List<Integer> automaticWeighList = saveOrUpdateVO.getDeliveryDetailList().stream()
                .filter(e -> !"del".equals(e.getRowState())).map(DeliveryDetailVO::getDetailAutomaticWeigh)
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(automaticWeighList)
                && automaticWeighList.contains(AutomaticWeighEnums.关闭自动称重.getCode())
                && automaticWeighList.contains(AutomaticWeighEnums.开启自动称重.getCode())) {
            throw new BusinessException("送货单明细必须全部是自动称重或全部不是自动称重的材料，不能两者混合发货！");
        }
        // 设置主表为自动称重或非自动称重
        Integer automaticWeigh = CollectionUtils.isNotEmpty(automaticWeighList) ?
                automaticWeighList.get(0) : AutomaticWeighEnums.关闭自动称重.getCode();
        saveOrUpdateVO.setAutomaticWeigh(automaticWeigh);

        DeliveryEntity entity = BeanMapper.map(saveOrUpdateVO, DeliveryEntity.class);
        entity.setPlatformCode(orderEntity.getPlatformCode());
        entity.setPlatformType(orderEntity.getPlatformType());
        entity.setLinkId(orderEntity.getLinkId());
        entity.setLinkName(orderEntity.getLinkName());
        entity.setOrderExpiryDate(orderEntity.getExpiryDate());
        if (entity.getId() == null || entity.getId() == 0) {
            BillCodeParam billCodeParam = BillCodeParam
                    .build(BILL_CODE, InvocationInfoProxy.getTenantid(), saveOrUpdateVO);
            CommonResponse<String> billCode = billCodeApi.generateBillCode(billCodeParam);
            if (billCode.isSuccess()) {
                entity.setBillCode(billCode.getData());//此处需要根据实际修改 删除本行或者上一行
            } else {
                throw new BusinessException("网络异常， 编码生成失败， 请稍后再试");
            }
        }
        // 计算数量
        // 查询订单信息
//        OrderEntity orderEntity = orderService.selectById(entity.getOrderId());
        // 各个材料的接收数量
        Map<Long, BigDecimal> receiveNumsSumMap = orderEntity.getOrderDetailList().stream().collect(
                Collectors.toMap(OrderDetailEntity::getId, OrderDetailEntity::getReceiveNumsSum));
        // 查询已送货订单的数量
        Map<Long, BigDecimal> map = this.sumDeliveredGroupByDetailId(orderId);
        List<String> nameList = new ArrayList<>();
        // 对当前发货单进行计算
        for (DeliveryDetailEntity deliveryDetailEntity : entity.getDeliveryDetailList()) {
            Long orderDetailId = deliveryDetailEntity.getOrderDetailId();
            // 接收数量
            BigDecimal receiveNumsSum = receiveNumsSumMap.get(orderDetailId);
            // 已发货数量
            BigDecimal shippedNumSum = CommonUtils.setBigDecimalDefaultValue(map.get(orderDetailId))
                    .add(deliveryDetailEntity.getDeliveryNum());
            // 未发货数量
            BigDecimal notShippedNumSum = receiveNumsSum.subtract(shippedNumSum);
            if (notShippedNumSum.compareTo(BigDecimal.ZERO) < 0) {
                notShippedNumSum = BigDecimal.ZERO;
            }
            deliveryDetailEntity.setReceiveNumsSum(receiveNumsSum);
            deliveryDetailEntity.setShippedNumSum(shippedNumSum);
            deliveryDetailEntity.setNotShippedNumSum(notShippedNumSum);

            nameList.add(deliveryDetailEntity.getMaterialName());
        }

        entity.setMaterialName(StringUtils.join(nameList, ","));

        super.saveOrUpdate(entity, false);
        return BeanMapper.map(entity, DeliveryVO.class);
    }

    /**
     * 变更送货单状态
     *
     * @param saveOrUpdateVO 需要变更的送货单信息
     * @return 变更结果
     */
    @Override
    public Boolean updateDeliverState(DeliveryVO saveOrUpdateVO) {
        logger.info("进入送货单变更状态接口>>>>>>>>>>>>>>>>>>>>>>>>");
        logger.info("接收到数据:{}", JSONObject.toJSONString(saveOrUpdateVO));
        if (null == saveOrUpdateVO.getSourceId()) {
            throw new BusinessException("送货单信息为空！");
        }
        DeliveryEntity entity = super.selectById(saveOrUpdateVO.getSourceId());
        if (null == entity) {
            throw new BusinessException("送货单信息为空！");
        }
        if (saveOrUpdateVO.getCheckStatus() == null && saveOrUpdateVO.getCloseFlag() == null) {
            throw new BusinessException("状态不能为空！");
        }
        if (saveOrUpdateVO.getCheckStatus() != null) {
            entity.setCheckStatus(saveOrUpdateVO.getCheckStatus());
        }
        if (saveOrUpdateVO.getCloseFlag() != null) {
            entity.setCloseFlag(saveOrUpdateVO.getCloseFlag());
        }
        boolean flag = super.saveOrUpdate(entity, false);
        logger.info("送货单变更状态接口结束<<<<<<<<<<<<<<<<<<<<<<<<<<<");
        return flag;
    }

    /**
     * 根据订单初始化送货单详情
     *
     * @param orderId 订单id
     * @return 送货单详情
     */
    @Override
    public DeliveryVO queryInitDelivery(Long orderId) {
        OrderEntity orderEntity = orderService.selectById(orderId);
        LambdaQueryWrapper<DeliveryEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(DeliveryEntity::getProjectId, orderEntity.getProjectId());
        queryWrapper.in(DeliveryEntity::getBillState, BillStateEnum.PASSED_STATE.getBillStateCode(), BillStateEnum.COMMITED_STATE.getBillStateCode());
        queryWrapper.orderByDesc(DeliveryEntity::getDeliverDate);
        queryWrapper.last("limit 1");
        DeliveryEntity lastDelivery = getOne(queryWrapper, false);
        DeliveryVO vo = BeanMapper.map(orderEntity, DeliveryVO.class);
        vo.setOrderId(orderEntity.getId());
        vo.setOrderBillCode(orderEntity.getBillCode());
        vo.setEmployeeId(null);
        vo.setEmployeeName(null);
        vo.setDeliverName(null);
        vo.setDeliverPhone(null);
        vo.setArriveDate(null);
        vo.setLicensePlate(null);
        vo.setDepartmentId(null);
        vo.setDepartmentName(null);
        vo.setCloseFlag(null);
        vo.setId(null);
        vo.setCheckStatus(DeliverCheckStateEnum.WAIT_CHECK.getCode());
        vo.setMemo(null);
        vo.setCloseFlag(CloseFlagEnum.NORMAL.getCode());
        vo.setArriveDate(new Date());
        vo.setDeliverAddress(null != lastDelivery ? lastDelivery.getDeliverAddress() : null);
        List<DeliveryDetailVO> list = new ArrayList<>();
        for (OrderDetailEntity orderDetailEntity : orderEntity.getOrderDetailList()) {
            // 未送货数量大于0
            if (BigDecimal.ZERO.compareTo(CommonUtils.setBigDecimalDefaultValue(orderDetailEntity.getNotShippedNumSum())) < 0) {
                DeliveryDetailVO detailVO = BeanMapper.map(orderDetailEntity, DeliveryDetailVO.class);
                detailVO.setOrderId(orderDetailEntity.getOrderId());
                detailVO.setOrderDetailId(orderDetailEntity.getId());
                // 订单发货数量默认未发货数量
                detailVO.setDeliveryNum(orderDetailEntity.getNotShippedNumSum());
                detailVO.setCheckNum(null);
                detailVO.setReceiveNumsSum(orderDetailEntity.getReceiveNumsSum());
                detailVO.setShippedNumSum(orderDetailEntity.getDeliverNumsSum());
                detailVO.setId(null);
                list.add(detailVO);
            }
        }
        vo.setDeliveryDetailList(list);
        return vo;
    }

    /**
     * 根据orderId查询当前订单下已送货的材料数量
     *
     * @param orderId 订单id
     * @return 已送货材料数量
     */
    @Override
    public Map<Long, BigDecimal> sumDeliveredGroupByDetailId(Long orderId) {
        // 查询送货单
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("orderId", new Parameter(QueryParam.EQ, orderId));
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
                        BillStateEnum.PASSED_STATE.getBillStateCode())));
        List<DeliveryEntity> deliveryList = super.queryList(queryParam);
        logger.info("已送货的送货单信息：{}", JSONObject.toJSONString(deliveryList));
        Map<Long, BigDecimal> map = new HashMap<>();
        if (CollectionUtils.isNotEmpty(deliveryList)) {
            List<Long> deliveryIdList = new ArrayList<>();
            for (DeliveryEntity entity : deliveryList) {
                deliveryIdList.add(entity.getId());
            }
            // 查询子表信息
            // 送货单已送货的明细
            QueryParam detailParam = new QueryParam();
            detailParam.getParams().put("deliveryId", new Parameter(QueryParam.IN, deliveryIdList));
            List<DeliveryDetailEntity> deliveryDetailList = deliveryDetailService.queryList(detailParam);
            // 对送货单相同明细进行计算
            for (DeliveryDetailEntity detail : deliveryDetailList) {
                Long orderDetailId = detail.getOrderDetailId();
                if (map.containsKey(orderDetailId)) {
                    BigDecimal sum = detail.getDeliveryNum().add(map.get(orderDetailId));
                    map.put(orderDetailId, sum);
                } else {
                    map.put(orderDetailId, detail.getDeliveryNum());
                }
            }
        }
        return map;
    }

    /**
     * 处理终审后事件
     *
     * @param id 送货单id
     */
    @Override
    public DeliveryVO saveCommitAfter(Long id) {
        logger.info("进入推送送货单方法>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>");
        // 状态修改为待验收
        DeliveryEntity deliveryEntity = super.selectById(id);
        // 查询订单是否已关闭
        Long orderId = deliveryEntity.getOrderId();
        OrderEntity orderEntity = orderService.selectById(orderId);
        if (CloseFlagEnum.CLOSE.getCode().equals(orderEntity.getOrderFlag())) {
            throw new BusinessException("订单已关闭，提交失败！");
        }

        deliveryEntity.setCheckStatus(DeliverCheckStateEnum.WAIT_CHECK.getCode());
        // 单据状态手动修改为已提交
        deliveryEntity.setBillState(BillStateEnum.COMMITED_STATE.getBillStateCode());
        super.saveOrUpdate(deliveryEntity, false);
        // 校验订单下的所有送货单数量，是否已经送完
        // 查询订单
        Map<Long, BigDecimal> map = this.sumDeliveredGroupByDetailId(orderId);
        logger.info("已送货材料信息：{}", JSONObject.toJSONString(map));
        Integer orderDeliverState = OrderDeliverStateEnum.PART_DELIVERED.getCode();
        // 全部送货的材料数量
        int fullFlag = 0;
        // 对订单明细计算
        for (OrderDetailEntity entity : orderEntity.getOrderDetailList()) {
            // 如果接收数量是0，则不用送货，此条材料计算为已送货的材料
            if (BigDecimal.ZERO.compareTo(entity.getReceiveNumsSum()) == 0) {
                fullFlag++;
                continue;
            }
            BigDecimal delivery = map.get(entity.getId());
            // 送货单送货数量为空，代表该材料没有送，则肯定是部分送货
            if (delivery == null) continue;
            entity.setDeliverNumsSum(delivery); // 送货数量
            // a > b，代表该材料送货数量大于等于接收数量
            if (delivery.compareTo(entity.getReceiveNumsSum()) > -1) {
//                fullFlag = true;
                fullFlag++;
                entity.setNotShippedNumSum(BigDecimal.ZERO); // 未送货数量
            }
            // a < b，代表有材料未送完
            else {
//                fullFlag = false;
                entity.setNotShippedNumSum(entity.getReceiveNumsSum().subtract(delivery)); // 未送货数量
            }
        }
        // 如果全部送货的材料数量 = 接收的材料数量，则全部送货
        if (fullFlag > 0 && fullFlag == orderEntity.getOrderDetailList().size()) {
            orderDeliverState = OrderDeliverStateEnum.FULL_DELIVERED.getCode();
        }
        // 状态不同切订单状态不为已送货
        if (!orderDeliverState.equals(orderEntity.getDeliverState())
                && !OrderDeliverStateEnum.FULL_DELIVERED.getCode().equals(orderEntity.getDeliverState())) {
            orderEntity.setDeliverState(orderDeliverState);
        }
        orderService.saveOrUpdate(orderEntity, false);
        logger.info("剩余订单信息：{}", JSONObject.toJSONString(orderEntity));
//			String url = "delivery/supplierPushDelivery";
        logger.info("开始通知施工方发货单信息>>>>>>>>>>>>>>>>>>>");
//        String url = pushMaterialDelivery;
        String url = InteractiveConstants.PUSH_MATERIAL_DELIVERY;
        DeliveryDbVO tbVO = BeanMapper.map(deliveryEntity, DeliveryDbVO.class);
        tbVO.setSourceId(deliveryEntity.getId());
        tbVO.setMaterialDeliveryDetailList(null);
        if (CollectionUtils.isNotEmpty(deliveryEntity.getDeliveryDetailList())) {
            List<DeliveryDetailDbVO> tbVOList = new ArrayList<>();
            for (DeliveryDetailEntity deliveryDetailEntity : deliveryEntity
                    .getDeliveryDetailList()) {
                DeliveryDetailDbVO detailTbVO = BeanMapper.map(deliveryDetailEntity,
                        DeliveryDetailDbVO.class);
                detailTbVO.setSourceId(deliveryEntity.getId());
                detailTbVO.setSourceDetailId(deliveryDetailEntity.getId());
                tbVOList.add(detailTbVO);
            }
            tbVO.setMaterialDeliveryDetailList(tbVOList);
        }
        String postData = JSONObject.toJSONString(tbVO);
        String systemId = deliveryEntity.getSystemId();
        logger.info("发送参数===url:{},postData:[{}],systemId:{}", url, postData, systemId);
        CommonResponse<String> commonResponse = systemDataPushService
                .exchangeDataWithThirdSystem(url, RequestMethod.POST, postData, systemId);
        CommonUtils.checkCommonResponse(commonResponse, logger);
        logger.info("通知施工方发货单信息结束<<<<<<<<<<<<<<<<<<<<");

        logger.info("开始通知施工方订单送货状态>>>>>>>>>>>>>>>>>>>");
//        String orderUrl = pushMaterialOrderDeliver;
        String orderUrl = InteractiveConstants.PUSH_MATERIAL_ORDER_DELIVER;
        OrderVO orderVO = new OrderVO();
        orderVO.setId(orderEntity.getId());
        orderVO.setDeliverState(orderEntity.getDeliverState());
        String orderPostData = JSONObject.toJSONString(orderVO);
        CommonResponse<String> orderCommonResponse = systemDataPushService
                .exchangeDataWithThirdSystem(orderUrl, RequestMethod.POST, orderPostData, systemId);
        CommonUtils.checkCommonResponse(orderCommonResponse, logger);
        logger.info("通知施工方订单送货状态结束<<<<<<<<<<<<<<<<<<<");
        logger.info("推送送货单方法结束<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");

        if (0 == deliveryEntity.getAutomaticWeigh()) {
            return updateWeighkey(1, deliveryEntity.getId());
        }

        return BeanMapper.map(deliveryEntity, DeliveryVO.class);
    }

    private Lock lock = new ReentrantLock();// 锁对象

    @Override
    public DeliveryVO updateWeighkey(Integer saveType, Long deliveryId) {
        Date d = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String nowDate = sdf.format(d);// 默认当前系统时间
        Calendar calendar = new GregorianCalendar();
        calendar.setTime(d);
        calendar.add(calendar.HOUR, 48);
        String endDate = sdf.format(calendar.getTime());
        DeliveryEntity delivery = selectById(deliveryId);
        if (delivery.getAutomaticWeigh() == 1) {
            throw new BusinessException("非智能称重项目不允许设置口令");
        }
        //1-重置或生成口令；2-关闭口令
        if (saveType == 2) {
            delivery.setWeighkeyUnusedDate(nowDate);
            delivery.setWeighkey("");
        }
        WeighkeyRecordEntity weighkeyRecord = null;
        if (StringUtils.isNotEmpty(delivery.getWeighkey())) {
            weighkeyRecord = new WeighkeyRecordEntity();
            BeanUtils.copyProperties(delivery, weighkeyRecord);
            weighkeyRecord.setDeliveryId(delivery.getId());
        }
        lock.lock();// 得到锁
        try {
            if (saveType == 1) {
                delivery.setWeighkeyCreateDate(nowDate);
                delivery.setWeighkeyUnusedDate(endDate);
                delivery.setPkWeighkeyCreator(String.valueOf(InvocationInfoProxy.getUserid()));
                Set<String> set = weighkeySet();
                String random = null;
                if (set.size() <= 10000) {
                    do {
                        String ran = String.format("%04d", (int) (Math.random() * 10000));
                        if (!set.contains(ran)) {
                            random = ran;
                            break;
                        }
                    } while (1 == 1);
                } else {
                    do {
                        String ran = getStringRandom(4);
                        if (!set.contains(ran)) {
                            random = ran;
                            break;
                        }
                    } while (1 == 1);
                }
                delivery.setWeighkey(random);
            }
            baseMapper.updateById(delivery);
        } finally {
            lock.unlock();// 释放锁
        }
        if (null != weighkeyRecord) {
            weighkeyRecordService.saveOrUpdate(weighkeyRecord);
        }
        return BeanMapper.map(delivery, DeliveryVO.class);
    }

    @Override
    public Boolean checkWeighkey(String weighkey) {
        List<String> pkList = baseMapper.getListByWeighkey(weighkey);
        if (CollectionUtils.isEmpty(pkList)) {
            throw new BusinessException("口令未匹配，请联系发货方");
        }
        return true;
    }

    @Override
    public DeliveryVO getDetailByWeighkey(String weighkey) {
        checkWeighkey(weighkey);
        LambdaQueryWrapper<DeliveryEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.eq(DeliveryEntity::getWeighkey, weighkey);
        queryWrapper.eq(DeliveryEntity::getCloseFlag, 0);
        queryWrapper.le(DeliveryEntity::getWeighkeyCreateDate, getCurrDate());
        queryWrapper.ge(DeliveryEntity::getWeighkeyUnusedDate, getCurrDate());
        DeliveryEntity entity = getOne(queryWrapper, false);
        if (null != entity) {
            return BeanMapper.map(selectById(entity.getId()), DeliveryVO.class);
        }
        return null;
    }

    private String getCurrDate() {
        Date d = new Date();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdf.format(d);// 默认当前系统时间
    }

    /**
     * 字母数字随机码
     *
     * @param length
     * @return
     */
    public static String getStringRandom(int length) {
        String val = "";
        Random random = new Random();
        //参数length，表示生成几位随机数
        for (int i = 0; i < length; i++) {
            String charOrNum = random.nextInt(2) % 2 == 0 ? "char" : "num";
            //输出字母还是数字
            if ("char".equalsIgnoreCase(charOrNum)) {
                //输出是大写字母还是小写字母
                int temp = random.nextInt(2) % 2 == 0 ? 65 : 97;
                val += (char) (random.nextInt(26) + temp);
            } else if ("num".equalsIgnoreCase(charOrNum)) {
                val += String.valueOf(random.nextInt(10));
            }
        }
        return val;
    }

    private Set<String> weighkeySet() {
        Set<String> usedWeighkeySet = null;
        List<String> list = baseMapper.getUsedWeighkey();
        if (CollectionUtils.isNotEmpty(list)) {
            usedWeighkeySet = new HashSet<>(list);
        } else {
            usedWeighkeySet = new HashSet<>();

        }
        return usedWeighkeySet;
    }

}
