package com.ejianc.business.financeintegration.PMPayApply.controller.api;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ejianc.business.financeintegration.PMPayApply.bean.PMPayApplyEntity;
import com.ejianc.business.financeintegration.PMPayApply.service.IPMPayApplyService;
import com.ejianc.business.financeintegration.PMPayApply.util.RedisUtil;
import com.ejianc.business.financeintegration.PMPayApply.vo.PMPayApplyVO;
import com.ejianc.framework.cache.utils.RedisTool;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.CommonResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

/**
 * PM支付申请
 *
 * @author generator
 */
@RestController
@RequestMapping
public class PMPayApplyApiController {

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    private IPMPayApplyService pmPayService;

    @Autowired
    private JedisPool jedisPool;

    private final String OPERATE = "PM_PAY_APPLY_SAVE";

    private long retryInterval = 60000;

    private final int maxRetryTime = 10;

    // 付款申请生效
    @PostMapping(value = "/api/PMPayApply/takeEffect")
    public CommonResponse<PMPayApplyVO> takeEffect(@RequestBody PMPayApplyVO pmPayApplyVO) {
        Jedis jedis = null;
        String key = null;
        int retryTime = 0;
        boolean lock = false;
        try {
            key = getLockKey(pmPayApplyVO);
            jedis = jedisPool.getResource();
            while(!lock && retryTime <= maxRetryTime) {
                if(retryTime >0 ){
                    logger.info("键-{}第{}次尝试获取redis锁, 第{}毫秒后开始尝试", key, retryTime, retryTime * retryInterval);
                    Thread.sleep(retryTime * retryInterval);
                }
                retryTime++;
                lock = getLock(key, jedis);
            }

            return pmPayService.takeEffect(pmPayApplyVO);
        } catch (Exception e) {
            logger.error("付款结果处理失败！pmPayApplyVO-{}", JSONObject.toJSONString(pmPayApplyVO));
            return CommonResponse.error("付款结果处理失败!");
        } finally {
            //释放单据锁
            RedisUtil.unLock(jedis, false, key, OPERATE);
            logger.info("redisKey-{}进行Redis锁释放", key);
        }
    }

    // 付款申请弃审/关闭
    @PostMapping(value = "/api/PMPayApply/abandonOrClose")
    public CommonResponse<PMPayApplyVO> abandonOrClose(@RequestBody PMPayApplyVO pmPayApplyVO) {
        Jedis jedis = null;
        String key = null;
        int retryTime = 0;
        boolean lock = false;
        try {
            key = getLockKey(pmPayApplyVO);
            jedis = jedisPool.getResource();
            while(!lock && retryTime <= maxRetryTime) {
                if(retryTime >0 ){
                    logger.info("键-{}第{}次尝试获取redis锁, 第{}毫秒后开始尝试", key, retryTime, retryTime * retryInterval);
                    Thread.sleep(retryTime * retryInterval);
                }
                retryTime++;
                lock = getLock(key, jedis);
            }

            return pmPayService.abandonOrClose(pmPayApplyVO);
        } catch (Exception e) {
            logger.error("付款结果处理失败！pmPayApplyVO-{}", JSONObject.toJSONString(pmPayApplyVO));
            return CommonResponse.error("付款结果处理失败!");
        } finally {
            //释放单据锁
            RedisUtil.unLock(jedis, false, key, OPERATE);
            logger.info("redisKey-{}进行Redis锁释放", key);
        }
    }

    // 付款申请核销
    @PostMapping(value = "/api/PMPayApply/writeOff")
    public CommonResponse<PMPayApplyVO> writeOff(@RequestBody PMPayApplyVO pmPayApplyVO) {
        Jedis jedis = null;
        String key = null;
        int retryTime = 0;
        boolean lock = false;
        try {
            key = getLockKey(pmPayApplyVO);
            jedis = jedisPool.getResource();
            while(!lock && retryTime <= maxRetryTime) {
                if(retryTime > 0){
                    logger.info("键-{}第{}次尝试获取redis锁, 第{}毫秒后开始尝试", key, retryTime, retryTime * retryInterval);
                    Thread.sleep(retryTime * retryInterval);
                }
                retryTime++;
                lock = getLock(key, jedis);
            }

            return pmPayService.writeOff(pmPayApplyVO);
        } catch (Exception e) {
            logger.error("付款结果处理失败！pmPayApplyVO-{}", JSONObject.toJSONString(pmPayApplyVO));
            return CommonResponse.error("付款结果处理失败! ");
        } finally {
            //释放单据锁
            RedisUtil.unLock(jedis, false, key, OPERATE);
            logger.info("redisKey-{}进行Redis锁释放", key);
        }
    }

    private String getLockKey(PMPayApplyVO pmPayApplyVO) {
        logger.info("处理付款数据-{}前加锁", JSONObject.toJSONString(pmPayApplyVO));
        // 根据数据维度获取锁，维度：项目ID、供应商ID、合同登记ID、收款单位ID
        String redisKey = "PM::"+pmPayApplyVO.getProjectId() + "::" + pmPayApplyVO.getSupplierId() + "::" + pmPayApplyVO.getContractRegisterId() + "::" + pmPayApplyVO.getPayeeId();
        logger.info("对保存操作添加Redis锁，redisKey-{}", redisKey);
        return redisKey;
    }

    private Boolean getLock(String key, Jedis jedis) {
        logger.info("从jedisPool获取jedis对象，jedis对象-{}", jedis);
        try {
            // 在数据维度层面进行加锁
            return RedisTool.tryLock(jedis, key, OPERATE, 600);
        } catch (Exception e) {
            logger.info("根据键-{}获取reids锁异常", key, e);
            return false;
        }
    }

    /**
     * 根据参数查询财务中间表信息
     *
     * @param projectId 项目ID
     * @param supplyId 供应商Id
     * @param contractId 合同Id
     * @param payeeId 收款方Id
     * @return
     */
    @GetMapping(value = "/api/PMPayApply/getPaymentDetail")
    public CommonResponse<PMPayApplyVO> getPaymentDetail(@RequestParam Long projectId,
                                                       @RequestParam Long supplyId,
                                                       @RequestParam Long contractId,
                                                       @RequestParam Long payeeId) {
        QueryWrapper<PMPayApplyEntity> query = new QueryWrapper<>();
        query.eq("XMID", projectId);
        query.eq("GYSID", supplyId);
        query.eq("HTDJID", contractId);
        query.eq("SKF", payeeId);

        PMPayApplyEntity paymentDetail = pmPayService.getOne(query);
        if(null != paymentDetail) {
            return CommonResponse.success(BeanMapper.map(paymentDetail, PMPayApplyVO.class));
        }

        return CommonResponse.success("未查询到匹配付款信息！", null);
    }

}
