package org.jasig.cas;

import com.codahale.metrics.annotation.Counted;
import com.codahale.metrics.annotation.Metered;
import com.codahale.metrics.annotation.Timed;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.validation.constraints.NotNull;
import org.apache.commons.collections4.Predicate;
import org.jasig.cas.authentication.AcceptAnyAuthenticationPolicyFactory;
import org.jasig.cas.authentication.Authentication;
import org.jasig.cas.authentication.AuthenticationBuilder;
import org.jasig.cas.authentication.AuthenticationException;
import org.jasig.cas.authentication.AuthenticationManager;
import org.jasig.cas.authentication.ContextualAuthenticationPolicy;
import org.jasig.cas.authentication.ContextualAuthenticationPolicyFactory;
import org.jasig.cas.authentication.Credential;
import org.jasig.cas.authentication.DefaultAuthenticationBuilder;
import org.jasig.cas.authentication.MixedPrincipalException;
import org.jasig.cas.authentication.principal.DefaultPrincipalFactory;
import org.jasig.cas.authentication.principal.PersistentIdGenerator;
import org.jasig.cas.authentication.principal.Principal;
import org.jasig.cas.authentication.principal.PrincipalFactory;
import org.jasig.cas.authentication.principal.Service;
import org.jasig.cas.logout.LogoutManager;
import org.jasig.cas.logout.LogoutRequest;
import org.jasig.cas.services.AttributeReleasePolicy;
import org.jasig.cas.services.RegisteredService;
import org.jasig.cas.services.RegisteredServiceAccessStrategy;
import org.jasig.cas.services.RegisteredServiceProxyPolicy;
import org.jasig.cas.services.RegisteredServiceUsernameAttributeProvider;
import org.jasig.cas.services.ServiceContext;
import org.jasig.cas.services.ServicesManager;
import org.jasig.cas.services.UnauthorizedProxyingException;
import org.jasig.cas.services.UnauthorizedServiceException;
import org.jasig.cas.services.UnauthorizedServiceForPrincipalException;
import org.jasig.cas.services.UnauthorizedSsoServiceException;
import org.jasig.cas.ticket.ExpirationPolicy;
import org.jasig.cas.ticket.InvalidTicketException;
import org.jasig.cas.ticket.ServiceTicket;
import org.jasig.cas.ticket.Ticket;
import org.jasig.cas.ticket.TicketCreationException;
import org.jasig.cas.ticket.TicketException;
import org.jasig.cas.ticket.TicketGrantingTicket;
import org.jasig.cas.ticket.TicketGrantingTicketImpl;
import org.jasig.cas.ticket.UnrecognizableServiceForServiceTicketValidationException;
import org.jasig.cas.ticket.UnsatisfiedAuthenticationPolicyException;
import org.jasig.cas.ticket.registry.TicketRegistry;
import org.jasig.cas.util.DefaultUniqueTicketIdGenerator;
import org.jasig.cas.util.UniqueTicketIdGenerator;
import org.jasig.cas.validation.Assertion;
import org.jasig.cas.validation.ImmutableAssertion;
import org.jasig.inspektr.audit.annotation.Audit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

public final class CentralAuthenticationServiceImpl implements CentralAuthenticationService {
	private final Logger logger = LoggerFactory.getLogger(getClass());

	@NotNull
	private final TicketRegistry ticketRegistry;

	@NotNull
	private final AuthenticationManager authenticationManager;

	@NotNull
	private final UniqueTicketIdGenerator ticketGrantingTicketUniqueTicketIdGenerator;

	@NotNull
	private final Map<String, UniqueTicketIdGenerator> uniqueTicketIdGeneratorsForService;

	@NotNull
	private final ServicesManager servicesManager;

	@NotNull
	private final LogoutManager logoutManager;

	@NotNull
	private ExpirationPolicy ticketGrantingTicketExpirationPolicy;

	@NotNull
	private ExpirationPolicy serviceTicketExpirationPolicy;

	@NotNull
	private ContextualAuthenticationPolicyFactory<ServiceContext> serviceContextAuthenticationPolicyFactory = new AcceptAnyAuthenticationPolicyFactory();

	@NotNull
	private PrincipalFactory principalFactory = new DefaultPrincipalFactory();

	@NotNull
	private final UniqueTicketIdGenerator defaultServiceTicketIdGenerator = new DefaultUniqueTicketIdGenerator();

	public CentralAuthenticationServiceImpl(TicketRegistry ticketRegistry, AuthenticationManager authenticationManager,
			UniqueTicketIdGenerator ticketGrantingTicketUniqueTicketIdGenerator,
			Map<String, UniqueTicketIdGenerator> uniqueTicketIdGeneratorsForService,
			ExpirationPolicy ticketGrantingTicketExpirationPolicy, ExpirationPolicy serviceTicketExpirationPolicy,
			ServicesManager servicesManager, LogoutManager logoutManager) {
		this.ticketRegistry = ticketRegistry;
		this.authenticationManager = authenticationManager;
		this.ticketGrantingTicketUniqueTicketIdGenerator = ticketGrantingTicketUniqueTicketIdGenerator;
		this.uniqueTicketIdGeneratorsForService = uniqueTicketIdGeneratorsForService;
		this.ticketGrantingTicketExpirationPolicy = ticketGrantingTicketExpirationPolicy;
		this.serviceTicketExpirationPolicy = serviceTicketExpirationPolicy;
		this.servicesManager = servicesManager;
		this.logoutManager = logoutManager;
	}

	@Audit(action = "TICKET_GRANTING_TICKET_DESTROYED", actionResolverName = "DESTROY_TICKET_GRANTING_TICKET_RESOLVER", resourceResolverName = "DESTROY_TICKET_GRANTING_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "DESTROY_TICKET_GRANTING_TICKET_TIMER")
	@Metered(name = "DESTROY_TICKET_GRANTING_TICKET_METER")
	@Counted(name = "DESTROY_TICKET_GRANTING_TICKET_COUNTER", monotonic = true)
	public List<LogoutRequest> destroyTicketGrantingTicket(@NotNull String ticketGrantingTicketId) {
		try {
			this.logger.debug("Removing ticket [{}] from registry...", ticketGrantingTicketId);
			TicketGrantingTicket ticket = (TicketGrantingTicket) getTicketForDestroy(ticketGrantingTicketId,
					TicketGrantingTicket.class);

			this.logger.debug("Ticket found. Processing logout requests and then deleting the ticket...");
			List<LogoutRequest> logoutRequests = this.logoutManager.performLogout(ticket);

			this.ticketRegistry.deleteTicket(ticketGrantingTicketId);

			return logoutRequests;
		} catch (InvalidTicketException e) {
			this.logger.error(e.getMessage(), e);
			this.logger.debug("TicketGrantingTicket [{}] cannot be found in the ticket registry.",
					ticketGrantingTicketId);
		}
		return Collections.emptyList();
	}

	@Audit(action = "SERVICE_TICKET", actionResolverName = "GRANT_SERVICE_TICKET_RESOLVER", resourceResolverName = "GRANT_SERVICE_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "GRANT_SERVICE_TICKET_TIMER")
	@Metered(name = "GRANT_SERVICE_TICKET_METER")
	@Counted(name = "GRANT_SERVICE_TICKET_COUNTER", monotonic = true)
	public ServiceTicket grantServiceTicket(String ticketGrantingTicketId, Service service, Credential[] credentials) throws AuthenticationException, TicketException {
		TicketGrantingTicket ticketGrantingTicket = (TicketGrantingTicket) getTicket(ticketGrantingTicketId,
				TicketGrantingTicket.class);
		RegisteredService registeredService = this.servicesManager.findServiceBy(service);

		verifyRegisteredServiceProperties(registeredService, service);
		Set<Credential> sanitizedCredentials = sanitizeCredentials(credentials);

		Authentication currentAuthentication = null;
		if (sanitizedCredentials.size() > 0) {
			currentAuthentication = this.authenticationManager
					.authenticate((Credential[]) sanitizedCredentials.toArray(new Credential[0]));

			Authentication original = ticketGrantingTicket.getAuthentication();
			if (!currentAuthentication.getPrincipal().equals(original.getPrincipal())) {
				throw new MixedPrincipalException(currentAuthentication, currentAuthentication.getPrincipal(),
						original.getPrincipal());
			}

			ticketGrantingTicket.getSupplementalAuthentications().add(currentAuthentication);
		}

		if ((currentAuthentication == null)
				&& (!registeredService.getAccessStrategy().isServiceAccessAllowedForSso())) {
			this.logger.warn("ServiceManagement: Service [{}] is not allowed to use SSO.", service.getId());
			throw new UnauthorizedSsoServiceException();
		}

		Service proxiedBy = ticketGrantingTicket.getProxiedBy();
		if (proxiedBy != null) {
			this.logger.debug("TGT is proxied by [{}]. Locating proxy service in registry...", proxiedBy.getId());
			RegisteredService proxyingService = this.servicesManager.findServiceBy(proxiedBy);

			if (proxyingService != null) {
				this.logger.debug("Located proxying service [{}] in the service registry", proxyingService);
				if (!proxyingService.getProxyPolicy().isAllowedToProxy()) {
					this.logger.warn(
							"Found proxying service {}, but it is not authorized to fulfill the proxy attempt made by {}",
							Long.valueOf(proxyingService.getId()), service.getId());

					throw new UnauthorizedProxyingException(
							"Proxying is not allowed for registered service " + registeredService.getId());
				}
			} else {
				this.logger.warn(
						"No proxying service found. Proxy attempt by service [{}] (registered service [{}]) is not allowed.",
						service.getId(), Long.valueOf(registeredService.getId()));

				throw new UnauthorizedProxyingException(
						"Proxying is not allowed for registered service " + registeredService.getId());
			}
		} else {
			this.logger.trace("TGT is not proxied by another service");
		}

		getAuthenticationSatisfiedByPolicy(ticketGrantingTicket, new ServiceContext(service, registeredService));

		List<Authentication> authentications = ticketGrantingTicket.getChainedAuthentications();
		Principal principal = ((Authentication) authentications.get(authentications.size() - 1)).getPrincipal();

		Map principalAttrs = registeredService.getAttributeReleasePolicy().getAttributes(principal);
		if (!registeredService.getAccessStrategy().doPrincipalAttributesAllowServiceAccess(principalAttrs)) {
			this.logger.warn("ServiceManagement: Cannot grant service ticket because Service [{}] is not authorized for use by [{}].",service.getId(), principal);

			throw new UnauthorizedServiceForPrincipalException();
		}

		String uniqueTicketIdGenKey = service.getClass().getName();
		this.logger.debug("Looking up service ticket id generator for [{}]", uniqueTicketIdGenKey);
		UniqueTicketIdGenerator serviceTicketUniqueTicketIdGenerator = (UniqueTicketIdGenerator) this.uniqueTicketIdGeneratorsForService
				.get(uniqueTicketIdGenKey);

		if (serviceTicketUniqueTicketIdGenerator == null) {
			serviceTicketUniqueTicketIdGenerator = this.defaultServiceTicketIdGenerator;
			this.logger.debug("Service ticket id generator not found for [{}]. Using the default generator...",
					uniqueTicketIdGenKey);
		}

		String ticketPrefix = authentications.size() == 1 ? "ST" : "PT";
		String ticketId = serviceTicketUniqueTicketIdGenerator.getNewTicketId(ticketPrefix);
		ServiceTicket serviceTicket = ticketGrantingTicket.grantServiceTicket(ticketId, service,
				this.serviceTicketExpirationPolicy, currentAuthentication != null);

		this.ticketRegistry.addTicket(serviceTicket);

		this.logger.info("Granted ticket [{}] for service [{}] for user [{}]",
				new Object[] { serviceTicket.getId(), service.getId(), principal.getId() });

		return serviceTicket;
	}

	private static Set<Credential> sanitizeCredentials(Credential[] credentials) {
		if ((credentials != null) && (credentials.length > 0)) {
			Set<Credential> set = new HashSet<Credential>(Arrays.asList(credentials));
			Iterator<Credential> it = set.iterator();
			while (it.hasNext()) {
				if (it.next() == null) {
					it.remove();
				}
			}
			return set;
		}
		return Collections.emptySet();
	}

	@Audit(action = "SERVICE_TICKET", actionResolverName = "GRANT_SERVICE_TICKET_RESOLVER", resourceResolverName = "GRANT_SERVICE_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "GRANT_SERVICE_TICKET_TIMER")
	@Metered(name = "GRANT_SERVICE_TICKET_METER")
	@Counted(name = "GRANT_SERVICE_TICKET_COUNTER", monotonic = true)
	public ServiceTicket grantServiceTicket(String ticketGrantingTicketId, Service service) throws TicketException {
		try {
			return grantServiceTicket(ticketGrantingTicketId, service, (Credential[]) null);
		} catch (AuthenticationException e) {
			throw new IllegalStateException("Unexpected authentication exception", e);
		}
	}

	@Audit(action = "PROXY_GRANTING_TICKET", actionResolverName = "GRANT_PROXY_GRANTING_TICKET_RESOLVER", resourceResolverName = "GRANT_PROXY_GRANTING_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "GRANT_PROXY_GRANTING_TICKET_TIMER")
	@Metered(name = "GRANT_PROXY_GRANTING_TICKET_METER")
	@Counted(name = "GRANT_PROXY_GRANTING_TICKET_COUNTER", monotonic = true)
	public TicketGrantingTicket delegateTicketGrantingTicket(String serviceTicketId, Credential[] credentials) throws AuthenticationException, TicketException {
		ServiceTicket serviceTicket = (ServiceTicket) this.ticketRegistry.getTicket(serviceTicketId,
				ServiceTicket.class);

		if ((serviceTicket == null) || (serviceTicket.isExpired())) {
			this.logger.debug("ServiceTicket [{}] has expired or cannot be found in the ticket registry",
					serviceTicketId);
			throw new InvalidTicketException(serviceTicketId);
		}

		RegisteredService registeredService = this.servicesManager.findServiceBy(serviceTicket.getService());

		verifyRegisteredServiceProperties(registeredService, serviceTicket.getService());

		if (!registeredService.getProxyPolicy().isAllowedToProxy()) {
			this.logger.warn("ServiceManagement: Service [{}] attempted to proxy, but is not allowed.",
					serviceTicket.getService().getId());
			throw new UnauthorizedProxyingException();
		}

		Authentication authentication = this.authenticationManager.authenticate(credentials);

		String pgtId = this.ticketGrantingTicketUniqueTicketIdGenerator.getNewTicketId("PGT");

		TicketGrantingTicket proxyGrantingTicket = serviceTicket.grantTicketGrantingTicket(pgtId, authentication,
				this.ticketGrantingTicketExpirationPolicy);

		this.logger.debug("Generated proxy granting ticket [{}] based off of [{}]", proxyGrantingTicket,
				serviceTicketId);
		this.ticketRegistry.addTicket(proxyGrantingTicket);

		return proxyGrantingTicket;
	}

	@Audit(action = "SERVICE_TICKET_VALIDATE", actionResolverName = "VALIDATE_SERVICE_TICKET_RESOLVER", resourceResolverName = "VALIDATE_SERVICE_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "VALIDATE_SERVICE_TICKET_TIMER")
	@Metered(name = "VALIDATE_SERVICE_TICKET_METER")
	@Counted(name = "VALIDATE_SERVICE_TICKET_COUNTER", monotonic = true)
	public Assertion validateServiceTicket(String serviceTicketId, Service service) throws TicketException {
		RegisteredService registeredService = this.servicesManager.findServiceBy(service);
		verifyRegisteredServiceProperties(registeredService, service);

		ServiceTicket serviceTicket = (ServiceTicket) this.ticketRegistry.getTicket(serviceTicketId,
				ServiceTicket.class);

		if (serviceTicket == null) {
			this.logger.info("Service ticket [{}] does not exist.", serviceTicketId);
			throw new InvalidTicketException(serviceTicketId);
		}
		try {
			synchronized (serviceTicket) {
				if (serviceTicket.isExpired()) {
					this.logger.info("ServiceTicket [{}] has expired.", serviceTicketId);
					throw new InvalidTicketException(serviceTicketId);
				}

				if (!serviceTicket.isValidFor(service)) {
					this.logger.error("Service ticket [{}] with service [{}] does not match supplied service [{}]",
							new Object[] { serviceTicketId, serviceTicket.getService().getId(), service });

					throw new UnrecognizableServiceForServiceTicketValidationException(serviceTicket.getService());
				}
			}

			TicketGrantingTicket root = serviceTicket.getGrantingTicket().getRoot();
			Authentication authentication = getAuthenticationSatisfiedByPolicy(root,
					new ServiceContext(serviceTicket.getService(), registeredService));

			Principal principal = authentication.getPrincipal();

			AttributeReleasePolicy attributePolicy = registeredService.getAttributeReleasePolicy();
			this.logger.debug("Attribute policy [{}] is associated with service [{}]", attributePolicy,
					registeredService);

			Map attributesToRelease = attributePolicy != null ? attributePolicy.getAttributes(principal)
					: Collections.EMPTY_MAP;

			String principalId = registeredService.getUsernameAttributeProvider().resolveUsername(principal, service);
			Principal modifiedPrincipal = this.principalFactory.createPrincipal(principalId, attributesToRelease);
			AuthenticationBuilder builder = DefaultAuthenticationBuilder.newInstance(authentication);
			builder.setPrincipal(modifiedPrincipal);

			return new ImmutableAssertion(builder.build(),
					serviceTicket.getGrantingTicket().getChainedAuthentications(), serviceTicket.getService(),
					serviceTicket.isFromNewLogin());
		} finally {
			if (serviceTicket.isExpired())
				this.ticketRegistry.deleteTicket(serviceTicketId);
		}
	}

	@Audit(action = "TICKET_GRANTING_TICKET", actionResolverName = "CREATE_TICKET_GRANTING_TICKET_RESOLVER", resourceResolverName = "CREATE_TICKET_GRANTING_TICKET_RESOURCE_RESOLVER")
	@Timed(name = "CREATE_TICKET_GRANTING_TICKET_TIMER")
	@Metered(name = "CREATE_TICKET_GRANTING_TICKET_METER")
	@Counted(name = "CREATE_TICKET_GRANTING_TICKET_COUNTER", monotonic = true)
	public TicketGrantingTicket createTicketGrantingTicket(Credential[] credentials) throws AuthenticationException, TicketException {
		Set<Credential> sanitizedCredentials = sanitizeCredentials(credentials);
		if (!sanitizedCredentials.isEmpty()) {
			Authentication authentication = this.authenticationManager.authenticate(credentials);

			TicketGrantingTicket ticketGrantingTicket = new TicketGrantingTicketImpl(
					this.ticketGrantingTicketUniqueTicketIdGenerator.getNewTicketId("TGT"), authentication,
					this.ticketGrantingTicketExpirationPolicy);

			this.ticketRegistry.addTicket(ticketGrantingTicket);
			return ticketGrantingTicket;
		}
		String msg = "No credentials were specified in the request for creating a new ticket-granting ticket";
		this.logger.warn("No credentials were specified in the request for creating a new ticket-granting ticket");
		throw new TicketCreationException(new IllegalArgumentException(msg));
	}

	@Timed(name = "GET_TICKET_TIMER")
	@Metered(name = "GET_TICKET_METER")
	@Counted(name = "GET_TICKET_COUNTER", monotonic = true)
	public <T extends Ticket> T getTicketForDestroy(String ticketId, Class<? extends Ticket> clazz)
			throws InvalidTicketException {
		Assert.notNull(ticketId, "ticketId cannot be null");
		Ticket ticket = this.ticketRegistry.getTicket(ticketId, clazz);

		if (ticket == null) {
			this.logger.debug("Ticket [{}] by type [{}] cannot be found in the ticket registry.", ticketId,
					clazz.getSimpleName());
			throw new InvalidTicketException(ticketId);
		}

		return (T) ticket;
	}

	@Timed(name = "GET_TICKET_TIMER")
	@Metered(name = "GET_TICKET_METER")
	@Counted(name = "GET_TICKET_COUNTER", monotonic = true)
	public <T extends Ticket> T getTicket(String ticketId, Class<? extends Ticket> clazz)
			throws InvalidTicketException {
		Assert.notNull(ticketId, "ticketId cannot be null");
		Ticket ticket = this.ticketRegistry.getTicket(ticketId, clazz);

		if (ticket == null) {
			this.logger.debug("Ticket [{}] by type [{}] cannot be found in the ticket registry.", ticketId,
					clazz.getSimpleName());
			throw new InvalidTicketException(ticketId);
		}

		if ((ticket instanceof TicketGrantingTicket)) {
			synchronized (ticket) {
				if (ticket.isExpired()) {
					this.ticketRegistry.deleteTicket(ticketId);
					this.logger.debug("Ticket [{}] has expired and is now deleted from the ticket registry.", ticketId);
					throw new InvalidTicketException(ticketId);
				}
			}
		}
		return (T) ticket;
	}

	@Timed(name = "GET_TICKETS_TIMER")
	@Metered(name = "GET_TICKETS_METER")
	@Counted(name = "GET_TICKETS_COUNTER", monotonic = true)
	public Collection<Ticket> getTickets(Predicate predicate) {
		Collection c = new HashSet(this.ticketRegistry.getTickets());
		Iterator it = c.iterator();
		while (it.hasNext()) {
			if (!predicate.evaluate(it.next())) {
				it.remove();
			}
		}
		return c;
	}

	public void setServiceContextAuthenticationPolicyFactory(
			ContextualAuthenticationPolicyFactory<ServiceContext> policy) {
		this.serviceContextAuthenticationPolicyFactory = policy;
	}

	public void setTicketGrantingTicketExpirationPolicy(ExpirationPolicy ticketGrantingTicketExpirationPolicy) {
		this.ticketGrantingTicketExpirationPolicy = ticketGrantingTicketExpirationPolicy;
	}

	public void setServiceTicketExpirationPolicy(ExpirationPolicy serviceTicketExpirationPolicy) {
		this.serviceTicketExpirationPolicy = serviceTicketExpirationPolicy;
	}

	@Deprecated
	public void setPersistentIdGenerator(PersistentIdGenerator persistentIdGenerator) {
		this.logger.warn(
				"setPersistentIdGenerator() is deprecated and no longer available. Consider configuring the an attribute provider for service definitions.");
	}

	public void setPrincipalFactory(PrincipalFactory principalFactory) {
		this.principalFactory = principalFactory;
	}

	private Authentication getAuthenticationSatisfiedByPolicy(TicketGrantingTicket ticket, ServiceContext context)
			throws TicketException {
		ContextualAuthenticationPolicy policy = this.serviceContextAuthenticationPolicyFactory.createPolicy(context);

		if (policy.isSatisfiedBy(ticket.getAuthentication())) {
			return ticket.getAuthentication();
		}
		for (Authentication auth : ticket.getSupplementalAuthentications()) {
			if (policy.isSatisfiedBy(auth)) {
				return auth;
			}
		}
		throw new UnsatisfiedAuthenticationPolicyException(policy);
	}

	private void verifyRegisteredServiceProperties(RegisteredService registeredService, Service service) {
		if (registeredService == null) {
			String msg = String.format(
					"ServiceManagement: Unauthorized Service Access. Service [%s] is not found in service registry.",
					new Object[] { service.getId() });

			this.logger.warn(msg);
			throw new UnauthorizedServiceException("screen.service.error.message", msg);
		}
		if (!registeredService.getAccessStrategy().isServiceAccessAllowed()) {
			String msg = String.format(
					"ServiceManagement: Unauthorized Service Access. Service [%s] is not enabled in service registry.",
					new Object[] { service.getId() });

			this.logger.warn(msg);
			throw new UnauthorizedServiceException("screen.service.error.message", msg);
		}
	}
}