package org.jasig.cas.authentication.principal;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.validation.constraints.NotNull;

import org.jasig.cas.authentication.Credential;
import org.jasig.cas.authentication.UsernamePasswordCredential;
import org.jasig.services.persondir.IPersonAttributeDao;
import org.jasig.services.persondir.IPersonAttributes;
import org.jasig.services.persondir.support.StubPersonAttributeDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersonDirectoryPrincipalResolver implements PrincipalResolver {
	protected final Logger logger = LoggerFactory.getLogger(getClass());
	private boolean returnNullIfNoAttributes;

	@NotNull
	private IPersonAttributeDao attributeRepository = new StubPersonAttributeDao(new HashMap<String, List<Object>>());

	@NotNull
	private PrincipalFactory principalFactory = new DefaultPrincipalFactory();
	private String principalAttributeName;

	public boolean supports(Credential credential) {
		return true;
	}

	public final Principal resolve(Credential credential) {
		this.logger.debug("Attempting to resolve a principal...");

		String principalId = extractPrincipalId(credential);

		if (principalId == null) {
			this.logger.debug("Got null for extracted principal ID; returning null.");
			return null;
		}

		this.logger.debug("Creating SimplePrincipal for [{}]", principalId);

		IPersonAttributes personAttributes = this.attributeRepository.getPerson(principalId);
		Map<String, List<Object>> attributes;
		if (personAttributes == null)
			attributes = null;
		else {
			attributes = personAttributes.getAttributes();
		}

		if ((attributes == null) || (attributes.isEmpty())) {
			if (!this.returnNullIfNoAttributes) {
				return this.principalFactory.createPrincipal(principalId);
			}
			return null;
		}

		Map<String, Object> convertedAttributes = new HashMap<String, Object>();
		if ((credential instanceof UsernamePasswordCredential)) {
			UsernamePasswordCredential ucd = (UsernamePasswordCredential) credential;
			convertedAttributes.put("tenantId", ucd.getTenantid());

			Map<String, Object> map = ucd.getAttrparam();
			if (map != null)
				for (String key : map.keySet()) {
					Object value = map.get(key);
					convertedAttributes.put(key, value);
				}
		} else {
			for (Map.Entry<String, List<Object>> entry : attributes.entrySet()) {
				String key = (String) entry.getKey();
				List<Object> values = (List<Object>) entry.getValue();
				if (key.equalsIgnoreCase(this.principalAttributeName)) {
					if (values.isEmpty()) {
						this.logger.debug("{} is empty, using {} for principal", this.principalAttributeName,
								principalId);
					} else {
						principalId = values.get(0).toString();
						this.logger.debug("Found principal attribute value {}; removing {} from attribute map.",
								principalId, this.principalAttributeName);
					}

				} else {
					convertedAttributes.put(key, values.size() == 1 ? values.get(0) : values);
				}
			}
		}
		return this.principalFactory.createPrincipal(principalId, convertedAttributes);
	}

	public final void setAttributeRepository(IPersonAttributeDao attributeRepository) {
		this.attributeRepository = attributeRepository;
	}

	public void setReturnNullIfNoAttributes(boolean returnNullIfNoAttributes) {
		this.returnNullIfNoAttributes = returnNullIfNoAttributes;
	}

	public void setPrincipalAttributeName(String attribute) {
		this.principalAttributeName = attribute;
	}

	public void setPrincipalFactory(PrincipalFactory principalFactory) {
		this.principalFactory = principalFactory;
	}

	protected String extractPrincipalId(Credential credential) {
		return credential.getId();
	}
}