package org.jasig.cas.web.flow;

import java.util.Map;

import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;

import org.apache.commons.lang3.StringUtils;
import org.jasig.cas.CentralAuthenticationService;
import org.jasig.cas.MessageDescriptor;
import org.jasig.cas.authentication.AuthenticationException;
import org.jasig.cas.authentication.Credential;
import org.jasig.cas.authentication.HandlerResult;
import org.jasig.cas.authentication.principal.Service;
import org.jasig.cas.ticket.ServiceTicket;
import org.jasig.cas.ticket.TicketCreationException;
import org.jasig.cas.ticket.TicketException;
import org.jasig.cas.ticket.TicketGrantingTicket;
import org.jasig.cas.ticket.registry.TicketRegistry;
import org.jasig.cas.web.support.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.binding.message.MessageBuilder;
import org.springframework.binding.message.MessageContext;
import org.springframework.web.util.CookieGenerator;
import org.springframework.webflow.core.collection.LocalAttributeMap;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;

@SuppressWarnings("deprecation")
public class AuthenticationViaFormAction {
	public static final String SUCCESS = "success";
	public static final String SUCCESS_WITH_WARNINGS = "successWithWarnings";
	public static final String WARN = "warn";
	public static final String AUTHENTICATION_FAILURE = "authenticationFailure";
	public static final String ERROR = "error";
	public static final String PUBLIC_WORKSTATION_ATTRIBUTE = "publicWorkstation";
	protected final Logger logger = LoggerFactory.getLogger(getClass());

	@NotNull
	private CentralAuthenticationService centralAuthenticationService;

	@NotNull
	private CookieGenerator warnCookieGenerator;

	public final Event submit(RequestContext context, Credential credential, MessageContext messageContext) {
		if (!checkLoginTicketIfExists(context)) {
			return returnInvalidLoginTicketEvent(context, messageContext);
		}

		if (isRequestAskingForServiceTicket(context)) {
			return grantServiceTicket(context, credential);
		}

		return createTicketGrantingTicket(context, credential, messageContext);
	}

	public final Event submitModifyPW(RequestContext context, Credential credential, MessageContext messageContext) {
		if (!checkLoginTicketIfExists(context)) {
			return returnInvalidLoginTicketEvent(context, messageContext);
		}

		if (isRequestAskingForServiceTicket(context)) {
			return grantServiceTicket(context, credential);
		}

		return createTicketGrantingTicket(context, credential, messageContext);
	}

	protected boolean checkLoginTicketIfExists(RequestContext context) {
		String loginTicketFromFlowScope = WebUtils.getLoginTicketFromFlowScope(context);
		String loginTicketFromRequest = WebUtils.getLoginTicketFromRequest(context);

		this.logger.trace("Comparing login ticket in the flow scope [{}] with login ticket in the request [{}]",
				loginTicketFromFlowScope, loginTicketFromRequest);

		return StringUtils.equals(loginTicketFromFlowScope, loginTicketFromRequest);
	}

	protected Event returnInvalidLoginTicketEvent(RequestContext context, MessageContext messageContext) {
		String loginTicketFromRequest = WebUtils.getLoginTicketFromRequest(context);
		this.logger.warn("Invalid login ticket [{}]", loginTicketFromRequest);
		messageContext.addMessage(new MessageBuilder().error().code("error.invalid.loginticket").build());
		return newEvent("error");
	}

	protected boolean isRequestAskingForServiceTicket(RequestContext context) {
		String ticketGrantingTicketId = WebUtils.getTicketGrantingTicketId(context);
		Service service = WebUtils.getService(context);
		return (StringUtils.isNotBlank(context.getRequestParameters().get("renew"))) && (ticketGrantingTicketId != null)
				&& (service != null);
	}

	protected Event grantServiceTicket(RequestContext context, Credential credential) {
		String ticketGrantingTicketId = WebUtils.getTicketGrantingTicketId(context);
		try {
			Service service = WebUtils.getService(context);
			ServiceTicket serviceTicketId = this.centralAuthenticationService.grantServiceTicket(ticketGrantingTicketId,
					service, new Credential[] { credential });

			WebUtils.putServiceTicketInRequestScope(context, serviceTicketId);
			putWarnCookieIfRequestParameterPresent(context);
			return newEvent("warn");
		} catch (AuthenticationException e) {
			return newEvent("authenticationFailure", e);
		} catch (TicketCreationException e) {
			this.logger.warn(
					"Invalid attempt to access service using renew=true with different credential. Ending SSO session.",
					e);

			this.centralAuthenticationService.destroyTicketGrantingTicket(ticketGrantingTicketId);
		} catch (TicketException e) {
			return newEvent("error", e);
		}
		return newEvent("error");
	}

	protected Event createTicketGrantingTicket(RequestContext context, Credential credential,
			MessageContext messageContext) {
		try {
			TicketGrantingTicket tgt = this.centralAuthenticationService
					.createTicketGrantingTicket(new Credential[] { credential });
			WebUtils.putTicketGrantingTicketInScopes(context, tgt);
			putWarnCookieIfRequestParameterPresent(context);
			putPublicWorkstationToFlowIfRequestParameterPresent(context);
			if (addWarningMessagesToMessageContextIfNeeded(tgt, messageContext)) {
				return newEvent("successWithWarnings");
			}
			return newEvent("success");
		} catch (AuthenticationException e) {
			this.logger.debug(e.getMessage(), e);
			return newEvent("authenticationFailure", e);
		} catch (Exception e) {
			this.logger.debug(e.getMessage(), e);
			return newEvent("error", e);
		}
	}

	@SuppressWarnings("rawtypes")
	protected boolean addWarningMessagesToMessageContextIfNeeded(TicketGrantingTicket tgtId, MessageContext messageContext) {
		boolean foundAndAddedWarnings = false;
		for (Map.Entry entry : tgtId.getAuthentication().getSuccesses().entrySet()) {
			for (MessageDescriptor message : ((HandlerResult) entry.getValue()).getWarnings()) {
				addWarningToContext(messageContext, message);
				foundAndAddedWarnings = true;
			}
		}
		return foundAndAddedWarnings;
	}

	private void putWarnCookieIfRequestParameterPresent(RequestContext context) {
		HttpServletResponse response = WebUtils.getHttpServletResponse(context);

		if (StringUtils.isNotBlank(context.getExternalContext().getRequestParameterMap().get("warn")))
			this.warnCookieGenerator.addCookie(response, "true");
		else
			this.warnCookieGenerator.removeCookie(response);
	}

	private void putPublicWorkstationToFlowIfRequestParameterPresent(RequestContext context) {
		if (StringUtils.isNotBlank(context.getExternalContext().getRequestParameterMap().get("publicWorkstation"))) {
			context.getFlowScope().put("publicWorkstation", Boolean.TRUE);
		}
	}

	private Event newEvent(String id) {
		return new Event(this, id);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	private Event newEvent(String id, Exception error) {
		return new Event(this, id, new LocalAttributeMap("error", error));
	}

	public final void setCentralAuthenticationService(CentralAuthenticationService centralAuthenticationService) {
		this.centralAuthenticationService = centralAuthenticationService;
	}

	public final void setWarnCookieGenerator(CookieGenerator warnCookieGenerator) {
		this.warnCookieGenerator = warnCookieGenerator;
	}

	@Deprecated
	public void setTicketRegistry(TicketRegistry ticketRegistry) {
		this.logger.warn("setTicketRegistry() has no effect and will be removed in future CAS versions.");
	}

	private void addWarningToContext(MessageContext context, MessageDescriptor warning) {
		MessageBuilder builder = new MessageBuilder().warning().code(warning.getCode()).defaultText(warning.getDefaultMessage()).args(warning.getParams());

		context.addMessage(builder.build());
	}
}